# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl import fault_mode as ft

# 文件系统oid阈值
FS_OID_THRESHOLD = 53000000

# 巡检开启32位文件数量阈值
FS_32_THRESHOLD = 100

# 没有开启双活
FS_HYPER_OFF = "0"

# 开启32位补丁开关
RDC_ON_ONE = "1"

# 开启32位补丁开关
RDC_ON_THREE = "3"


class CheckItem(BaseCheckItem):
    def execute(self):
        result = self.dsl("exec_on_all {}", self.check_fs_oid_leaked)
        for value in result.values():
            if not value[0]:
                return CheckStatus.NOT_PASS, self.get_msg(value[1])
        return CheckStatus.PASS, ""

    def check_fs_oid_leaked(self):
        """
        检查oid使用是否超过阈值
        :param
        :return:True:通过;False:不通过
        """
        # 查找打开32位复用开关的文件系统
        fs_info = self.dsl(
            "exec_cli \
            'show file_system general |filterRow column=Support\s32bit\sInode predict=equal_to value=Yes |filterColumn include columnList=ID,Pool\sID' \
            | horizontal_parser",
            return_if={ft.CmdNoSupport: "not_support"})
        if fs_info == "not_support" or not fs_info:
            return True, ""
        # 检查文件系统数量
        check_fs_num = 1
        for fs in fs_info:
            # 文件系统检查数量超过规格
            if check_fs_num > FS_32_THRESHOLD:
                return False, "check.skip"
            fsid = fs.get("ID")
            fs_hyper = self.dsl("exec_diagnose 'fs ctrl showhypermetro -fsid {}' | vertical_parser".format(fsid))
            if len(fs_hyper) > 0:
                fs_mode = fs_hyper[0].get("fsMode")
            else:
                fs_mode = None
            # 不支持双活或没有开启双活
            if not fs_mode or fs_mode == FS_HYPER_OFF:
                result = self.deal_fs(fs, fsid)
            else:
                result = self.deal_fs_hyper(fsid, fs_hyper)
            if not result:
                return False, "check.not.pass"
            check_fs_num += 1
        return True, ""

    def deal_fs(self, fs, fsid):
        poolid = fs.get("Pool ID")
        # 查询文件系统oid
        result = self.check_oid_hyper(fsid, 0, poolid, None)
        if not result:
            pool_info = self.dsl(
                "exec_diagnose 'fs cfg queryfs -id {}' | vertical_parser".format(fsid))
            if pool_info:
                rdc = pool_info[0].get("rdc")
            if not self.check_rdc(rdc):
                return False
        # 查询文件系统下面的dtreeid
        dtree_info = self.dsl(
            "exec_cli 'show dtree general file_system_id_list={} |filterColumn include columnList=Dtree\sID' \
            | horizontal_parser".format(fsid))
        if not dtree_info:
            return True
        for dtree in dtree_info:
            dtreeid = dtree.get("Dtree ID").split("@")[-1]
            # 查询Dtree下面oid
            result = self.check_oid_hyper(fsid, dtreeid, poolid, None)
            if not result:
                return False
        return True

    def deal_fs_hyper(self, fsid_local, fs_hyper):
        fsid = fs_hyper[0].get("hyper fs fsId")
        clsid = fs_hyper[0].get("hyper fs clsId")
        pool_info = self.dsl(
            "exec_diagnose 'fs cfg queryfs -id {} -cls {}' | vertical_parser".format(fsid, clsid))
        poolid = pool_info[0].get("storage pool Id")
        rdc = pool_info[0].get("rdc")
        # 查询文件系统下面oid
        result = self.check_oid_hyper(fsid, 0, poolid, clsid)
        if not result and not self.check_rdc(rdc):
            return False
        # 查询文件系统下面的dtreeid
        dtree_info = self.dsl(
            "exec_cli 'show dtree general file_system_id_list={} |filterColumn include columnList=Dtree\sID' \
            | horizontal_parser".format(fsid_local))
        if not dtree_info:
            return True
        for dtree in dtree_info:
            dtreeid = dtree.get("Dtree ID").split("@")[-1]
            # 查询Dtree下面oid
            result = self.check_oid_hyper(fsid, dtreeid, poolid, clsid)
            if not result:
                return False
        return True

    def check_oid_hyper(self, fsid, dtreeid, poolid, clsid):
        """
        查询双活文件系统oid，oid对比FS_OID_THRESHOLD
        :param fsid:文件系统id
        :param dtreeid:dtreeid
        :param poolid:存储池id
        :param clsid: 集群
        :return:True:通过;False:不通过
        """
        if not clsid:
            # 查询文件oid
            oid_result = self.dsl(
                "exec_diagnose 'fs cfg oid -id {} -dtree {} -pool {}' \
                | regex 'Next\s*:\s*\d+\s+(\d+)\s+\d+' | get_index(0)".format(fsid, dtreeid, poolid))

            if int(oid_result) > FS_OID_THRESHOLD:
                return False
            # 查询目录oid
            oid_result = self.dsl(
                "exec_diagnose 'fs cfg oid -d -id {} -dtree {} -pool {}' \
                | regex 'Next\s*:\s*\d+\s+(\d+)\s+\d+' | get_index(0)".format(fsid, dtreeid, poolid))

            if int(oid_result) > FS_OID_THRESHOLD:
                return False
        else:
            # 查询文件oid
            oid_result = self.dsl(
                "exec_diagnose 'fs cfg oid -id {} -dtree {} -pool {} -cls {}' \
                | regex 'Next\s*:\s*\d+\s+(\d+)\s+\d+' | get_index(0)".format(fsid, dtreeid, poolid, clsid))

            if int(oid_result) > FS_OID_THRESHOLD:
                return False
            # 查询目录oid
            oid_result = self.dsl(
                "exec_diagnose 'fs cfg oid -d -id {} -dtree {} -pool {} -cls {}' \
                | regex 'Next\s*:\s*\d+\s+(\d+)\s+\d+' | get_index(0)".format(fsid, dtreeid, poolid, clsid))

            if int(oid_result) > FS_OID_THRESHOLD:
                return False
        return True

    def check_rdc(self, rdc):
        """
        查询文件系统是否打补丁
        :param rdc:rdc开关
        :return:True:开启;False:没有开启开关
        """
        return RDC_ON_ONE == rdc or RDC_ON_THREE == rdc
