# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.
import re

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.product_util import compare_version, compare_patch_version

#1822V120高端共享卡ID列表
HIGH_END_SHARED_CARD_LIST = (
                             2375, 2376, 2377, 2378, 2379, 2380, 2381, 2382, 2383, 2384,
                             2385, 2386, 2387, 2388, 2393, 2412, 2413, 2414, 2421, 2422
)


class CheckItem(BaseCheckItem):
    def execute(self):
        self.logger.info("check high end shared card start")
        # 步骤1：查询环境中是否有1822V120高端共享卡，如果没有则通过
        if not self.check_high_end_shared_card():
            return CheckStatus.PASS, ""
        # 步骤2：获取产品版本和补丁版本,获取不到产品版本和补丁版本返回不通过
        version_info = get_version_info(self.dsl)
        if not version_info:
            self.logger.error("Failed to obtain version info. check failed") 
            return CheckStatus.NOT_PASS, self.get_msg("check.not.pass.version")
        # 1.2.1：上报告警
        # 6.1.5/1.2.0版本：小于SPH30补丁版本则上报告警
        # 6.1.6/1.3.0版本：小于SPH7补丁版本则上报告警
        base_version = version_info.get("base_version").get("Current Version", 0)
        patch_version = version_info.get("patch_version").get("Current Version", 0)
        if re.compile(r"SPH(\d+)").match(patch_version):
            now_version = base_version + '.' + patch_version
        else:
            now_version = base_version
        err_msg = self.get_err_msg("check.not.pass", "item.suggestion", (now_version, ))
        if compare_version(base_version, "1.2.1") == 0:
            return CheckStatus.WARNING, err_msg
        elif ((compare_version(base_version, "6.1.5") == 0) or (
                compare_version(base_version, "1.2.0") == 0)) and compare_patch_version(patch_version, "SPH30") < 0:
            return CheckStatus.WARNING, err_msg
        elif ((compare_version(base_version, "6.1.6") == 0) or (
                compare_version(base_version, "1.3.0") == 0)) and compare_patch_version(patch_version, "SPH7") < 0:
            return CheckStatus.WARNING, err_msg

        return CheckStatus.PASS, ""


    def check_high_end_shared_card(self):
        module_info = self.dsl("exec_rest '/intf_module'")
        for info in module_info:
            model_num = info.get('MODEL', 0)
            if int(model_num) in HIGH_END_SHARED_CARD_LIST:
                return True
        return False
