# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl import fault_mode as ft
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.util.product_util import compare_patch_version

pass_version_dic = {
    "V500R007C71SPC100": "V500R007C71SPH130",
    "V500R007C73SPC100": "V500R007C73SPH110",
    "6.1.0": "SPH30",
    "6.1.2": "SPH30",
    "6.1.3": "SPH10",
}


class CheckItem(BaseCheckItem):

    def execute(self):
        # 检查方法步骤1：检查补丁版本号是否满足要求
        if self.check_patch_version():
            return CheckStatus.PASS, ""

        # 检查方法步骤2：检查是否有前端eth或roce端口
        if self.check_port():
            return CheckStatus.PASS, ""

        return CheckStatus.NOT_PASS, self.get_msg("check.not.pass")

    def check_patch_version(self):
        version_info = get_version_info(self.dsl)
        patch_version = version_info.get("patch_version").get("Current Version")
        base_version = version_info.get("base_version").get("Current Version")

        if base_version in pass_version_dic and \
                compare_patch_version(patch_version, pass_version_dic.get(base_version)) >= 0:
            return True

        return False

    def get_ctrl_encl_list(self):
        enclosures = self.dsl("exec_cli 'show enclosure' | horizontal_parser",
                              return_if={ft.FindStr("Error:"): "not_check"})
        if enclosures == "not_check":
            return []
        cte_enclosure_ids = [enclosure.get("ID") for enclosure in enclosures if
                             enclosure.get("ID").startswith("CTE")]
        return cte_enclosure_ids

    def get_port_list(self, port_type):
        host_port_info = self.dsl("exec_cli 'show port general logic_type=Host_Port'")
        if (port_type + ' port') not in host_port_info:
            return []
        port_list = \
            self.dsl(
                "exec_cli 'show port general logic_type=Host_Port physical_type={} '"
                "| horizontal_parser".format(port_type),
                return_if={ft.FindStr("Error:"): "not_check"})
        return [] if port_list == "not_check" else port_list

    def check_port(self):
        port_list = []
        for port_type in ('ETH', 'RoCE'):
            single_port_list = self.get_port_list(port_type)
            port_list += single_port_list
        if not port_list:
            return False
        cte_list = self.get_ctrl_encl_list()
        if not cte_list:
            return False
        all_id_info = ' '.join([port.get("ID") for port in port_list])
        for cte_id in cte_list:
            if (cte_id + '.A' in all_id_info) and (cte_id + '.B' in all_id_info):
                continue
            if cte_id + '.IOM' in all_id_info:
                continue
            return False
        return True
