# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

import re

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus


# 盘编码白名单
M14_CODE_WHITE_LIST = (
    '02312YXY', '02313EDD', '02313EDF', '02313KKD', '02314BAR', '02314CLB', '02314BAS', '02314CLG',
    '02313EDG', '02313EDH', '02313TQT', '02313KKH', '02314CLD', '02314BAT', '02314CLH', '02314BAU',
    '02314DJK', '02550078', '02550079', '02313FBD', '02313FBE', '02313KKS', '02550168', '02314BAC',
    '02314CKU', '02550522', '02550552', '02313KKT', '02313XJU', '02314BAJ', '02314CKV', '02550164',
    '02550553', '02550523', '02313NUU', '02314BAK', '02314CKW', '02550064', '02313EDT', '02313EDU',
    '02550166', '02313KJW', '02313NUW', '02313XJW', '02550554', '02550524', '02314BAL', '02314CKX',
    '02314BAQ', '02314CKY', '02314BAM', '02314CLJ', '02314DJP', '02550065', '02313EDV', '02313EDW',
    '02313KJV', '02314BAN', '02314CLA', '02314BAP', '02314CLK', '02550323', '02550555', '02550525',
    '02550324', '02550556', '02550526', '02550062', '02550156', '02550557', '02550527', '02550063',
    '02550157', '02550558', '02550528', '02550066', '02550559', '02550529', '02312UBN', '02312UCF',
    '02312UCG', '02312UCH', '02312UCL', '02312UCW', '02312UCX', '02312XDV', '02312XDW', '02312UED',
    '02312UEE', '02312UEH', '02312UEJ'
)

# 寿命剩余小于1年，建议优化，反馈最小剩余时间
WARNING_SUGG_THRE = (365 * 1)
# 寿命剩余小于1年，不通过，反馈最小剩余时间
WARNING_NOT_PASS_THRE = 180
# 最大寿命
LIFE_REMAIN_ENOUGH = (365 * 10)
# 磨损门限
WEAR_THRESHOLD = 18
# FSPUNC门限
FSPUNC_THRESHOLD = 20000

HSSD_TYPE_SAS = "sas"
HSSD_TYPE_NVME = "nvme"

DICT_KEYS_SN = "sn"
DICT_KEYS_ID = "id"
DICT_KEYS_UNC = "unc"
DICT_KEYS_FSPUNC = "fspunc"
DICT_KEYS_REMAIN_TIME = "remain_time"
DICT_KEYS_TYPE = "type"
ITEM_NAME_EN = "check_v6hssd_life_risk"

UPGRADE_FW_TIPS_CN = b"请立即升级设备中硬盘固件版本到8320及以上版本，升级时请至少带一块备件".decode("utf-8")
CONTACT_RD_TIPS_CN = b"请联系研发工程师处理".decode("utf-8")
UPGRADE_FW_TIPS_EN = b"Upgrade the disk firmware version to 8320 or later, Prepare at least one " \
                     b"spare part during the upgrade.".decode("utf-8")
CONTACT_RD_TIPS_EN = b"Contact R&D engineers".decode("utf-8")


class CheckItem(BaseCheckItem):
    def __init__(self, context):
        super(CheckItem, self).__init__(context)
        self.not_supp_scene = 0

    def execute(self):
        disk_info_list = self.dsl("exec_developer 'show disk general|filterColumn include columnList=ID,Type,Model,"
                                  "Firmware\\sVersion,Serial\\sNumber,Run\\sTime(Day),Degree\\sof\\sWear(%)'| "
                                  "horizontal_parser")
        if not disk_info_list:
            return CheckStatus.PASS, ""

        no_pass_upg_fw = []  # 不通过，提示升级
        no_pass_contact_rd = []  # 剩余时间小于180的盘列表
        risk_sn_lst = {}
        for disk in disk_info_list:
            if not self.check_is_risk_hssd(disk):
                continue

            # 获取当前磨损，上电时间，编码的具体数值，不用判断数据正确性了
            pon, wl, sn = self.get_calc_param(disk)
            self.init_risk_dick(disk, risk_sn_lst, pon, wl)

        self.get_ssd_smart(risk_sn_lst)
        self.get_last_risk_info(risk_sn_lst, no_pass_upg_fw, no_pass_contact_rd)
        rlt = self.get_output_info(no_pass_upg_fw, no_pass_contact_rd)

        if no_pass_contact_rd:
            return CheckStatus.NOT_PASS, rlt
        elif no_pass_upg_fw:
            return CheckStatus.NOT_PASS, rlt
        else:
            return CheckStatus.PASS, ""

    def get_ssd_smart(self, risk_sn_lst):
        # 只有超级管理员权限，才能在minisystem下执行命令
        if not self.is_supper_admin():
            return

        # 控制框的盘
        self.query_hssd_drv(risk_sn_lst)

        # IP框smart信息查询
        enclosures = self.dsl("exec_developer 'show enclosure' | horizontal_parser")
        dae_enclosure_ids = [enclosure.get("ID") for enclosure in enclosures if enclosure.get("ID").startswith("DAE")]
        if not dae_enclosure_ids:
            return

        for dae_enclosure_id in dae_enclosure_ids:
            # 遍历IP框
            try:
                if self.login_enclosure(dae_enclosure_id):
                    self.query_hssd_drv(risk_sn_lst)
            finally:
                if self.not_supp_scene == 0:
                    self.login_out_self_exit(dae_enclosure_id)

    # sas盘判断风险策略，剩余时间大于等于0，提示升级固件；剩余时间小于0，联系RD
    def sas_get_last_risk_info(self, risk_sn_lst, no_pass_upg_fw, no_pass_contact_rd):
        for _, ssd in risk_sn_lst.items():
            if ssd.get(DICT_KEYS_TYPE) == HSSD_TYPE_NVME:
                continue

            if ssd.get(DICT_KEYS_REMAIN_TIME) >= 0:
                no_pass_upg_fw.append(ssd.get(DICT_KEYS_ID))
            else:
                no_pass_contact_rd.append(ssd.get(DICT_KEYS_ID))

    # nvme盘 剩余时间大于等于0，提示升级；剩余时间小于0，但UNC等于0 且fspunc 小于20K；提示升级；其他，联系RD
    def nvme_get_last_risk_info(self, risk_sn_lst, no_pass_upg_fw, no_pass_contact_rd):
        for _, ssd in risk_sn_lst.items():
            if ssd.get(DICT_KEYS_TYPE) == HSSD_TYPE_SAS:
                continue
            if ssd.get(DICT_KEYS_REMAIN_TIME) >= 0:
                no_pass_upg_fw.append(ssd.get(DICT_KEYS_ID))
            else:
                if ssd.get(DICT_KEYS_UNC) == 0 and ssd.get(DICT_KEYS_FSPUNC) < FSPUNC_THRESHOLD:
                    no_pass_upg_fw.append(ssd.get(DICT_KEYS_ID))
                else:
                    no_pass_contact_rd.append(ssd.get(DICT_KEYS_ID))

    def get_last_risk_info(self, risk_sn_lst, no_pass_upg_fw, no_pass_contact_rd):
        self.sas_get_last_risk_info(risk_sn_lst, no_pass_upg_fw, no_pass_contact_rd)
        self.nvme_get_last_risk_info(risk_sn_lst, no_pass_upg_fw, no_pass_contact_rd)

    def init_risk_dick(self, disk, risk_sn_lst, pon, wl):
        risk_disk = {
            DICT_KEYS_SN: disk.get("Serial Number"),
            DICT_KEYS_ID: disk.get("ID"),
            DICT_KEYS_UNC: 0,
            DICT_KEYS_FSPUNC: 0,
            DICT_KEYS_REMAIN_TIME: 0,
            DICT_KEYS_TYPE: HSSD_TYPE_SAS}
        risk_sn_lst[risk_disk[DICT_KEYS_SN]] = risk_disk
        risk_sn_lst[disk.get("Serial Number")][DICT_KEYS_REMAIN_TIME] = self.calc_remain_time(pon, wl)

    def check_is_risk_hssd(self, disk):
        """
        执行方法：判断合理性 ：代次，编码白名单，是否合理可用
        """
        fw = disk.get("Firmware Version")
        if not fw.isdigit():
            return False

        # 8318 及以后是解决了的版本
        if int(fw) >= int(8318):
            return False

        judge_ret = [
            self.get_hssd_code(disk) not in M14_CODE_WHITE_LIST,
            self.get_param_valid(disk.get("Run Time(Day)")) == -1,
            self.get_param_valid(disk.get("Degree of Wear(%)")) == -1
        ]
        if any(judge_ret):
            return False

        return True

    def get_param_valid(self, pstr):
        """
        执行方法：判断参数能否转换成float类型
        返回值：-1：异常数据不能转换； 其他：成功
        """
        try:
            return float(pstr)
        except ValueError:
            # 异常值反馈-1，因为寿命磨损可能为0，区分一下
            return -1

    def get_calc_param(self, disk):
        """
        执行方法：获取计算的参数，上电时长，磨损，和编码；放在execute导致方法太大
        """
        pon = self.get_param_valid(disk.get("Run Time(Day)"))
        wl = self.get_param_valid(disk.get("Degree of Wear(%)"))
        sn = disk.get("Serial Number")

        return pon, wl, sn

    def get_output_info(self, no_pass_upg_fw, no_pass_contact_rd):
        """
        执行方法： 组装输出的具体提示信息
        """
        no_pass_contact_rd.sort()
        no_pass_upg_fw.sort()
        rlt = []
        name = self.get_msg("item.name")
        msg_upg = UPGRADE_FW_TIPS_CN
        msg_ctt = CONTACT_RD_TIPS_CN
        if name == ITEM_NAME_EN:
            msg_upg = UPGRADE_FW_TIPS_EN
            msg_ctt = CONTACT_RD_TIPS_EN
        if no_pass_contact_rd:
            rlt.append(self.get_msg("check.not.pass", ", ".join(no_pass_contact_rd), msg_ctt))
        if no_pass_upg_fw:
            rlt.append(self.get_msg("check.not.pass", ", ".join(no_pass_upg_fw), msg_upg))

        return "\n".join(rlt)

    def get_hssd_code(self, disk):
        """
        执行方法： 通过sn获取code，21打头，取第3到第8位，03打头，用03+第1到第6位
        """
        sn = disk.get("Serial Number")
        if sn.startswith("21") and len(sn) == 20:
            return sn[2:10]
        elif sn.startswith("03") and len(sn) == 16:
            return "03" + sn[0:6]
        return ""

    def calc_remain_time(self, poh, used):
        """
        执行方法：计算HSSD的剩余时间
        计算公式： 剩余时长 = 当前上电时长（day） / 磨损 * 磨损门限 - 当前使用时长
        """
        if float(used) < 1:
            return LIFE_REMAIN_ENOUGH

        remain_time = float(poh) / float(used) * float(WEAR_THRESHOLD) - float(poh)
        if remain_time > LIFE_REMAIN_ENOUGH:
            return LIFE_REMAIN_ENOUGH
        return remain_time

    def update_unc_and_fspunc(self, sn_lst, sn, unc_val, fspunc_val):
        for key in sn_lst.keys():
            if key == sn:
                sn_lst[key][DICT_KEYS_UNC] = unc_val
                sn_lst[key][DICT_KEYS_FSPUNC] = fspunc_val
                sn_lst[key][DICT_KEYS_TYPE] = HSSD_TYPE_NVME
                break

    def nvme_update_unc_and_fspunc(self, sn_lst, sn, ret):
        unc = re.findall("unc\(after raid\): ([0-9a-zA-Z ]+)", ret)
        fspunc = re.findall("unc\(before raid\): ([0-9a-zA-Z ]+)", ret)
        if unc or fspunc:
            unc_val = self.hex_to_dec(unc[0])
            fspunc_val = self.hex_to_dec(fspunc[0])
            self.update_unc_and_fspunc(sn_lst, sn, unc_val, fspunc_val)

    # 判断自研盘的代次，是否是V5，V6，V7代次
    def judging_hssd_generation(self, model):
        if model.startswith("HWE5") or model.startswith("HWE6") or model.startswith("HWE7"):
            return True
        if model.startswith("HSSD-D6") or model.startswith("HSSD-D7") or model.startswith("HSSD-D8"):
            return True

        return False

    def query_hssd_drv(self, sn_lst):
        try:
            result = self.dsl("exec_mini_system 'bdm_diagnose_op.sh sddebug showdev'",
                              end_str_list=["minisystem>", "minisystem> "])
            tool_name = self.get_drive_life()
            ssds = re.findall("id (\d+).*\n.*\n.*model ([\w-]+).*sn (\w+).*\n.*\n.*\nblock.*name ([\w-]+)", result)

            for ssd in ssds:
                # 通过正则表达式匹配到的格式[(id, model, sn, pf), ...]，可不判断下标
                model, sn, pf = ssd[1], ssd[2], ssd[3]
                if not self.judging_hssd_generation(model):
                    continue
                if pf.startswith("sd"):
                    cmd = "exec_mini_system '{} -A o -z smartval /dev/{}'".format(tool_name, pf)
                    self.dsl(cmd, end_str_list=["minisystem>", "minisystem> "])
                else:
                    cmd_vd_smart = "exec_mini_system '{} -A N -z getsmart /dev/{}'".format(tool_name, pf)
                    ret = self.dsl(cmd_vd_smart, end_str_list=["minisystem>", "minisystem> "])
                    self.nvme_update_unc_and_fspunc(sn_lst, sn, ret)

        except Exception as e:
            self.logger.info("Hssd Get smart Error: {} ".format(e))

    def get_drive_life(self):
        """
        获取工具版本信息
        """
        tool_name = "disk_repair.sh"
        check_tool_version = "exec_mini_system '{} -v' |splitlines |regex '(^Error[\\s\\S]+)'|" \
                             "get_index(0)|get_index(0)".format("disk_repair.sh")
        tool_version = self.dsl(check_tool_version)
        if tool_version:
            check_tool_version_new = "exec_mini_system '{} -v' |splitlines |regex '(^Error[\\s\\S]+)'|" \
                                     "get_index(0)|get_index(0)".format("disktool")
            tool_version_new = self.dsl(check_tool_version_new)
            if not tool_version_new:
                tool_name = "disktool"
        else:
            tool_name = "disk_repair.sh"

        return tool_name

    def login_enclosure(self, object_name):
        try:
            turn_ptr = object_name + ('.A')
            result = self.dsl("exec_mini_system 'sshtoipenclosure %s'" % turn_ptr)
            if result.find("does not exist or has an incorrect format") != -1:
                self.not_supp_scene = 1
                return False

            self.dsl("exec_cli '%s'" % self.context.dev_node.password, need_log=False)
            self.dsl("exec_cli 'minisystem'")
            return True

        except Exception:
            return False

    # unc(before raid): 12 34 56 00 00 00 00 00
    # 该函数将 12 34 56 00 00 00 00 00 转换为0x563412，再转换成10进制5649426
    def hex_to_dec(self, values):
        try:
            value = values.split(' ')
            '''十六进制转十进制'''
            a = ''
            for i in reversed(value):
                a += i
            return int(a.replace(' ', ''), 16)
        except Exception:
            return 0

    # 退出IP框登录
    def login_out_self_exit(self, object_name):
        # CTE为控制框退出流程，DAE为IP框退出流程
        if 'CTE' in object_name:
            self.dsl("exec_mini_system 'exit'")
            self.logger.info("exit to develop")
            self.dsl("exec_developer 'exit'")
            self.logger.info("exit to admin")
        if 'DAE' in object_name:
            cli_connection = "CLI_CONNECTION"
            cli = self.dsl_obj.dsl_context.context.get_item(cli_connection + self.context.get_dev_node().ip + \
                                                            self.context.get_dev_node().sn)
            cli.execCmd("exit")
            cli.execCmd("y")
            cli.execCmd("exit")
            cli.execCmd("y")
            # 退出minisystem模式
            self.dsl("exec_mini_system 'exit'")
            self.logger.info("exit to develop")
            # 退出developer模式
            self.dsl("exec_developer 'exit'")
            self.logger.info("exit to admin")
