# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import re

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.entity.execute_env import UpgradeExecuteEnv, ExpansionExecuteEnv
from psdk.dsl.dsl_common import get_engine_height
from psdk.platform.util.product_util import get_ctrl_id_by_node_id
from psdk.platform.util.base_util import get_common_msg

product_bios_version_map = {
    'V500R007C10': {
        "err_bios_flag": ['V500R007C30SPH127BiosFlag', 'V500R007C61SPH018BiosFlag'],
        "bios_flag": 'V500R007C10SPH033BiosFlag', "bios_version": '50.02.06T12'},
    'V500R007C30SPC100': {
        "err_bios_flag": ['V500R007C61SPH018BiosFlag'],
        "bios_flag": 'V500R007C30SPH127BiosFlag',
        "bios_version": '50.02.06T11'},
    'V500R007C61': {
        "err_bios_flag": [],
        "bios_flag": 'V500R007C61SPH018BiosFlag',
        "bios_version": '50.02.06T13'}
}



class CheckItem(BaseCheckItem):
    def execute(self):
        if not self.is_supper_admin():
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "loginUser.name.level.must.be.super.admin")

        # 检查方法: 步骤1：查询当前阵列产品版本
        product_version = self.context.dev_node.version
        # 查看步骤1的产品型号是否为V500R007C10 V500R007C30SPC100 V500R007C61版本，不为该版本则通过检查
        product_bios_version_dict = product_bios_version_map.get(product_version)
        if not product_bios_version_dict:
            self.logger.info("no product version in given bios version map. check pass")
            return CheckStatus.PASS, ""
        # 步骤2:  执行命令show version all，查询当前阵列各个控制器的BIOS版本号
        versions = self.dsl(
            "exec_cli 'show version all' | regex 'Controller:([\\s\\S]+)' | get_index(0) | vertical_parser")
        bios_versions_dict = {version.get("Controller"): version.get("BIOS Version") for version in versions}
        if not bios_versions_dict:
            self.logger.info("no bios version in controllers. check pass")
            return CheckStatus.PASS, ""
        # 步骤3  在minisystem下键入命令ls -l /startup_disk/conf/conf_local 查看是否Flag文件
        ctrl_flag_files = self.dsl("exec_on_all {}", self.query_flag_file)
        engine_height = get_engine_height(self.dsl, self.context)
        for node_id, ctrl_files in ctrl_flag_files.items():
            ctrl_id = get_ctrl_id_by_node_id(node_id, engine_height)
            # 如果该环境上面没有flag文件，则需要判断对应版本的bios版本是否存在，存在则返回不通过，否则返回通过
            if not ctrl_files:
                if bios_versions_dict.get(ctrl_id) == product_bios_version_dict.get("bios_version"):
                    return CheckStatus.NOT_PASS, self.get_msg("check.not.pass.bios.version", product_version)
                continue
            # 检查获取到的bios flag 文件是否在对应的黑名单中，如果在则返回检查不通过
            for bios_flag in ctrl_files:
                if bios_flag in product_bios_version_dict.get("err_bios_flag"):
                    return CheckStatus.NOT_PASS, self.get_msg("check.not.pass.bios.flag", product_version)
            # 步骤4  比较BIOS版本和产品版本对应信息,有对应的bios flag文件但是对应的bios版本不配套或者
            # 没有对应的bios flag文件但是对应的bios版本配套，这两种情况都返回检查不通过
            if product_bios_version_dict.get("bios_flag") in ctrl_files and \
                    bios_versions_dict.get(ctrl_id) != product_bios_version_dict.get("bios_version"):
                return CheckStatus.NOT_PASS, self.get_msg("check.not.pass.product.bios.version.match", product_version)
            if product_bios_version_dict.get("bios_flag") not in ctrl_files and \
                    bios_versions_dict.get(ctrl_id) == product_bios_version_dict.get("bios_version"):
                return CheckStatus.NOT_PASS, self.get_msg("check.not.pass.product.bios.version.match", product_version)
        return CheckStatus.PASS, ""

    def query_flag_file(self):
        bios_flag_list = self.dsl("exec_mini_system 'ls -l /startup_disk/conf/conf_local' | splitlines")
        file_names = [line.split(" ")[-1] for line in bios_flag_list if line.split(" ")[-1].startswith("V5")]
        return file_names
