# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2025. All rights reserved.
import re

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.base_util import get_common_msg
from psdk.platform.util.echo_parser import get_horizontal_cli_ret

PRODUCT_PATCH_VERSION_MAP = {
    '1.2.0': 36,
    '1.3.0': 18,
    '1.5.0': 8,
}

DANGER_CAP = 27487790694  # = 34359738368KB * 0.8 = 32TB * 0.8
WARNING_CAP = 17179869184  # = 34359738368KB * 0.5 = 32TB * 0.5


class CheckItem(BaseCheckItem):

    @staticmethod
    def get_system_version_without_ret(cli_ret):
        product_info = []
        hot_patch_begin_index = cli_ret.find("HotPatch Version")
        software_begin_index = cli_ret.find("Software Version")

        if software_begin_index == -1 or hot_patch_begin_index == -1:
            return False, product_info
        software_version = get_horizontal_cli_ret(cli_ret[software_begin_index:hot_patch_begin_index])[0]. \
            get("Current Version")
        controller_names = [d.get('Name')
                            for d in get_horizontal_cli_ret(cli_ret[software_begin_index:hot_patch_begin_index])]
        hot_patch_version = get_horizontal_cli_ret(cli_ret[hot_patch_begin_index:])[0].get("Current Version")
        if hot_patch_version == "--":
            hot_patch_version = "0"
        product_info.append(software_version)
        product_info.append(hot_patch_version)
        product_info.append(controller_names)
        return True, product_info

    @staticmethod
    def check_product_version(software_version, hot_patch_version):
        return int(re.search(r'\d+', hot_patch_version).group()) >= PRODUCT_PATCH_VERSION_MAP.get(software_version)

    @staticmethod
    def get_ssd_dd_engine_info(ssd_disk_domain_infos, engine_map):
        for ssd_disk_domains_info in ssd_disk_domain_infos:
            engine_list = ssd_disk_domains_info.get('Controller enclosure', '').split(',')
            for key in engine_list:
                if not engine_map.get(key):
                    engine_map[key] = 0
                engine_map[key] += 1
        return engine_map

    def get_disk_domain_info(self):
        cli_ret = self.dsl("exec_cli 'show disk_domain general |"
                           "filterColumn include columnList=ID' | horizontal_parser")

        disk_domains = [item.get("ID") for item in cli_ret]
        return len(disk_domains) < 1, disk_domains

    def get_file_system_info(self):
        """
        获取所有文件系统，并识别是否存在文件系统，是则继续检查，否则检查通过
        """
        cli_ret = self.dsl("exec_cli 'show file_system general |"
                           "filterColumn include columnList=ID' | horizontal_parser")

        file_system = [item.get("ID") for item in cli_ret]
        return len(file_system) < 1

    def get_before_compress_cap(self, disk_domain, kvs_id, controller_name):
        """
        获取所有文件系统，并识别是否存在文件系统，是则继续检查，否则检查通过
        """

        pool_id_hex = hex(int(disk_domain)) + '0005'
        pool_id = int(pool_id_hex, 16)
        cli_ret = self.dsl("exec_developer 'show kvs pool_capacity pool_id={} kvs_id={} controller={}"
                           "'| vertical_parser".format(pool_id, kvs_id, controller_name))

        before_compress_cap = [item.get('Before Compress', ' ') for item in cli_ret]
        return int(before_compress_cap[0])

    def check_kvs_info(self, disk_domain, controller_names, result):
        """
        通过硬盘域ID，获取kvs id、 vnode id、 work node id
        """
        kvs_info_s = self.dsl("exec_developer 'show pool_manager key_value_store disk_pool_id={}' |"
                              "horizontal_parser".format(disk_domain))
        tmp_vnode_id = -1

        for kvs_info in kvs_info_s:
            if tmp_vnode_id == kvs_info.get("Vnode ID"):
                continue
            else:
                tmp_vnode_id = kvs_info.get("Vnode ID")
                controller_name = controller_names[int(kvs_info.get("Work Node"))]
                kvs_id = kvs_info.get("Kvs ID")
                before_compress_cap = self.get_before_compress_cap(disk_domain, kvs_id, controller_name)
                if (before_compress_cap > DANGER_CAP) and (result < 1):
                    result = 1
                elif (before_compress_cap > WARNING_CAP) and (result < 0):
                    result = 0
        return result

    def execute(self):
        if not self.is_supper_admin():
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "loginUser.name.level.must.be.super.admin")
        # 检查方法步骤1：获取产品版本信息，非指定版本，检查通过
        # 获取产品信息
        flag, product_info = \
            self.get_system_version_without_ret(self.dsl("exec_cli 'show upgrade package'"))
        if not product_info:
            return CheckStatus.PASS, ""

        software_version, hot_patch_version, controller_names = product_info

        if software_version not in PRODUCT_PATCH_VERSION_MAP:
            return CheckStatus.PASS, ""
        # 检查是否为非风险版本
        if self.check_product_version(software_version, hot_patch_version):
            return CheckStatus.PASS, ""

        # 检查方法步骤2：获取硬盘域信息，硬盘域个数小于1，检查通过
        disk_domain_check_res, disk_domains = self.get_disk_domain_info()
        if disk_domain_check_res:
            return CheckStatus.PASS, ""

        # 检查方法步骤3：查看是否存在文件系统，若不存在，检查通过
        if self.get_file_system_info():
            return CheckStatus.PASS, ""

        result = -1
        for disk_domain in disk_domains:
            result = self.check_kvs_info(disk_domain, controller_names, result)

        if result < 0:
            return CheckStatus.PASS, ""
        elif result == 0:
            return CheckStatus.WARNING, ""
        else:
            return CheckStatus.NOT_PASS, self.get_msg("check.not.pass")

