# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.
# 检查是否删除内存故障告警

import ast

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus


class CheckItem(BaseCheckItem):
    def get_cleared_events(self):
        # 执行命令获取已清除的事件
        return self.dsl("exec_cli 'show event status=cleared' | horizontal_parser")

    def get_event_details(self, sequence):
        # 执行命令获取事件的详细信息
        return self.dsl("exec_cli 'show event sequence={}' | vertical_parser".format(sequence))

    def extract_controller_id(self, detail):
        # 从 Detail 字段中提取控制器 ID
        try:
            enclosure_part = detail.split("controller enclosure")[1].split(",")[0].strip()
            enclosure_number = enclosure_part.split("CTE")[-1].strip()
            controller_name = detail.split("controller")[2].strip()[0]

            return "{}{}".format(enclosure_number, controller_name)

        except (IndexError, ValueError) as e:
            self.logger.info(
                """Ignore this sequence, because can not extract controller ID.
                Detail: {}
                Error: {}.""".format(detail, str(e))
            )
            return None  # 如果解析失败，返回 None

    def validate_controller_ids(self, ctrl_name, controller_ids):
        # 如果 controller_ids 为空，返回 True
        if not controller_ids:
            return True

        # 检查 controller_ids 中的所有信息是否与 ctrl_name 完全一致
        return all(ctrl_id == ctrl_name for ctrl_id in controller_ids)

    def execute(self):
        # 获取客户选取的备件控制器的编号
        selected_controller = ast.literal_eval(self.context.execute_env.selected_fru)
        ctrl_name = selected_controller.get("id")
        # 初始化 controller_ids 集合
        controller_ids = set()
        # 获取已清除的事件
        cleared_events = self.get_cleared_events()
        if not cleared_events:
            return CheckStatus.PASS, ""
        # 遍历事件，查找对应的控制器 ID
        for event in cleared_events:
            alarm_id = event.get("ID")
            sequence = event.get("Sequence")
            if alarm_id in ["0xF00ED000F", "0xF00ED000E"]:  # 根据告警码过滤
                event_details = self.get_event_details(sequence)
                detail = event_details[0].get("Detail")
                controller_id = self.extract_controller_id(detail)
                if controller_id:
                    controller_ids.add(controller_id)  # 使用 add 方法添加到集合中
        validation_result = self.validate_controller_ids(ctrl_name, controller_ids)
        if validation_result:
            return CheckStatus.PASS, ""
        else:
            return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion")