# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.checkitem.common.fru_common import is_selected_fan_in_disk_enclosure
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.product_util import compare_patch_version
from psdk.dsl.dsl_common import get_version_info
from psdk.dsl import fault_mode as ft

# 带NAS业务的最大端口数量
MAX_NAS_BUSINESS_PORT_NUM = 47


class CheckItem(BaseCheckItem):
    def execute(self):
        # 获取基本信息
        tool_name = self.context.execute_env.tool_name
        self.logger.info("tool_name: {}".format(tool_name))
        version_info = get_version_info(self.dsl)
        patch_version = version_info.get("patch_version").get("Current Version")
        version = version_info.get("base_version").get("Current Version")

        # 判断当前更换是否为硬盘框风扇
        if is_selected_fan_in_disk_enclosure(self):
            self.logger.info("fan in disk enclosure, check pass")
            return CheckStatus.PASS, ""

        # 6.1.0.SPH15及以后或6.1.2.SPH6及以后，检查通过
        if not self.is_risk_version_and_patch(version, patch_version):
            return CheckStatus.PASS, ""

        # 检查方法步骤1：使用dsl执行命令查询端口的信息
        local_ports = self.dsl(
            "exec_cli 'show logical_port general' | horizontal_parser",
            return_if={ft.FindStr("Error:"): "not_check"})
        if local_ports == "not_check":
            check_status = CheckStatus.NOT_CHECK \
                if tool_name == "inspect_daily" or tool_name.startswith("exp") else CheckStatus.NOT_PASS
            return check_status, self.get_msg("query.result.abnormal")
        result = [local_port for local_port in local_ports
                  if self.has_nas_business(local_port)]
        if len(result) <= MAX_NAS_BUSINESS_PORT_NUM:
            return CheckStatus.PASS, ""
        return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion", str(len(result)))

    @staticmethod
    def is_risk_version_and_patch(version, patch_version):
        """
        判断是否为风险版本和补丁
        :param version: 软件版本
        :param patch_version: 补丁版本
        :return:
        """
        return not any([version == "6.1.0" and
                        compare_patch_version(patch_version, "SPH15") >= 0,
                        version == "6.1.2" and
                        compare_patch_version(patch_version, "SPH6") >= 0])

    @staticmethod
    def has_nas_business(local_port):
        """
        判断端口是否带有NAS业务
        :param local_port:端口信息
        :return:
        """
        return any(["NFS" in local_port.get("Support Protocol"),
                    "CIFS" in local_port.get("Support Protocol")])
