# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import re

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.util.product_util import compare_patch_version
from psdk.platform.util.product_util import compare_version

PASS_VERSION_DICT = {
    "6.1.5": "SPH20",
    "6.1.3": "SPH22",
    "6.1.2": "SPH52",
}

VERSION_V6 = "6.1.6RC1"
VERSION_V2 = "6.1.2RC1"
OCEANDISK_VERSION = "1.1.0RC1"


class CheckItem(BaseCheckItem):

    def execute(self):

        # 判断是否存在NVMe over RoCE协议的逻辑端口
        port_general = self.dsl("exec_cli 'show logical_port general' | horizontal_parser")
        if not port_general:
            return CheckStatus.PASS, ""
        port_num = 0
        for item in port_general:
            if item.get("Support Protocol") == "NVMe over RoCE":
                port_num += 1
                break
        # 不存在NVMe over RoCE协议的逻辑端口 巡检通过
        if port_num == 0:
            return CheckStatus.PASS, ""

        # 获取基本信息
        version_info = get_version_info(self.dsl)
        patch_version = version_info.get("patch_version").get("Current Version")
        base_version = version_info.get("base_version").get("Current Version")

        if re.compile(r"SPH(\d+)").match(patch_version):
            now_version = base_version + '.' + patch_version
        else:
            now_version = base_version
        err_msg = self.get_msg("check.not.pass", now_version)

        # 检查版本软件1.1.0或之后版本通过
        if re.compile(r"^1").match(base_version):
            if compare_version(base_version, OCEANDISK_VERSION) < 0:
                return CheckStatus.WARNING, err_msg
            else:
                return CheckStatus.PASS, ""

        # 检查版本软件6.1.6或之后版本通过
        if compare_version(base_version, VERSION_V6) >= 0:
            return CheckStatus.PASS, ""

        if compare_version(base_version, VERSION_V2) < 0:
            return CheckStatus.WARNING, err_msg

        if base_version in PASS_VERSION_DICT:
            if compare_patch_version(patch_version, PASS_VERSION_DICT[base_version]) >= 0:
                return CheckStatus.PASS, ""
            else:
                return CheckStatus.WARNING, err_msg

        return CheckStatus.WARNING, err_msg
