# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.base_util import get_common_msg
from psdk.platform.util.product_util import compare_patch_version


class CheckItem(BaseCheckItem):
    def execute(self):
        # 后续需要超级用户权限，检查是否超级用户登录
        if not self.is_supper_admin():
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "loginUser.name.level.must.be.super.admin")

        # 解决问题的补丁版本检查通过
        version_info = get_version_info(self.dsl)
        base_version = version_info.get("base_version").get("Current Version")
        patch_version = version_info.get("patch_version").get("Current Version")
        if self.pass_version_check(base_version, patch_version):
            return CheckStatus.PASS, ""

        # 所有控制器正常才执行巡检
        controller_number = 0
        controller_list_cli = self.dsl("exec_cli 'show controller general' | vertical_parser")
        for controller in controller_list_cli:
            controller_number += 1
            if "Normal" != controller.get("Health Status"):
                return CheckStatus.PASS, ""

        # 硬盘域个数获取
        diskdomain_list_cli = self.dsl("exec_cli 'show disk_domain general' | horizontal_parser")
        diskdomain_number = len(diskdomain_list_cli)
        # 性能层个数获取
        performancelayer_list_cli = self.dsl("exec_cli 'show performance_layer general' | horizontal_parser")
        performancelayer_number = len(performancelayer_list_cli)
        # 所有控总的定时任务数
        runningtimer_number = 0
        all_controller_runningtimer = self.dsl("exec_on_all {}", self.count_runningtimer_in_controller)
        for _, number in all_controller_runningtimer.items():
            runningtimer_number += number

        # 判断规则 控制器个数 * 2 + 14 * 硬盘域个数 + 性能层个数 = 定时器总数
        if (controller_number * 2 + diskdomain_number * 14 + performancelayer_number) == runningtimer_number:
            return CheckStatus.PASS, ""

        if "6.1.3" == base_version:
            error_info = self.get_msg("check.not.pass.613.message")
        elif "6.1.5" == base_version:
            error_info = self.get_msg("check.not.pass.615.message")
        else:
            error_info = self.get_msg("check.not.pass.616.message")

        return CheckStatus.NOT_PASS, error_info

    def count_runningtimer_in_controller(self):
        """
        统计当前控制器的定时器个数
        """
        runningtimer_number = 0

        runningtimer_str = self.dsl("exec_diagnose 'pmgr showPollTask'")
        runningtimer_str.strip()
        if len(runningtimer_str) == 0:
            return runningtimer_number

        runningtimer_list = runningtimer_str.split('\n')
        for timer in runningtimer_list:
            timer_field_list = timer.split()
            if len(timer_field_list) >= 6 and (timer_field_list[1] == "LOOP") and ("STOP" not in timer_field_list[-3]):
                runningtimer_number += 1

        return runningtimer_number

    def pass_version_check(self, base_version, patch_version):
        """
        问题解决版本为6.1.3.SPH30，6.1.5.SPH30，6.1.6.SPH10
        """
        if "6.1.3" == base_version and compare_patch_version("SPH30", patch_version) <= 0:
            return True
        if "6.1.5" == base_version and compare_patch_version("SPH30", patch_version) <= 0:
            return True
        if "6.1.6" == base_version and compare_patch_version("SPH10", patch_version) <= 0:
            return True

        return False

