# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2022-2022. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.entity.execute_env import UpgradeExecuteEnv, ExpansionExecuteEnv
from psdk.dsl import fault_mode as ft
from psdk.platform.base import constants
from psdk.platform.util import product_util

CERT_TYPE = "Type"
EXPIRE_TIME = "Expire Time"

PRECONFIG_CERT_ISSUER_INFO = "C=CN,O=Huawei,CN=Huawei IT Product CA"


class CheckItem(BaseCheckItem):
    def execute(self):
        # 检查方法步骤1：执行命令：show certificate general获取证书类型
        cert_infos = self.dsl("exec_cli 'show certificate general' | horizontal_parser",
                              return_if={ft.FindStr("Error"): CheckStatus.NOT_CHECK})
        if cert_infos == CheckStatus.NOT_CHECK:
            return CheckStatus.NOT_CHECK, self.get_msg("query.result.abnormal")

        if not cert_infos:
            return CheckStatus.PASS, ""

        cert_names = [cert_info["Type"] for cert_info in cert_infos]

        preconfig_certs = self.get_preconfig_certs(cert_names)
        if preconfig_certs:
            return CheckStatus.WARNING, self.get_msg("check.item.not.pass", "\n".join(preconfig_certs))
        return CheckStatus.PASS, ""

    def get_preconfig_certs(self, cert_names):
        preconfig_certs = list()
        for cert_name in cert_names:
            cert_type = constants.CERT_TYPE_2_NAME_DICT.get(cert_name.lower(), "")
            if not cert_type:
                continue
            # 执行命令：show certificate general type=xxx detail=yes获取证书信息
            cert_info = self.dsl(
                "exec_cli 'show certificate general type={} detail=yes' | vertical_parser".format(cert_type),
                return_if={ft.FindException: []})
            if not cert_info:
                continue
            preconfig_cert = self.get_preconfig_cert(cert_info[0])
            if preconfig_cert:
                preconfig_certs.append(preconfig_cert)
        return preconfig_certs

    def is_support_fingerprint(self):
        """
        指纹列表：（适用于V5R7C00~V5R7C60SPC300）
        """
        version = self.context.dev_node.version
        if "V5" in version and product_util.compare_version(version, "V500R007C00") >= 0 \
                and product_util.compare_version(version, "V500R007C60SPC300") <= 0:
            return True
        return False

    def get_preconfig_cert(self, cert_info):
        if self.is_preconfig_cert(cert_info):
            return self.get_preset_cert_info(cert_info)
        return ""

    def is_preconfig_cert(self, cert_info):
        if self.is_support_fingerprint() and self.is_preconfig_cert_by_fingerprint(cert_info):
            return True
        if self.is_preconfig_cert_by_issuer(cert_info):
            return True
        return False

    def get_preset_cert_info(self, cert_info):
        return ",".join(
            [self.get_cert_key_value(cert_info, CERT_TYPE), self.get_cert_key_value(cert_info, EXPIRE_TIME)])

    def get_cert_key_value(self, cert_info, param):
        return "{}: {}".format(param, cert_info.get(param, "--"))

    def is_preconfig_cert_by_fingerprint(self, cert_info):
        """
        指纹列表：（适用于V5R7C00~V5R7C60SPC300）
        HyperMetro Arbitration：2F:73:81:36:78:77:BA:D4:8E:4D:01:36:E6:CA:F4:E4:39:63:04:21
        在指纹列表中

        """
        cert_type = cert_info.get(CERT_TYPE, "--")
        return constants.CERT_TYPE_2_FINGERPRINT_DICT.get(cert_type, "") == cert_info.get("CA Fingerprint", "--")

    def is_preconfig_cert_by_issuer(self, cert_info):
        """
        Issuer字段为C=CN,O=Huawei,CN=Huawei IT Product CA则是预置证书 or
        CA Issuer字段为C=CN,O=Huawei,CN=Huawei IT Product CA则是预置证书
        """

        return cert_info.get("Issuer", "--") == PRECONFIG_CERT_ISSUER_INFO or \
               cert_info.get("CA Issuer", "--") == PRECONFIG_CERT_ISSUER_INFO
