# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

import re

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus

MAX_DEV_NUM = 789

SLOT_PORT_MAP_2P = {
    "5": ["0x0f2011", "0x0f2010"], "6": ["0x0f2001", "0x0f2000"], "7": ["0x0f2021", "0x0f2020"],
    "8": ["0x0f2031", "0x0f2030"]
}

SLOT_PORT_MAP_4P = {
    "5": ["0x0f2010", "0x0f2050"], "6": ["0x0f2011", "0x0f2051"], "7": ["0x0f2030", "0x0f2070"],
    "8": ["0x0f2031", "0x0f2071"]
}

SLOT_PORT_MAP = [SLOT_PORT_MAP_2P, SLOT_PORT_MAP_4P]

PORT_CARD_MAP = [
    {"0x0f2011": 1, "0x0f2001": 0, "0x0f2021": 2, "0x0f2031": 3, "0x0f2010": 1, "0x0f2000": 0, "0x0f2020": 2,
     "0x0f2030": 3},
    {"0x0f2010": 1, "0x0f2011": 1, "0x0f2030": 3, "0x0f2031": 3, "0x0f2050": 5, "0x0f2051": 5, "0x0f2070": 7,
     "0x0f2071": 7}
]

ENG_EXP_DISK_NUM_LIST = [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]


class CheckItem(BaseCheckItem):
    def execute(self):
        if self.is_a8():
            return CheckStatus.PASS, ""
        self.get_all_exp_disk_num()
        target = self.get_targets()
        self.log_info(str(target))
        res = self.dsl("exec_on_all {}", self.check_resource, target_ctrl=target)
        self.log_info(res)
        for v in res.values():
            if not v:
                return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion")
        return CheckStatus.PASS, ""

    def get_targets(self):
        info = self.dsl("exec_mini_system 'showsysstatus'")
        res = re.findall(r'(\d+)\s+\w+\s*normal\s*\d+\s*(\d+)', info)
        targets = dict()
        for pos in res:
            node_id = pos[0]
            eng_id = pos[1]
            if ((int(eng_id) < len(ENG_EXP_DISK_NUM_LIST)) and (ENG_EXP_DISK_NUM_LIST[int(eng_id)] != 0)
                    and (int(targets.get(eng_id, ['255'])[0]) > int(node_id))):
                targets[eng_id] = [node_id]
        return targets

    def log_info(self, info):
        self.logger.info("[check_sas_card_resource]{}.".format(info))

    def get_all_exp_disk_num(self):
        global ENG_EXP_DISK_NUM_LIST
        disk_list = self.context.execute_env.ori_env.get("exp_disk_list")
        if not disk_list:
            disk_list = self.context.execute_env.ori_env.get("expDiskList")
        if not disk_list:
            self.log_info("get expand disk num failed")
            return

        for disk_item in disk_list:
            self.log_info(str(disk_item))
            eng_name = disk_item.get("logicEng", None)
            disk = disk_item.get("diskNum", None)
            disk_type = disk_item.get("diskModel", None)
            if disk is None or eng_name is None:
                self.log_info("input err")
                continue
            if disk_type is not None and "NVMe" in disk_type:
                continue
            if eng_name == "new":
                for i in range(len(ENG_EXP_DISK_NUM_LIST)):
                    ENG_EXP_DISK_NUM_LIST[i] += int(disk)
            else:
                eng = int(eng_name)
                if eng >= len(ENG_EXP_DISK_NUM_LIST):
                    self.log_info("input eng id:{} is abnormal".format(eng))
                    continue
                ENG_EXP_DISK_NUM_LIST[eng] += int(disk)

    def check_rsc(self, ports, disk, model):
        encl_num = int(disk / 24)
        if disk % 24 != 0:
            encl_num += 1
        if model == 0:
            need_rsc = disk * 2 + encl_num * 4
        else:
            need_rsc = disk + encl_num * 2
        for p in ports:
            rsc = self.get_residual_dev_num(p)
            self.log_info("port:{} res:{} need:{}".format(p, rsc, need_rsc))
            if rsc < need_rsc:
                return False
        return True

    def get_exp_disk_num(self, eng):
        if eng < len(ENG_EXP_DISK_NUM_LIST):
            return ENG_EXP_DISK_NUM_LIST[eng]
        self.log_info("eng id:{} err".format(eng))
        return 0

    def check_resource(self):
        try:
            model = self.get_board_model()
            eng, node = self.get_ctrl_pos()
            slot_map = SLOT_PORT_MAP[model]
            sas_ports = self.get_sas_ports(eng, slot_map, PORT_CARD_MAP[model])
            self.log_info("get sas ports:{}".format(str(sas_ports)))
            disk_num = self.get_exp_disk_num(eng)
            res = self.check_rsc(sas_ports, disk_num, model)
            self.log_info("eng:{} node:{} model:{} disk:{} res:{}".format(eng, node, model, disk_num, res))
            return res
        except Exception as e:
            self.log_info("{}".format(str(e)))
            return False

    def get_ctrl_pos(self):
        res = self.dsl("exec_mini_system 'showsysstatus'")
        node_id = re.search(r"local node id\s*:\s*(\d+)", res).group(1)
        pat = "{}{}".format(node_id, r"\s+(master|slave)\s*normal\s*\d+\s*(\d+)")
        eng_id = re.search(pat, res).group(2)
        return int(eng_id), int(node_id)

    def get_sas_ports(self, eng, slot_map, card_map):
        ports = list()
        cards = list()
        res = self.dsl("exec_cli 'show interface_module'")
        m = re.findall(r"CTE" + str(eng) + r".IOM.[LH][5678]\s*Normal\s*Running.*SAS", res)
        for i in m:
            s = re.search(r"CTE\d+.IOM.[LH]([5678])\s*Normal\s*Running.*SAS", i)
            slot = s.group(1)
            port = slot_map.get(slot, None)
            for p in port:
                card = card_map.get(p)
                if card not in cards:
                    cards.append(card)
                    ports.append(p)
        self.log_info("get cards:{}".format(str(cards)))
        return ports

    def get_residual_dev_num(self, port):
        cmd = "exec_diagnose 'drvsascbb showrsc port_id=%s'" % port
        res = self.dsl(cmd)
        num = MAX_DEV_NUM - int(re.search(r'max Dev num.*?used:(\d+)', res).group(1))
        return max(num, 0)

    def is_a8(self):
        k = self.dsl("exec_mini_system 'upgrade.sh kernel showversion'")
        m = re.search("kernel version: (\d+)\.(\d+)\.(\d+)\.(\d+)", k)
        # V5版本无此命令
        if not m:
            return False
        # V6内核版本低于1.1.8.0
        if int(m.group(1)) * 1000 + int(m.group(2)) * 100 + int(m.group(3)) * 10 + int(m.group(4)) < 1180:
            return False
        return True

    def get_board_model(self):
        # 检查是否是高端2p单板
        cpu_info = self.dsl("exec_diagnose 'drvsascbb showrsc port_id=0x0f2001'")
        if re.search("show IO and Dev Num", cpu_info):
            return 0
        return 1
