# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.echo_parser import get_horizontal_cli_ret_compatible_chinese
from psdk.dsl import fault_mode as ft
from psdk.platform.util.base_util import get_common_msg


class CheckItem(BaseCheckItem):
    def execute(self):
        status, check_info = self.check_smart_move_domain()
        if status == CheckStatus.PASS or status == CheckStatus.NOT_CHECK or status == CheckStatus.NOT_PASS:
            return status, check_info
        return self.check_smart_move_fs_pair()

    def check_smart_move_domain(self):
        # 1.若设备不支持'show smart_move_domain general'命令或没有SmartMove迁移域，则通过检查，否则继续检查；
        # 2.若设备支持'show smart_move_domain general'命令，
        # 则检查是否存在运行状态("Running Status")为恢复中("Recovering")的SmartMove迁移域，若不存在，则继续检查，否则不通过检查；
        # Dorado 3000 存在不支持NAS的場景，需要单独做判断
        # Error: The NAS HyperMetro domain cannot be operated because the current device does not support NAS.
        domain_info = self.dsl("exec_cli 'show smart_move_domain general'",
                               return_if={ft.CmdNoSupport: "not_support",
                                          ft.FindStr("not support NAS"): "not_support",
                                          ft.FindStr("No matching records"): "not_support"})
        self.logger.info("the smart_move_domain info is {}".format(domain_info))
        # 由于不支持NAS的回显中存在Error，所以不能使用ft.FindStr,故单独判断存在Error的场景
        if "Error:" in domain_info:
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "query.result.abnormal")
        if domain_info == "not_support":
            return CheckStatus.PASS, ""
        domain_info_dict = get_horizontal_cli_ret_compatible_chinese(domain_info)
        for one_domain in domain_info_dict:
            running_status = one_domain.get("Running Status", "")
            if running_status == "Recovering":
                return CheckStatus.NOT_PASS, self.get_msg("check.smart.move.domain.not.pass")
        return CheckStatus.UNKNOWN, ""

    def check_smart_move_fs_pair(self):
        # 3.若设备不支持'show smart_move_fs_pair general'命令或没有SmartMove Fs Pair，则通过检查，否则继续检查；
        # 4.若设备支持'show smart_move_fs_pair general'命令，
        # 并且存在交换状态("Exchange Status")不为交换完成("Exchanged")的SmartMove Fs Pair，则检查不通过，否则通过检查；
        fs_pair_info = self.dsl("exec_cli 'show smart_move_fs_pair general'| horizontal_parser",
                                return_if={ft.CmdNoSupport: "not_support", ft.FindStr("Error:"): "not_check",
                                           ft.FindStr("No matching records"): "not_support"})
        self.logger.info("the smart_move_fs_pair info is {}".format(fs_pair_info))
        if fs_pair_info == "not_check":
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "query.result.abnormal")
        if fs_pair_info == "not_support":
            return CheckStatus.PASS, ""
        for one_fs_pair in fs_pair_info:
            exchange_status = one_fs_pair.get("Exchange Status", "")
            if exchange_status != "Exchanged":
                return CheckStatus.NOT_PASS, self.get_msg("check.smart.move.fs.pair.not.pass")
        return CheckStatus.PASS, ""
