# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.base_util import get_common_msg
from psdk.platform.util.product_util import compare_patch_version
from psdk.dsl import fault_mode as ft

# 步骤1 以admin用户登录设备；
# 步骤2 执行命令：show upgrade package，查询软件版本和补丁版本；
# 步骤3 执行命令: show disk_domain general, 查询硬盘域ID disk_domain_id
# 步骤4 执行命令: show controller general 查询控制器controller_id
# 步骤5 执行命令：change user_mode current_mode user_mode=developer，进入开发者模式；
# 步骤6 执行命令: show pool_manager logic_pool disk_pool_id=[disk_domain_id], 遍历查询所有硬盘域的pool
# 步骤7 执行命令: show zone_recycle io pool_id=[disk_domain_id] controller=[controller_id],遍历查询所有硬盘域 + 控制器

# 1 若步骤2中软件版本为6.1.3且补丁版本小于SPH9, 或者软件版本为6.1.5且补丁版本小于SPH3,则继续检查，否则检查结果为通过；
# 2 若步骤6中回显pool不存在SMARTCACHE_LOGIC_POOL,则检查结果为通过,否则继续检查;
# 3 若步骤7中defrag ck和update ftl同时有 Back Bad计数,代表垃圾回收失败,则检查不通过,检查结束,否则检查通过;
CHECK_613_BASE_VERSION = "6.1.3"
NOT_CHECK_613_FLAG_PATCH_VERSION = "SPH9"
CHECK_615_BASE_VERSION = "6.1.5"
NOT_CHECK_615_FLAG_PATCH_VERSION = "SPH3"


class CheckItem(BaseCheckItem):

    def execute(self):
        # 步骤1 以admin用户登录设备，并查看产品型号、版本和补丁版本；
        # 若步骤1中版本号为Dorado V6 6.1.3 SPH9之前的版本 或者 Dorado V6 6.1.5 SPH3之前的版本则继续检查，其余版本检查通过
        version_info = get_version_info(self.dsl)
        cur_version = version_info.get("base_version").get("Current Version")
        patch_version = version_info.get("patch_version").get("Current Version")
        if (cur_version == CHECK_613_BASE_VERSION) and (
                compare_patch_version(patch_version, NOT_CHECK_613_FLAG_PATCH_VERSION) >= 0):
            return CheckStatus.PASS, ""
        if (cur_version == CHECK_615_BASE_VERSION) and (
                compare_patch_version(patch_version, NOT_CHECK_615_FLAG_PATCH_VERSION) >= 0):
            return CheckStatus.PASS, ""
        disk_domain_list = self.get_disk_domain_info()
        if len(disk_domain_list) == 0:
            return CheckStatus.PASS, ""
        results = self.check_smartcache_exist(disk_domain_list)
        if results:
            return CheckStatus.PASS, ""
        res = self.check_recycle_err(disk_domain_list)
        if res:
            return CheckStatus.PASS, ""
        return CheckStatus.NOT_PASS, self.get_msg("check.not.pass")

    def get_disk_domain_info(self):
        disk_domain_results = self.dsl("exec_cli 'show disk_domain general' |horizontal_parser",
                                       return_if={ft.FindStr("Error:"): 'not_check'})
        disk_domain_list = []
        for dd_info in disk_domain_results:
            disk_domain_list.append(dd_info.get("ID"))
        return disk_domain_list

    def get_controller_info(self):
        controller_results = self.dsl("exec_cli 'show controller general' |vertical_parser",
                                      return_if={ft.FindStr("Error:"): 'not_check'})
        controller_list = []
        for dd_info in controller_results:
            controller_list.append(dd_info.get("Controller"))
        return controller_list

    def check_smartcache_exist(self, disk_domain_list):
        # change user_mode current_mode user_mode=developer，进入开发者模式;
        # 进入developer模式,执行命令,show pool_manager logic_pool disk_pool_id=[disk_domain_id], 遍历查询所有硬盘域的pool;
        # 回显pool不存在SMARTCACHE_LOGIC_POOL,则检查结果为通过,否则继续检查;
        for dd_id in disk_domain_list:
            dsl_cmd = "exec_developer 'show pool_manager logic_pool disk_pool_id={}' |horizontal_parser".format(
                dd_id
            )
            pool_results = self.dsl(dsl_cmd, return_if={ft.FindStr("Error:"): 'not_check'})
            if "SMARTCACHE_LOGIC_POOL" in str(pool_results):
                return False
        return True

    def check_recycle_count(self, results):
        defrag_count = 0
        defrag_mem_count = 0
        update_count = 0
        update_mem_count = 0
        for result in results:
            if "defrag ck" in result.values():
                defrag_count = int(result.get('Back Bad'))
                defrag_mem_count = int(result.get('Back No Memory'))
            if "update ftl" in result.values():
                update_count = int(result.get('Back Bad'))
                update_mem_count = int(result.get('Back No Memory'))
            if defrag_count > 0 and update_count > 0:
                return False
            if defrag_mem_count > 0 and update_mem_count > 0:
                return False
        return True

    def check_recycle_err(self, disk_domain_list):
        # change user_mode current_mode user_mode=developer，进入开发者模式;
        # 进入debug模式,执行命令,show pool_manager logic_pool disk_pool_id=[disk_domain_id], 遍历查询所有硬盘域的pool;
        # defrag ck和update ftl同时有 Back Bad计数,代表垃圾回收失败,则检查不通过,检查结束,否则检查通过;
        controller_list = self.get_controller_info()
        if len(controller_list) == 0:
            return True
        for controller_id in controller_list:
            for dd_id in disk_domain_list:
                dsl_cmd = "exec_developer 'show zone_recycle io pool_id={} controller={}' |horizontal_parser".format(
                    dd_id, controller_id)
                results = self.dsl(dsl_cmd, return_if={ft.FindStr("Error:"): 'not_check'})
                if len(results) == 0:
                    continue
                res = self.check_recycle_count(results)
                if not res:
                    return res
        return True