# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2023. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.dsl import fault_mode as ft
from psdk.dsl.dsl_common import get_engine_num
from psdk.dsl.dsl_common import get_engine_height
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.product_util import get_ctrl_id_by_node_id
from psdk.platform.util.base_util import get_common_msg


class CheckItem(BaseCheckItem):
    def execute(self):
        # 检查方法步骤1：使用dsl执行命令查询历史版本信息
        version_info = get_version_info(self.dsl)
        historys = version_info.get("base_version").get("History Version")
        if self.no_history_risk(historys):
            return CheckStatus.PASS, ""
        engine_num = get_engine_num(self.dsl)
        # 单引擎设备则通过
        if engine_num == 1:
            return CheckStatus.PASS, ""
        if not self.is_supper_admin():
            return CheckStatus.NOT_CHECK, get_common_msg(self.lang, "loginUser.name.level.must.be.super.admin")
        # 检查方法步骤2：使用dsl执行命令sys showlog检查退出标记
        exit_flag_results = self.dsl("exec_on_all {}", self.check_exit_flag)
        all_msgs = []
        engine_height = get_engine_height(self.dsl, self.context)
        for node_id, ab_nodes in exit_flag_results.items():
            if ab_nodes:
                ctrl_id = get_ctrl_id_by_node_id(node_id, engine_height)
                all_msgs.append(self.get_msg("check.not.pass", ctrl_id, ab_nodes))
        if all_msgs:
            return CheckStatus.NOT_PASS, "\n".join(all_msgs)
        return CheckStatus.PASS, ""

    def check_exit_flag(self):
        # 检查方法：使用dsl执行命令sys showlog检查退出标记
        lines = self.dsl("exec_diagnose 'sys showlog -i' | splitlines | regex '(?P<time>\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}).*Set node\(nid (?P<node_id>\d+)\) ready exit flag to (?P<flag>(True|False))'",
                         return_if={ft.FindException: []})

        if not lines:
            return []
        return self.get_abnormal_nodes(lines)

    @staticmethod
    def no_history_risk(historys):
        """
        检查标准1：系统软件历史版本是V300R003或之后，则检查结果为通过
        备注：查询系统软件历史版本异常，检查结果也通过
        """
        return bool(historys >= "V300R003" or historys == "--" or not historys)

    @staticmethod
    def get_abnormal_nodes(lines):
        """
        检查标准2：存在任一控制器的节点最后时间字段值为True，则检查结果为不通过，否则检查结果为通过。
        """
        lines.sort(key=lambda x: x["time"])
        latest_time = lines[-1]
        node = str(latest_time.get("node_id"))
        flag = latest_time.get("flag")
        return [node] if flag == 'True' else []
