# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import time
import re

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl import fault_mode as ft
from psdk.dsl.dsl_common import get_version_info


class CheckItem(BaseCheckItem):

    def execute(self):
        version_info = get_version_info(self.dsl)
        cur_version = version_info.get("base_version").get("Current Version")

        if (cur_version.startswith("V3") and cur_version < "V300R006C30") or \
                (cur_version.startswith("V5") and cur_version < "V500R007C20"):
            return self.check_sys_flow_hang_old_version()
        """
        V3/V5新版本和Kunpeng系列均使用新版本检查方法
        """
        return self.check_sys_flow_hang_new_version()

    def check_sys_flow_hang_old_version(self):
        results = self.dsl("exec_on_all {}", self.check_old_version, target_ctrl="master")
        for result in results.values():
            if result == CheckStatus.NOT_PASS:
                return result, self.get_err_msg("check.not.pass", "item.suggestion")
            return result, ""

    """
    小于V300R006C30、V500R007C20的检查方法
    """
    def check_old_version(self):
        if self.is_sys_trace_running_timeout() or self.is_sys_flow_trace_running_timeout():
            return CheckStatus.NOT_PASS
        return CheckStatus.PASS

    def get_cur_sys_time_s(self):
        cur_sys_time = self.dsl("exec_cli 'show system general' | regex 'Time.*(\d{4}-\d{2}-\d{2}/\d{2}:\d{2}:\d{2})' "
                                "| get_index(0)")
        if not cur_sys_time:
            return 0
        """
        只截取时间，把时区信息剃掉
        """
        time_array = time.strptime(cur_sys_time.split()[0], "%Y-%m-%d/%H:%M:%S")
        cur_sys_time_s = time.mktime(time_array)
        return cur_sys_time_s

    def is_sys_trace_running_timeout(self):
        cur_sys_time_s = self.get_cur_sys_time_s()
        if not cur_sys_time_s:
            return False

        """
        超时时间为1800秒
        """
        over_time_s = 1800
        flows = self.dsl("exec_diagnose 'sys showtrace' | horizontal_parser")
        for flow in flows:
            if flow.get("Status") == "Running":
                FlowId = flow.get("FlowId")
                # trace流程中86~90不做检测
                if int(FlowId) >= 86 and int(FlowId) <= 90:
                    continue
                tracesteps = self.dsl("exec_diagnose 'sys showtrace {}' | splitlines".format(FlowId))
                cur_flow_time_s = self.query_trace_flow_cur_time(tracesteps, cur_sys_time_s)
                if (cur_sys_time_s - cur_flow_time_s) > over_time_s:
                    return True
        return False

    def is_sys_flow_trace_running_timeout(self):
        cur_sys_time_s = self.get_cur_sys_time_s()
        if not cur_sys_time_s:
            return False

        """
        超时时间为1800秒
        """
        over_time_s = 1800
        flows = self.dsl("exec_diagnose 'sys showflowtrace' | horizontal_parser")
        for flow in flows:
            if flow.get("Status") == "Running":
                FlowId = flow.get("FlowId")
                tracesteps = self.dsl("exec_diagnose 'sys showflowtrace {}' | splitlines".format(FlowId))
                cur_flow_time_s = self.query_trace_flow_cur_time(tracesteps, cur_sys_time_s)
                if (cur_sys_time_s - cur_flow_time_s) > over_time_s:
                    return True
        return False

    def query_trace_flow_cur_time(self, tracesteps, cur_sys_time_s):
        last_flow_time_s = 0
        for step in tracesteps:
            regx = ".*(?P<time>\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}).*"
            cur_step_time_list = re.compile(regx).search(step)
            if not cur_step_time_list:
                continue
            cur_step_time = cur_step_time_list.groupdict().get("time")

            time_array = time.strptime(cur_step_time, "%Y-%m-%d %H:%M:%S")
            cur_step_time_s = time.mktime(time_array)
            if cur_step_time_s > last_flow_time_s:
                last_flow_time_s = cur_step_time_s
        """
        防呆，避免未获取到正确的时间，导致误报
        """
        return cur_sys_time_s if last_flow_time_s == 0 else last_flow_time_s
    """
    大于等于V300R006C30、V500R007C20的检查方法
    """
    def check_sys_flow_hang_new_version(self):
        return self.check_sys_flow_hang()

    def check_sys_flow_hang(self):
        results = self.dsl("exec_on_all {}", self.query_sys_flow, target_ctrl="master")
        for ctrl_result in results.values():
            """
            超时时间为1800秒
            """
            over_time_s = 1800
            for result in ctrl_result:
                if int(result.get("spend_time")) > over_time_s:
                    return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion")
        return CheckStatus.PASS, ""

    def query_sys_flow(self):
        results = self.dsl("exec_diagnose 'sys showcmdlist' | splitlines | regex 'Running, spend: (?P<spend_time>\d+)'",
                           return_if=ft.DEFAULT_FT)
        return results
