# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.
from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl.dsl_common import get_version_info

MAX_HOST_NUM = 20
TARGET_VERSION = ["V500R007C71SPH150", "V500R007C73SPH120", "V500R007C71SPH137"]


class CheckItem(BaseCheckItem):
    @staticmethod
    def check_host_info(host_info):
        ret_list = [item for item in host_info if item.get("Operating System", "") == "VMware ESX"]
        return len(ret_list)

    @staticmethod
    def parse_diagnose_cli(diagnose_info, esx_host_num):
        for result in diagnose_info.values():
            if result and esx_host_num <= MAX_HOST_NUM:
                return CheckStatus.WARNING
            elif result and esx_host_num > MAX_HOST_NUM:
                return CheckStatus.NOT_PASS
        return CheckStatus.PASS

    def execute(self):
        # 先判断补丁版本是否是规定的版本
        version_info = get_version_info(self.dsl)
        patch_version = version_info.get("patch_version").get("Current Version")
        if patch_version not in TARGET_VERSION:
            return CheckStatus.PASS, ""
        # 检查该阵列上是否存在ESX主机,如果不存在则检查通过
        host_info = self.dsl("exec_developer 'show host general' | horizontal_parser")
        esx_host_num = self.check_host_info(host_info)
        if esx_host_num == 0:
            return CheckStatus.PASS, ""
        #     检查unmap开关是否打开
        unmap_result = self.dsl("exec_on_all {}", self.check_unmap_switch_is_on)
        ret = self.parse_diagnose_cli(unmap_result, esx_host_num)
        if ret != CheckStatus.PASS:
            return ret, self.get_msg("check.not.pass")
        # 检查tgt并发是否为-1
        tgt_result = self.dsl("exec_on_all {}", self.is_tgt_switch_on)
        ret = self.parse_diagnose_cli(tgt_result, esx_host_num)
        if ret != CheckStatus.PASS:
            return ret, self.get_msg("check.not.pass")
        return CheckStatus.PASS, ""

    def check_unmap_switch_is_on(self):
        """
        检查标准1：unmap开关是否打开
        """
        unmap_info = self.dsl("exec_diagnose 'devlun showunmapctrlinfo' | vertical_parser")
        for item in unmap_info:
            if "isSwitchOn" in item.keys():
                switch_info = item.get("isSwitchOn")
                if "False" == switch_info:
                    return False
                else:
                    return True
        return True

    def is_tgt_switch_on(self):
        """
        检查标准2：tgt并发是否为-1
        """
        tgt_info = self.dsl("exec_diagnose 'tgt show io maxunmap' | vertical_parser")
        for item in tgt_info:
            if "The the current number of unmap io" in item.keys():
                switch_info = item.get("The the current number of unmap io")
                if "(-1)" in switch_info:
                    return False
                else:
                    return True
        return True

