# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.entity.check_status import CheckStatus
from psdk.platform.util.product_util import compare_patch_version

PRODUCT_BIOS_VERSION_MAP = {
    "V500R007C71SPC100": {
        "start": "V500R007C71SPH136",
        "end": "V500R007C71SPH139"
    },
    "V500R007C73SPC100": {
        "start": "V500R007C71SPH116",
        "end": "V500R007C71SPH119",
    },
}
ERR_BIOS_C_MID = ["23.03.10T81", "28.30.10T02", "28.10.10T49"]


class CheckItem(BaseCheckItem):
    def execute(self):
        version_info = get_version_info(self.dsl)
        version = version_info.get("base_version").get("Current Version")
        patch_version = version_info.get("patch_version").get("Current Version")
        # 检查方法步骤1：show version all
        bios_version = self.dsl(
            "exec_cli 'show version all' | regex 'Controller:([\\s\\S]+)' | get_index(0) | vertical_parser")
        bios_versions_dict = {v.get("Controller"): v.get("BIOS Version") for v in bios_version}
        if not bios_versions_dict or version not in PRODUCT_BIOS_VERSION_MAP:
            self.logger.info("no product version in given bios version map. check pass")
            return CheckStatus.PASS, ""
        for cur_bios_ver in bios_versions_dict.values():
            if cur_bios_ver not in ERR_BIOS_C_MID:
                continue
            if not self.check_patch_version(PRODUCT_BIOS_VERSION_MAP.get(version), patch_version):
                self.logger.info("patch version in given bios version map. check not pass")
                return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion", (patch_version, ))
        return CheckStatus.PASS, ""

    def check_patch_version(self, version_dict, patch_version):
        # 返回True表明patch版本未命中该问题，返回False表明命中
        # 这里满足start < 0 和end > 0 说明补丁版本不在检查范围
        if compare_patch_version(patch_version, version_dict.get("start")) < 0:
            self.logger.info("patch version not in given bios version map. check pass")
            return True
        if compare_patch_version(patch_version, version_dict.get("end")) > 0:
            self.logger.info("patch version not in given bios version map. check pass")
            return True
        return False
