# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl.dsl_common import get_version_info
from datetime import datetime
from psdk.platform.util.product_util import compare_patch_version

# 产品版本对应的补丁版本map
product_patch_version_map = {
    'V500R007C60': '',
    'V500R007C60SPC100': 'V500R007C60SPH108',
    'V500R007C60SPC200': '',
    'V500R007C60SPC300': 'V500R007C60SPH322',
    'V500R007C70': '',
    'V500R007C70SPC200': 'V500R007C70SPH203',
    'V500R007C71': '',
    'V500R007C71SPC100': 'V500R007C71SPH111',
    'V500R007C72': '',
    'V500R007C72SPC100': 'V500R007C72SPH101',
    'V500R007C73': '',
    'V500R007C73SPC100': 'V500R007C73SPH102',
    '6.0.0': '',
    '6.0.1': '6.0.1.SPH25',
    '6.1.0': '6.1.0.SPH15',
    '6.1.2': '6.1.2.SPH7'
}
# 融合产品版本列表
V5R7_PORDUCT_VERSION_LIST = ['V500R007C60', 'V500R007C60SPC100', 'V500R007C60SPC200', 'V500R007C60SPC300', 'V500R007C70', 'V500R007C70SPC200',
                             'V500R007C71', 'V500R007C71SPC100', 'V500R007C72', 'V500R007C72SPC100', 'V500R007C73', 'V500R007C73SPC100']
# Dorado产品版本列表
DORADO_PORDUCT_VERSION_LIST = ['6.0.0', '6.0.1', '6.1.0', '6.1.2']
# 一年总天数
ONE_YEAR_OF_DAYS = 365
# 融合产品节点字符串
V5R7_NODE_POWERON_STR = "NODE_POWER_ON"
# Dorado产品节点字符串
DORADO_NODE_POWERON_STR = "NODE_POWER_ON: TaskOver"
# 节点升级事件ID
NODE_UPGRADE_EVENT_ID = "0x100F010D0023"


class CheckItem(BaseCheckItem):
    def execute(self):
        # 获取基本信息
        model = self.context.dev_node.model
        if not model:
            self.logger.info("no model info. check pass")
            return CheckStatus.PASS, ""
        version_info = get_version_info(self.dsl)
        if not version_info:
            self.logger.info("no version info. check pass")
            return CheckStatus.PASS, ""
        # 获取产品版本和补丁版本,获取不到产品版本和补丁版本返回不通过
        pro_version = version_info.get("base_version").get("Current Version")
        sph_version = version_info.get("patch_version").get("Current Version")
        if not pro_version or not sph_version:
            return CheckStatus.NOT_PASS, self.get_msg("check.not.pass.version")
        patch_version = product_patch_version_map.get(pro_version)
        # 判断当前版本补丁版本与对应产品版本补丁版本map中对应的补丁版本号比较，如果大于等于则通过，否则不通过
        if patch_version and compare_patch_version(sph_version, patch_version) >= 0:
            self.logger.info("sph version bigger or equal than patch version. check pass")
            return CheckStatus.PASS, ""
        # 查询系统当前时间和系统正常上电时间以及正常升级时间
        cur_sys_time = self.dsl("exec_cli 'show system general' | regex 'Time.*(\d{4}-\d{2}-\d{2})' | get_index(0)")
        sys_poweron_time = self.query_sys_poweron_time(pro_version)
        if not cur_sys_time or not sys_poweron_time:
            return CheckStatus.PASS, ""
        # 如果有过相关的升级则需要判断升级时间和系统上电时间哪个最新，以最新的时间为准
        sys_upgrade_time = self.get_node_upgrade_time()
        if sys_upgrade_time != "" and \
                (datetime.strptime(sys_upgrade_time, '%Y-%m-%d') - datetime.strptime(sys_poweron_time, '%Y-%m-%d')).days > 0:
            sys_poweron_time = sys_upgrade_time
        # 特定的产品版本以及运行时间需要给出不同的错误提示安装补丁
        run_time_over_year = False
        if (datetime.strptime(cur_sys_time, '%Y-%m-%d') - datetime.strptime(sys_poweron_time, '%Y-%m-%d')).days > ONE_YEAR_OF_DAYS:
            run_time_over_year = True
        check_result = CheckStatus.NOT_PASS if run_time_over_year else CheckStatus.WARNING
        return check_result, self.get_result(run_time_over_year, pro_version, patch_version)

    def get_result(self, run_time_over_year, pro_version, patch_version):
        err_upgrade_key = "check.not.pass.upgrade.must.version"\
            if run_time_over_year else "check.not.pass.upgrade.suggest.version"
        err_sph_key = "check.not.pass.sph.must.version" if run_time_over_year else "check.not.pass.sph.suggest.version"

        if (pro_version == 'V500R007C60'):
            return self.get_msg(err_upgrade_key, pro_version, 'V500R007C60SPC100', "V500R007C60SPH108")
        if (pro_version == 'V500R007C60SPC200'):
            return self.get_msg(err_upgrade_key, pro_version, 'V500R007C60SPC300', "V500R007C60SPH322")
        if (pro_version == 'V500R007C70' or pro_version == 'V500R007C71'):
            return self.get_msg(err_upgrade_key, pro_version, 'V500R007C71SPC100', "V500R007C71SPH111")
        if (pro_version == 'V500R007C72'):
            return self.get_msg(err_upgrade_key, pro_version, 'V500R007C72SPC100', "V500R007C72SPH101")
        if (pro_version == 'V500R007C73'):
            return self.get_msg(err_upgrade_key, pro_version, 'V500R007C73SPC100', "V500R007C73SPH102")
        if (pro_version == '6.0.0'):
            return self.get_msg(err_upgrade_key, pro_version, '6.0.1', "6.0.1.SPH25")
        else:
            return self.get_msg(err_sph_key, pro_version, patch_version)

    def query_node_poweron_time(self, pro_version):
        node_poweron_time = ""
        # 如果是融合产品环境需要找sys showtrace命令查找
        if pro_version in V5R7_PORDUCT_VERSION_LIST:
            node_poweron_time = self.get_v5r7_node_poweron_time()
        # 如果是dorado产品环境需要找sys showflowtrace 166命令查找
        elif pro_version in DORADO_PORDUCT_VERSION_LIST:
            node_poweron_time = self.get_dorado_node_poweron_time()
        return node_poweron_time

    def get_v5r7_node_poweron_time(self):
        node_poweron_time = ""
        node_traceflows = self.dsl("exec_diagnose 'sys showtrace' | horizontal_parser")
        for traceflow in node_traceflows:
            if traceflow.get("Setup") == V5R7_NODE_POWERON_STR:
                return traceflow.get("Date Time").split(" ")[0]
        return node_poweron_time

    def get_dorado_node_poweron_time(self):
        node_poweron_time = ""
        node_traceflows = self.dsl("exec_diagnose 'sys showflowtrace 166' | splitlines")
        for traceflow in node_traceflows:
            if DORADO_NODE_POWERON_STR in traceflow:
                return traceflow.split()[0]
        return node_poweron_time

    def query_sys_poweron_time(self, pro_version):
        poweron_times = self.dsl("exec_on_all {}", self.query_node_poweron_time, pro_version)
        poweron_min_times = ""
        for node_id, poweron_time in poweron_times.items():
            if poweron_min_times == "":
                poweron_min_times = poweron_time
                continue
            if (datetime.strptime(poweron_min_times, '%Y-%m-%d') - datetime.strptime(poweron_time, '%Y-%m-%d')).days > 0:
                poweron_min_times = poweron_time
        return poweron_min_times

    def get_node_upgrade_time(self):
        node_poweron_times = self.dsl("exec_cli 'show event |filterRow column=ID predict=equal_to value=0x100F010D0023' | horizontal_parser")
        upgrade_time = ""
        for node_poweron_time in node_poweron_times:
            if node_poweron_time.get("ID") == NODE_UPGRADE_EVENT_ID:
                node_curryon_time = node_poweron_time.get("Occurred On")
                return node_curryon_time.split("/")[0]
        return upgrade_time
