# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2023-2023. All rights reserved.
import re
import ast

from psdk.checkitem.common.base_dsl_check import BaseCheckItem
from psdk.platform.entity.check_status import CheckStatus
from psdk.dsl.dsl_common import get_version_info
from psdk.platform.util.product_util import compare_patch_version
from psdk.platform.protocol.cli_con_mgr import get_available_conn
from psdk.platform.protocol.cli_service import enter_developer_from_cli


class CheckItem(BaseCheckItem):
    def execute(self):
        # 对于风扇模块，如果后续也要做该检查项，注意风扇模块可以多选，并且可能object_id参数中有空格
        selected_fru = ast.literal_eval(self.context.execute_env.selected_fru)
        # 获取name值
        object_name = selected_fru.get("name", "")
        try:
            ret = self.check_subrack_type(object_name)
            self.logger.info("psu_replace_psu_replace_after_check!")
            # 电源风扇检查
            results_fans = self.psu_fans_after_check()
            self.logger.info(results_fans)
            if not results_fans:
                self.logger.info('results_fans is None.')
                # item.suggestion_*为.json中对应内容
                return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion_fans")
            results_state = self.psu_state_after_check()
            if not results_state:
                self.logger.info('results_state is None.')
                return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion_state")
            # 电源功率检查
            results_power = self.psu_power_after_check()
            if not results_power:
                self.logger.info('results_power is None.')
                return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion_power")
            # 电源代码检查
            results_code = self.psu_code_after_check()
            if results_code:
                return CheckStatus.PASS, ""
            else:
                return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion_code")
        except ret == False:
            # 跳框失败
            return CheckStatus.NOT_PASS, self.get_err_msg("check.not.pass", "item.suggestion_code")
        else:
            # 检查执行成功
            return CheckStatus.PASS, ""
        finally:
            self.self_exit(object_name)

    # 异常退出
    def self_exit(self, object_name):
        # CTE为控制框退出流程，DAE为IP框退出流程
        if 'CTE' in object_name:
            self.dsl("exec_mini_system 'exit'")
            self.logger.info("exit to develop")
            self.dsl("exec_developer 'exit'")
            self.logger.info("exit to admin")
        if 'DAE' in object_name:
            cli_connection = "CLI_CONNECTION"
            cli = self.dsl_obj.dsl_context.context.get_item(cli_connection + self.context.get_dev_node().ip + \
                                                            self.context.get_dev_node().sn)
            cli.execCmd("exit")
            cli.execCmd("y")
            cli.execCmd("exit")
            cli.execCmd("y")
            # 退出minisystem模式
            self.dsl("exec_mini_system 'exit'")
            self.logger.info("exit to develop")
            # 退出developer模式
            self.dsl("exec_developer 'exit'")
            self.logger.info("exit to admin")

    # 判断框类型
    def check_subrack_type(self, object_name):
        try:
            if 'CTE' in object_name:
                # 获取控制框号
                subrack_num_info = re.search(r'CTE\d', object_name).group()
                subrack_num = subrack_num_info[subrack_num_info.index("CTE") + 3]
                # 跳转至目标框节点，node_cfg为固定参数2
                self.dsl("exec_mini_system 'sshtoremoteExt %s'" % str(int(subrack_num) * 2))
                self.dsl("exec_cli '%s'" % self.context.dev_node.password, need_log=False)
                return True
            if 'DAE' in object_name:
                turn_ptr = object_name + ('.A')
                self.dsl("exec_mini_system 'sshtoipenclosure %s'" % turn_ptr)
                self.dsl("exec_cli '%s'" % self.context.dev_node.password, need_log=False)
                self.dsl("exec_cli 'minisystem'")
                return True
            else:
                return False
        except Exception:
            return False

    # 查询风扇转速
    def psu_fans_after_check(self):
        psu0_fans = self.dsl("exec_mini_system 'eam.sh psu_upgrade_get_power_fanspeed 1000 1'") # 查询PSU0风扇转速
        psu1_fans = self.dsl("exec_mini_system 'eam.sh psu_upgrade_get_power_fanspeed 1001 1'") # 查询PSU1风扇转速

        self.logger.info("psu_fans_speed Check begin!")  # log 检查
        self.logger.info(psu0_fans)  # log 检查
        self.logger.info(psu1_fans)  # log 检查
        psu0_fans_speed_info_16 = re.search(r"0x00: (\w+) (\w+)", psu0_fans)
        psu1_fans_speed_info_16 = re.search(r"0x00: (\w+) (\w+)", psu1_fans)
        psu0_fans_speed = int(psu0_fans_speed_info_16.group(2), 16) * 16 * 16 + \
                              int(psu0_fans_speed_info_16.group(1), 16)
        psu1_fans_speed = int(psu1_fans_speed_info_16.group(2), 16) * 16 * 16 + \
                              int(psu1_fans_speed_info_16.group(1), 16)
        self.logger.info(psu0_fans_speed)  # log 检查
        self.logger.info(psu1_fans_speed)  # log 检查
        self.logger.info("psu_fans_speed Check End!")  # log 检查
        if psu0_fans_speed >= 500 and psu1_fans_speed >= 500:
            return True
        else:
            return False

    # 查询电源状态
    def psu_state_after_check(self):
        psu_state = self.dsl("exec_mini_system 'eam.sh psu_upgrade_psu_state_check'") # 查询PSU工作状态
        self.logger.info("psu_state_check!")  # log 检查
        self.logger.info(psu_state)  # log 检查
        if '0x00' in psu_state:
            return True
        else:
            return False

    # 查询电源功率
    def psu_power_after_check(self):
        power_psu0_pow_info = self.dsl("exec_mini_system 'eam.sh psu_upgrade_power_check 1000 0'") # 查询PSU0功率
        power_psu1_pow_info = self.dsl("exec_mini_system 'eam.sh psu_upgrade_power_check 1001 0'") # 查询PSU1功率
        self.logger.info("psu_power_check!")  # log 检查
        # 获取回显中"power："开始的信息
        power_psu0_pow = re.search(r"power:(\d+)", power_psu0_pow_info)
        power_psu1_pow = re.search(r"power:(\d+)", power_psu1_pow_info)
        self.logger.info(power_psu0_pow)  # log 检查
        self.logger.info(power_psu1_pow)  # log 检查
        # 获取具体功率数值
        power_psu0_pow_num = int(power_psu0_pow.group(1))
        power_psu1_pow_num = int(power_psu1_pow.group(1))
        self.logger.info(power_psu0_pow_num)  # log 检查
        self.logger.info(power_psu1_pow_num)  # log 检查
        if power_psu0_pow_num and power_psu1_pow_num:
            power_total = power_psu0_pow_num + power_psu1_pow_num
            self.logger.info(power_total)  # log 检查
            if (power_psu0_pow_num * 10) > power_total and (power_psu1_pow_num * 10) > power_total:
                return True
            else:
                return False
        else:
            return False

    # 查询电源代码
    def psu_code_after_check(self):
        psu0_code = self.dsl("exec_mini_system 'eam.sh psu_upgrade_get_psu_huaweicode 1000 0'") # 查询PSU0代码
        psu1_code = self.dsl("exec_mini_system 'eam.sh psu_upgrade_get_psu_huaweicode 1001 0'") # 查询PSU0代码
        self.logger.info("psu_code_check!")  # log 检查
        self.logger.info(psu0_code)  # log 检查
        self.logger.info(psu1_code)  # log 检查
        if '02312HGY' in psu0_code or '02312HGY' in psu1_code:
            return False
        else:
            return True
























