# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from psdk.dsl.common import ReturnException
from psdk.platform.base.exception import MessageException


class BaseFaultMode(object):
    def match(self, content):
        raise NotImplementedError


class FindStr(BaseFaultMode):
    def __init__(self, search_key):
        self.search_key = search_key

    def match(self, content):
        return self.search_key.lower() in content.lower()


class ExecuteTimeout(BaseFaultMode):
    def match(self, content):
        return "time out" in content or "TOOLKIT_SEND_CMD_TIME_OUT" in content


class CmdNoSupport(BaseFaultMode):
    def match(self, content):
        return "^" in content or "command not found" in content.lower()


class NoLicense(BaseFaultMode):
    def match(self, content):
        return "no license" in content.lower


class FindException(BaseFaultMode):
    def __init__(self, code=None):
        self.code = code
        self.type = "exception"

    def match(self, ex):
        if self.code is None:
            # 缺省时，表示捕获所有异常
            return True
        if hasattr(ex, "get_code"):
            return ex.get_code() == self.code
        return False


class ReturnIfHandler:
    def __init__(self, return_setting):
        self.setting = return_setting

    def process(self, origin_ret):
        if not self.setting:
            return origin_ret
        for mode in self.setting:
            if isinstance(mode, FindException):
                continue
            if isinstance(mode, BaseFaultMode):
                mode_obj = mode
            else:
                mode_obj = mode()
            if isinstance(mode_obj, FindException):
                continue
            if mode_obj.match(origin_ret):
                raise ReturnException(self.setting.get(mode))
        return origin_ret

    def process_exception(self, ex):
        if not self.setting:
            raise ex
        for mode in self.setting:
            if isinstance(mode, BaseFaultMode):
                mode_obj = mode
            else:
                mode_obj = mode()
            if not isinstance(mode_obj, FindException):
                continue
            if mode_obj.match(ex):
                return self.setting.get(mode)
        raise ex


# 默认的故障模式
DEFAULT_FT = {
    ExecuteTimeout(): MessageException("cmd.execute.timeout"),
    FindStr("Error"): MessageException("cmd.execute.error")
}
