# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from expparser import *
from psdk.dsl.adapter import dsl_adapter
from psdk.dsl.operator.base_operator import BaseOperator
from psdk.dsl.common import DslException
from psdk.dsl.fault_mode import ReturnIfHandler


class BaseExecutor(BaseOperator):
    def __init__(self, dsl_context):
        BaseOperator.__init__(self, dsl_context)
        self.cmd = ""
        self.cli = None

    def get(self):
        """
        exec_cli 'show lun general'
        exec_cli '{}'
        :return:
        """
        key = self.get_type()
        a = CharsNotIn("'")
        b = Suppress("'")
        cmd_section = Group(b + OneOrMore(a) + b).setResultsName("cmd")

        pattern = key + cmd_section
        pattern.setParseAction(self.base_action)
        return pattern

    def get_type(self):
        return Literal("exec_cli").setResultsName("type")

    def action(self, s, loc, toks):
        self.logger.info("enter {} action ".format(toks.type))
        self.cmd = " ".join(toks.cmd)
        if self.dsl_context.args:
            self.cmd = self.cmd.format(*self.dsl_context.args)

        retry_times = self.dsl_context.kwargs.get("retry_times", 1)
        reconnected = bool(self.dsl_context.kwargs.get("reconnected"))
        conn_to_ctrl = self.dsl_context.kwargs.get("conn_to_ctrl")

        cmd_params = dict(
            need_log=self.dsl_context.kwargs.get("need_log", True),
            time_out=self.dsl_context.kwargs.get("time_out", 5 * 60),
            end_str_list=self.dsl_context.kwargs.get("end_str_list")
        )
        self.need_log = cmd_params.get("need_log")
        while retry_times > 0:
            retry_times -= 1
            self.cli = self.get_cli(conn_to_ctrl)
            try:
                flag, result = self.exec_cmd(**cmd_params)
                self.dsl_context.origin_info.append(result)
                result = ReturnIfHandler(
                    self.dsl_context.kwargs.get("return_if")).process(result)
                if not flag:
                    raise DslException(result)
                return result
            except DslException as e:
                if reconnected:
                    self.cli = self.get_cli(conn_to_ctrl)
                if retry_times <= 0:
                    self.logger.info("exec cmd={} error={}"
                                     .format(self.cmd, e.get_code()))

    def get_cli(self, conn_to_ctrl=True):
        conn_to_ctrl = True if conn_to_ctrl is None else conn_to_ctrl
        return dsl_adapter.get_con(self.dsl_context.context, True, conn_to_ctrl)

    def exec_cmd(self, **kwargs):
        raise NotImplementedError


class ExecCliDsl(BaseExecutor):

    def __init__(self, dsl_context):
        BaseExecutor.__init__(self, dsl_context)
        self.operator_type = "data_source"

    def get_type(self):
        return Literal("exec_cli").setResultsName("type")

    def exec_cmd(self, **kwargs):
        return dsl_adapter.exec_cli(self.cli, self.cmd, **kwargs)

    def get_cli(self, conn_to_ctrl=False):
        conn_to_ctrl = False if conn_to_ctrl is None else conn_to_ctrl
        return dsl_adapter.get_con(self.dsl_context.context, True, conn_to_ctrl)


class ExecDiagnoseDsl(BaseExecutor):
    def __init__(self, dsl_context):
        BaseExecutor.__init__(self, dsl_context)
        self.operator_type = "data_source"

    def get_type(self):
        return Literal("exec_diagnose").setResultsName("type")

    def exec_cmd(self, **kwargs):
        return dsl_adapter.exec_diagnose(self.cli, self.cmd, **kwargs)


class ExecDeveloperDsl(BaseExecutor):
    def __init__(self, dsl_context):
        BaseExecutor.__init__(self, dsl_context)
        self.operator_type = "data_source"

    def get_type(self):
        return Literal("exec_developer").setResultsName("type")

    def exec_cmd(self, **kwargs):
        return dsl_adapter.exec_developer(self.cli, self.cmd, **kwargs)

    def get_cli(self, conn_to_ctrl=False):
        conn_to_ctrl = False if conn_to_ctrl is None else conn_to_ctrl
        return dsl_adapter.get_con(self.dsl_context.context, True, conn_to_ctrl)


class ExecMiniSystemDsl(BaseExecutor):
    def __init__(self, dsl_context):
        BaseExecutor.__init__(self, dsl_context)
        self.operator_type = "data_source"

    def get_type(self):
        return Literal("exec_mini_system").setResultsName("type")

    def exec_cmd(self, **kwargs):
        return dsl_adapter.exec_mini_system(self.cli, self.cmd, **kwargs)
