# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

from psdk.platform.adapter import java_connector_factory as connector

"""
封装Java侧的API，为python侧提供连接的基础接口，对接ConnectorFactory
基于py4j进行接口的调用
"""
from psdk.platform.base.exception import SshConnectionException
from psdk.platform.util.base_util import ignore_all_exception
import traceback


def create_cli_conn(context, ip):
    """
    根据ip创建cli连接，支持跳转
    每个IP只能创建一个连接

    :param context: 上下文对象
    :param ip: 创建cli连接的ip
    :return: 创建成功返回cli连接，否则为None
    """
    logger = context.get_logger()
    base_dev = context.get_java_dev()

    # 先保存原有ip，SN号，使用新ip建立连接后进行恢复
    original_ip = base_dev.getIp()

    try:
        # 生成临时SN用于建立cli连接，一个SN号建立一个连接
        base_dev.setIp(ip)
        conn = CliConnection(context)
        return conn.create(base_dev)
    except Exception:
        logger.debug(traceback.format_exc())
        return None
    finally:
        # 恢复为原有ip，SN号
        base_dev.setIp(original_ip)


class CliConnection(object):
    CLI_CAN_NOT_CONNECT_CODE = "1073949185"  # 与设备通信异常，请检查网络连接或设备状态是否正常
    RE_CONNECTION_TIMES = 3

    def __init__(self, context):
        self._cli = None
        self.logger = context.get_logger()

    def create(self, java_dev):
        re_connection_times = 0
        while self._cli is None and re_connection_times < 2:
            self._cli = connector.get_cli_con(java_dev)
            re_connection_times += 1
        return self._cli

    def close(self):
        if self._cli is not None:
            try:
                self.exit_heartbeat_cli()
                self._cli.close()
            except:
                return
            finally:
                self._cli = None
        return

    def exit_heartbeat_cli(self):
        try:
            for _ in range(3):
                cliRet = self._cli.execCmdNoLogTimout("exit", 5)

                while "(y/n)" in cliRet:
                    cliRet = self._cli.execCmdNoLogTimout("y", 5)

                # Compatible for debug version.
                if "Storage:~ #" in cliRet:
                    self._cli.execCmdNoLogTimout("exit", 5)
        except Exception:
            if self.logger:
                self.logger.debug("Heart beat except.")


@ignore_all_exception
def re_conn_silence(conn):
    conn.reConnect()


def check_connect(conn, logger, allow_not_admin=False):
    """
    获取连接并检查有效性

    :param conn: 原连接
    :param logger: logger
    :param allow_not_admin: 是否允许获取到的连接是非admin模式的，如果不允许则非admin模式时需要重连接以回到admin模式
    :return:
    """
    if conn is None:
        raise SshConnectionException("con is none")

    if not conn.isConnected():
        if logger:
            logger.debug("Ssh connection not connected, "
                         "reconnecting...")
        re_conn_silence(conn)
    if allow_not_admin:
        return conn
    try:
        check_conn_normal(conn)
    except Exception:
        conn.reConnect()
    return conn


def check_conn_normal(conn):
    """
    此处检查连接是否正确，并不只是新建场景才会出现，还有可能是之前已经建立了连接，但是模式不在admin而已。
    因此判断存在场景问题，当获取到的cli是在diagnose模式时，6.1.0版本cli连接reconnect（先断开再重连）的时候导致diagnose进程残留。
    :param conn: 连接
    :return: 如果非正常模式raise异常。
    """
    from psdk.platform.protocol.cli_service import enter_target_mode
    cli_ret = conn.execCmd("show cli configuration")

    if "-bash:" in cli_ret or "Storage:~" in cli_ret:
        raise SshConnectionException("in storage mode")
    if "minisystem>" in cli_ret:
        raise SshConnectionException("in mini mode")

    if "Try 'help'" in cli_ret or "does not exist or has an incorrect format" in cli_ret:
        raise SshConnectionException("incorrect format")

    if "diagnose>" in cli_ret:
        enter_target_mode(conn, "cli", 'en')
        raise SshConnectionException("in diagnose mode")

    if "More enabled" not in cli_ret:
        raise SshConnectionException("no more enable")
