# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import re
from psdk.platform.base.exception import CmdExecuteException
from psdk.platform.util.echo_parser import get_vertical_cli_ret

DIAGNOSE_RET_END_FLAG = "diagnose>"
MINISYSTEM_MODEL_FLAG = "minisystem>"
STORAGE_MODEL_FLAG = "Storage:~ #"
DEBUG_MODEL_FLAG = ":/diagnose>"
DEVELOPER_MODEL_FLAG = r"developer@*\w*:/>"

CLI_EXECUTE_CMD_SUCCESS = "Command executed successfully"


def execute_cmd(
        cli,
        cmd,
        need_log=True,
        **kwargs
):
    """执行CLI命令的接口

    :param cli: cli对象
    :param cmd: 待执行命令
    :param need_log 是否需要记录日志
    :return: flag, cliRet, errMsg
             flag:
                    True: 执行命令正常
                    False: 执行命令不正常
             cliRet: cli回显
    """
    # 命令结束符列表
    end_str_list = kwargs.get("end_str_list")

    # 超时时间
    time_out = kwargs.get("time_out", 2 * 60)
    try:
        if end_str_list and need_log:
            cli_ret = cli.execCmd(cmd, end_str_list, time_out)
        elif end_str_list and not need_log:
            cli_ret = cli.execCmdNoLogTimout(cmd, time_out, end_str_list)
        elif not end_str_list and need_log:
            cli_ret = cli.execCmdWithTimout(cmd, time_out)
        else:
            cli_ret = cli.execCmdNoLogTimout(cmd, time_out)
        while cli_ret.strip().endswith('(y/n)') or cli_ret.strip().endswith('[y/n]'):
            cli_ret = cli.execCmd('y')
        return True, cli_ret
    except Exception:
        raise CmdExecuteException("cmd exception")


def execute_cmd_and_check_result(cli,
                                 cmd,
                                 need_log=True,
                                 end_str_list=None,
                                 time_out=2 * 60):
    """
    执行CLI命令的接口

    :param cli: cli对象
    :param cmd: 待执行命令
    :param need_log: 是否需要以有log的方式执行cli命令下发
    :param end_str_list: 命令结束符列表
    :param time_out: 超时时间
    :return: (falg, cliRet, errMsg)
             flag:
                    True: 执行命令正常
                    False: 执行命令不正常
                cliRet: cli回显
    """
    _, cli_ret = execute_cmd(cli, cmd, need_log,
                             end_str_list=end_str_list, time_out=time_out)
    return check_cli_ret(cli_ret)


def check_cli_ret(cli_ret):
    if len(cli_ret) == 0:
        raise CmdExecuteException("cli.result.is.empty", cli_ret)

    if is_not_support_feature(cli_ret):
        raise CmdExecuteException("cmd is not support", cli_ret)

    if not hasCliExecPrivilege(cli_ret):
        raise CmdExecuteException("has.not.cli.privilege", cli_ret)

    if is_none_license(cli_ret):
        raise CmdExecuteException("has.not.license", cli_ret)

    if queryResultWithNoRecord(cli_ret):
        return True, cli_ret

    if is_internal_error(cli_ret):
        raise CmdExecuteException("internal error", cli_ret)

    lines = cli_ret.splitlines()
    for line in lines:
        in_black_list, error_key = check_line_in_black_list(line)
        if in_black_list:
            raise CmdExecuteException(error_key, cli_ret)
    return True, cli_ret


def is_not_support_feature(cli_ret):
    """
    判断是否支持某特性

    :param cli_ret: cli回显
    :return:
    """
    cliRetList = cli_ret.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "use the tab key to check that the command format is correct" in lowerLine:
            return True
        if "error:" in lowerLine and "not support" in lowerLine:
            return True
        # 【注意】升级类工具专用：全是使用admin执行，所以不存在权限不足情况
        if lowerLine.strip() == "^":
            return True

    return False


def is_none_license(cliRet):
    """判断回显结果里面是否包含需要license

    :param cliRet: cli回显
    :return: True: cli回显包含需要license, False: cli回显不包含需要license
    """
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "license" in lowerLine and (
                "error" in lowerLine or "suggestion" in lowerLine):
            return True

    return False


def is_internal_error(cli_ret):
    """判断回显结果是否包含内部错误信息

    :param cli_ret:
    :return: cli回显
    """
    return "internal error" in cli_ret.lower()


def queryResultWithNoRecord(cliRet):
    """
    判断回显是否为Command executed successfully

    :param cliRet: cli回显
    :return:
    """
    if re.match(CLI_EXECUTE_CMD_SUCCESS, cliRet, re.IGNORECASE):
        return True
    return False


def check_line_in_black_list(line):
    """
    判断cli语句行是否在黑名单中

    :param line: cli语句行
    :return:  True: cli语句行在黑名单中
              False: cli语句行不在黑名单中
    """

    from psdk.platform.protocol.cli_resource import BLACKLIST_DICT
    for dictItems in BLACKLIST_DICT:
        if dictItems.get("key_word") in line:
            return True, dictItems.get("error_key", "")
    return False, ""


def needOpenDeveloperSwitch(cli, lang):
    '''
    @summary: #兼容V3R6版本，查看切developer模式开关是否打开
    '''
    developercmd = "show user_mode enabled"
    execute_success, cli_ret = execute_cmd(
        cli, developercmd, True)

    cliRetLinesList = get_vertical_cli_ret(cli_ret)
    for line in cliRetLinesList:
        if "Disabled" == line.get("Developer View", ""):
            return True, cli_ret
    return False, cli_ret


def openDeveloperSwitch(cli, lang):
    '''
            打开切换developer模式开关
    '''
    opencmd = "change user_mode enabled user_mode=developer enabled=yes"
    return execute_cmd(cli, opencmd, True)


def enter_developer_from_cli(cli, lang, debugPasswd=None):
    '''
    @summary: 进入developer模式
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, cliRet, errMsg)
        flag:
            True: 进入developer模式成功
            False: 进入developer模式失败
        cliRet: cli回显
        errMsg: 进入developer模式失败时的错误消息
    '''
    cli_ret_all = ""
    cmd = "change user_mode current_mode user_mode=developer"
    success, cli_ret = execute_cmd(cli, cmd, True)
    if not success:
        return CmdExecuteException("cannot.access.developer.mode", cli_ret)

    # 需要确认直接输入y
    cnt = 1
    while "y/n" in cli_ret and cnt <= 3:
        cli_ret = cli.execCmd("y")
        cnt += 1
    cli_ret_all += cli_ret

    if isInDeveloperMode(cli_ret):
        return True, cli_ret_all

    if "password" in cli_ret.lower():
        for i in range(0, 3):
            success, cli_ret = execute_cmd(
                cli, debugPasswd, False)
            cli_ret_all += cli_ret
            if isInDeveloperMode(cli_ret):
                return True, cli_ret_all
            if "password is wrong" in cli_ret.lower():
                raise CmdExecuteException(
                    "developer.password.is.wrong", cli_ret_all)

    raise CmdExecuteException("cannot.access.developer.mode", cli_ret_all)


def isInMinisystemMode(cliRet):
    '''
    @summary: 判断当前是否在minisystem模式下
    '''
    if re.search(MINISYSTEM_MODEL_FLAG, cliRet, re.IGNORECASE):
        return True
    return False


def isInDebugMode(cliRet):
    """
    @summary: check the ":/diagnose>" is in the return strings.
    """
    if re.search(DEBUG_MODEL_FLAG, cliRet, re.IGNORECASE):
        return True
    return False


def isInDeveloperMode(cliRet):
    '''
    @summary: 判断当前是否在developer模式下
    @param cliRet: cli回显
    @return:
        True: 当前在developer模式下
        False: 当前不在developer模式下
    '''
    if re.search(DEVELOPER_MODEL_FLAG, cliRet, re.IGNORECASE):
        return True
    return False


def isInStorageMode(cliRet):
    '''
    @summary: 判断当前是否在Storage模式下，debug 版本.
    @param cliRet: cli回显
    @return:
        True: 当前在Storage模式下
        False: 当前不在Storage模式下
    '''
    if re.search(STORAGE_MODEL_FLAG, cliRet, re.IGNORECASE):
        return True
    return False

def hasCliExecPrivilege(cliRet):
    """判断是否具有执行cli命令的权限

    :param cliRet: cli回显
    :return:
    """
    for line in cliRet.splitlines():
        if line.strip() == "^":
            return False
        if "use the tab key to check that the command format is correct and then try again" in line.lower():
            return False
        if "does not have the permission" in line:
            return False
    return True
