# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2022. All rights reserved.

from psdk.platform.util import json_util
from psdk.platform.protocol.rest import rest_constant
from psdk.platform.base.exception import RestExecuteException
from psdk.platform.adapter import java_connector_factory as connector


class RestService(object):
    def __init__(self, dev_node, rest_type=""):
        self.dev_node = dev_node
        self.rest_type = rest_type

    def exec_post(self, uri, param_dict):
        """
        执行post rest请求
        :param uri: 请求uri
        :param param_dict: 请求参数
        :return:
        """
        con = self.get_connection()
        url = self.get_url(con, uri)
        try:
            param_json = json_util.covert_dict_to_json(param_dict)
            response_str = con.execPost(url, param_json).getContent()
            response = json_util.covert_json_to_dict(response_str)
            err_code = self.get_err_code(response)
            if err_code != "0":
                raise RestExecuteException(err_code)
            return self.get_data(response)
        except Exception as ex:
            self.handle_rest_error(ex)

    def exec_put(self, uri, param_dict):
        """
        执行post rest请求
        :param uri: 请求uri
        :param param_dict: 请求参数
        :return:
        """
        con = self.get_connection()
        url = self.get_url(con, uri)
        try:
            param_json = json_util.covert_dict_to_json(param_dict)
            response_str = con.execPut(url, param_json).getContent()
            response = json_util.covert_json_to_dict(response_str)
            err_code = self.get_err_code(response)
            if err_code != "0":
                raise RestExecuteException(err_code)
            return self.get_data(response)
        except Exception as ex:
            self.handle_rest_error(ex)

    def exec_get(self, uri, param_dict):
        con = self.get_connection()
        url = self.get_url(con, uri)
        try:
            param_json = json_util.covert_dict_to_json(param_dict)
            response_str = con.execGet(url, param_json).getContent()
            response = json_util.covert_json_to_dict(response_str)
            err_code = self.get_err_code(response)
            if str(err_code) != "0":
                raise RestExecuteException(err_code)
            return self.get_data(response)
        except Exception as ex:
            self.handle_rest_error(ex)

    @classmethod
    def handle_rest_error(cls, ex):
        if isinstance(ex, RestExecuteException):
            raise ex
        raise RestExecuteException(rest_constant.ErrorCode.REST_CAN_NOT_EXECUTE)

    def exec_get_for_page(self, uri, batch_size=100):
        all_datas = []
        index = 0
        while True:
            range_str = "range=[%s-%s]" % (index, index + batch_size)
            uri_with_range = self.add_range_uri(uri, range_str)
            datas = self.exec_get(uri_with_range)
            all_datas.extend(datas)
            if len(datas) < batch_size:
                break
            index = index + batch_size
        return all_datas

    def exec_delete(self, uri, param_dict):
        con = self.get_connection()
        url = self.get_url(con, uri)
        try:
            param_json = json_util.covert_dict_to_json(param_dict)
            response_str = con.execDelete(url, param_json).getContent()
            response = json_util.covert_json_to_dict(response_str)
            err_code = self.get_err_code(response)
            if err_code != "0":
                raise RestExecuteException(err_code)
            return self.get_data(response)
        except Exception as ex:
            self.handle_rest_error(ex)

    def get_connection(self):
        return connector.get_rest_con(self.dev_node)

    @classmethod
    def add_range_uri(cls, uri, range_str):
        if "?" in uri:
            return uri.replace("?", "?" + range_str + "&")
        return uri + "?" + range_str

    @classmethod
    def get_url(cls, rest, uri):
        return rest.getBaseUrl() + uri

    @classmethod
    def get_err_code(cls, response):
        return str(response.get("error", {}).get("code", "0"))

    @classmethod
    def get_data(cls, response):
        return response.get("data", [])
