# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.

import re


def is_svp_dev(dev_type, dev_version):
    from psdk.platform.adapter import java_common
    return java_common.has_svp_module(dev_type, dev_version)


def is_support_cross_engine_ssh(dev_ver, dev_type):
    """
    通过版本型号判断是否支持跨ssh连接
    :return: True/False
    """
    support_start_version = {"V3": "V300R006C00SPC100",
                             "V5": "V500R007C00SPC100"}
    if is_digital_version(dev_ver):
        return True
    if "Dorado" in dev_type:
        if dev_ver < "V300R001C20":
            return False
        return True
    for version_start in support_start_version:
        version = support_start_version.get(version_start)
        if dev_ver.startswith(version_start) and dev_ver < version:
            return False
    return True


def is_digital_version(version):
    """
    判断是否为数字版本号
    :param version: 版本号
    :return: True：数字版本号，False：非数字版本号
    """
    if re.compile(r"\d+\.\d+.*").match(version):
        return True
    return False


def compare_str_version(version1, version2):
    if version1 == version2:
        return 0
    return 1 if version1 > version2 else -1


def compare_version(version1, version2):
    """
    比较版本大小：
    """
    version1 = get_version_without_patch(version1)
    version2 = get_version_without_patch(version2)
    if is_kunpeng(version1) and is_kunpeng(version2):
        return compare_str_version(format_kunpeng_version(version1),
                                   format_kunpeng_version(version2))
    if is_digital_version(version1) and is_digital_version(version2):
        return compare_str_version(format_digital_version(version1),
                                   format_digital_version(version2))
    return compare_str_version(version1, version2)


def compare_patch_version(version1, version2):
    return compare_str_version(format_patch_version(version1),
                               format_patch_version(version2))


def is_kunpeng(version):
    return "kunpeng" in version.lower()


def get_version_without_patch(version):
    """
    去掉版本的补丁，然后返回版本。如：6.1.0.SPH10，返回6.1.0
    :param version:
    :return:
    """
    return version.split(".SPH")[0]


def format_kunpeng_version(version):
    return version.replace("Kunpeng", "").strip()


def format_digital_version(digital_version):
    versions = digital_version.split(".")
    ret = [version.zfill(5) for version in versions[:2]]
    third_sector = versions[2]
    if "RC" in third_sector:
        third_digital_versions = third_sector.split("RC")
        third_sector = "{}RC{}".format(third_digital_versions[0].zfill(5), third_digital_versions[1].zfill(5))
    if third_sector.isdigit():
        third_sector = "{}RC{}".format(third_sector.zfill(5), "99999")
    ret.append(third_sector)
    return ".".join(ret + versions[3:])


def format_patch_version(version):
    find_versions = re.findall(r"SPH(\d+)", version)
    sph_ver = find_versions[0] if find_versions else "0"
    return "SPH{}".format(sph_ver.zfill(5))


def get_node_id_by_ctrl_id(ctrl_id, one_engine_ctrl_num):
    ctrl_trans_dict = {"A": 0, "B": 1, "C": 2, "D": 3}
    eng_id = int(ctrl_id[:-1])
    ctrl_name = ctrl_id[-1]
    node_id = eng_id * one_engine_ctrl_num + ctrl_trans_dict.get(ctrl_name)
    return str(node_id)


def get_ctrl_id_by_node_id(node_id, one_engine_ctrl_num):
    ctrl_trans_dict = {"0": "A", "1": "B", "2": "C", "3": "D"}
    node_id = int(node_id)
    return str(int(node_id / one_engine_ctrl_num)) + ctrl_trans_dict[
        str(node_id % one_engine_ctrl_num)]
