#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2021-2021. All rights reserved.
# 获取EMC_Isilon设备基本信息
import re
import traceback

NA = "NA"


def execute(context):
    """
    添加Isilon设备，查询设备基本信息
    :param context: 上下文
    :return: 查询结果
    """
    try:
        cli = context.get("SSH")
        config_mode_content = execute_cmd(cli, "isi config")
        # 进入config模式后需要执行quit退出
        execute_cmd(cli, "quit")
        sn, version = get_sn_and_version_from_config(config_mode_content)
        if not sn:
            sn = get_serial_number(cli)
        if not version:
            version = get_version(cli)
        return {"name": "EMC", "type": "Isilon", "sn": sn, "version": version}
    except Exception:
        logger = context.get("Logger")
        logger.error("Get EMC Isilon login info error, reason: %s" % str(traceback.format_exc()))
        return {"name": NA, "type": NA, "sn": NA, "version": NA}


def get_sn_and_version_from_config(config_mode_content):
    """
    根据config模式信息获取SN和版本信息
    Node build: Isilon OneFS v8.1.0.2 B_MR_8_1_0_2_105(RELEASE)
    Node serial number: JWXNT175100232
    :param config_mode_content: 进入config模式显示信息
    :return: SN, Version
    """
    sn = ''
    version = ''
    for line in config_mode_content.splitlines():
        if not line:
            continue
        strip_line = line.strip()
        items = strip_line.split()
        if strip_line.startswith("Node build:") and len(items) > 4:
            version = items[4]
        if strip_line.startswith("Node serial number:") and len(items) > 3:
            sn = items[3]
    return sn, version


def get_serial_number(cli):
    """
    获取设备序列号
    :param cli: ssh连接
    :return: sn
    """
    sn_info = execute_cmd(cli, "isi license list --verbose")
    for line in sn_info.splitlines():
        line = line.strip()
        if not line.startswith("OneFS License ID:"):
            continue
        items = line.split(":")
        if len(items) > 1:
            return items[1].strip()
    return cli.getHost()


def get_version(cli):
    """
    获取设备版本号
    :param cli: ssh连接
    :return: version
    """
    version_info = execute_cmd(cli, "isi version")
    for line in version_info.splitlines():
        line = line.strip()
        if not line.startswith("Isilon OneFS"):
            continue
        items = re.split("\\s+", line)
        if len(items) > 2:
            return items[2].strip()
    return NA


def execute_cmd(cli, cmd):
    """
    执行isilon的命令，可能会遇见连接异常，导致回文错误（多行长的空白行，没有命令的回文），此时需要重新连接
    :param cli: ssh连接
    :param cmd: cmd
    :return: 回文
    """
    content = cli.execCmdHasLogTimout(cmd, 60)
    for line in content.splitlines():
        if line.strip().endswith(cmd):
            return content
    cli.reConnect()
    return cli.execCmdHasLogTimout(cmd, 60)
