#  coding=UTF-8
#  Copyright (c) Huawei Technologies Co., Ltd. 2019-2021. All rights reserved.
# 获取华为存储设备基本信息

import traceback


def execute(context):
    """
    执行主函数
    :param context: 上下文
    :return: 设备基本信息
    """
    cli = context.get("SSH")
    try:
        dev_info = get_common_system_info(cli)
        if not dev_info:
            dev_info = get_tv1_system_info(cli)
        if dev_info:
            return dev_info
    except Exception:
        logger = context.get("Logger")
        logger.error("get huawei system info error, reason:%s" % str(traceback.format_exc()))
    return {"name": "", "series": "", "type": "", "sn": "", "version": ""}


def get_tv1_system_info(cli):
    """
    获取TV1设备系统信息，系统信息为键值对类型，纵向排列，以“|”分隔
    例如:
    admin:/>showsys
    ===============================================
                   System Information
    -----------------------------------------------
    System Name           | Device_name
    Device Type           | OceanStor xxx
    ...
    ===============================================
    admin:/>
    :param cli: cli连接
    :return: 设备信息字典
    """
    system_info = cli.execCmdHasLogTimout("showsys", 50)
    if "not exist" in system_info:
        return {}
    system_info_dict = {}
    for line in system_info.splitlines():
        if " | " in line:
            info_pair = line.split(" | ")
            system_info_dict[info_pair[0].strip()] = info_pair[1].strip()
    dev_model = system_info_dict.get("Device Model", "NA")
    dev_type = get_device_type(system_info_dict.get("Device Type", dev_model))
    return {"name": "Huawei",
            "series": dev_type.split()[-1],
            "type": dev_type,
            "sn": system_info_dict.get("Device Serial Number", "NA"),
            "version": system_info_dict.get("Product Version", "NA")}


def get_common_system_info(cli):
    """
    获取TV1之后的设备通用的系统信息，系统信息为键值对类型，纵向排列，以“:”分隔
    例如：
    admin:/>show system general
    System Name         : Device_Name
    Health Status       : Normal
    Running Status      : Normal
    ...
    admin:/>
    :param cli: cli连接
    :return: 设备信息字典
    """
    system_info = cli.execCmdHasLogTimout("show system general", 50)
    if "System Name" not in system_info:
        return {}
    system_info_dict = {}
    for line in system_info.splitlines():
        if " : " in line:
            info_pair = line.split(" : ")
            system_info_dict[info_pair[0].strip()] = info_pair[1].strip()
    dev_type = get_device_type(system_info_dict.get("Product Model", "NA"))
    product_version = system_info_dict.get("Product Version", "NA")
    patch_version = system_info_dict.get("Patch Version", "")
    version = get_version(product_version, patch_version)
    return {"name": "Huawei",
            "series": dev_type.split()[-1],
            "type": dev_type,
            "sn": system_info_dict.get("SN", "NA"),
            "version": version}


def get_device_type(device_type):
    """
    若设备类型没有OceanStor，自动拼接
    :param device_type: 采集的设备型号
    :return: OceanStor xxx
    """
    if not device_type.lower().startswith("oceanstor"):
        return "OceanStor " + device_type
    return device_type


def get_version(version, patch_version):
    """
    获取版本号
    :param version: 版本号
    :param patch_version: 补丁版本信息
    :return: 版本号
    """
    if patch_version:
        return version + patch_version.split()[-1]
    return version
