# -*- coding: UTF-8 -*-

#交换机型号
SWITCH_TYPE_DICT = {
"71":"300",
"118":"6505",
"109":"6510",
"133":"6520",
"120":"DCX8510-4",
"121":"DCX8510-8"
}

#交换机对应内部使用时的型号
SWITCH_MODEL_DICT = {
"300":"SNS2124",
"6505":"SNS2224",
"6510":"SNS2248",
"6520":"SNS3096",
"DCX8510-4":"SNS5192",
"DCX8510-8":"SNS5384"
}

def execute(cli):
    """
    name（设备名称）----Value
    type（设备类型）----Value
    sn（设备序列号）----Value
    version（产品版本号）----Value
    """
    devInfoDict = {}
    
    #无法下发CLI则认为设备异常
    if False == checkSysNormal(cli):
        return devInfoDict
    
    (flagName, name)      = getSwitchName(cli)
    (flagType, type)      = getSwitchType(cli)
    (flageSN, sn)         = getSwitchWWN(cli)
    (flagVersion,version) = getSwitchVersion(cli)
    
    devInfoDict = {"name":name, "type":type, "sn":sn, "version":version}
    return devInfoDict


def excuteCmdInCliMode(cli, cmd, isHasLog):
    '''
    @summary: 获取cli模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param isHasLog: 是否需要以有log的方式执行cli命令下发
    @return: (falg, cliRet, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        cliRet: cli回显
        errMsg: 错误消息
    '''
    errMsg = ""
    cliRet = ""
    
    if cli is None:
        return (False, cliRet, errMsg)
    
    try:
        if isHasLog:
            cliRet = cli.execCmd(cmd)
        else:
            cliRet = cli.execCmdNoLog(cmd)
    except:
        return (False, cliRet, errMsg)
    
    if len(cliRet) == 0:
        return (False, cliRet, errMsg)
        
    return (True, cliRet, errMsg)

def checkSysNormal(cli):
    '''
    @summary: 查看系统是否正常，无法下发CLI则异常
    @param cli: cli对象
    @param lang: 语言lang
    @return: 
        flag: 
            True： 正常
            False： 异常
    '''     
    cmd      = "switchshow"
    checkRet = excuteCmdInCliMode(cli, cmd, True)
    if checkRet[0] != True: 
        return False
    return True
    
    
def getSwitchName(cli):
    '''
    @summary: 获取交换机名称
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True： 判断时正常
            False： 判断时异常
        ret: 
            flag为True时，交换机名称
            flag为False时，返回cli回显
        errMsg: 错误时的消息
    '''   
  
    cmd      = "switchshow"
    checkRet = excuteCmdInCliMode(cli, cmd, True)
    if checkRet[0] != True: 
        return (False, "--")
    
    cliRet     = checkRet[1]
    cliRetList = cliRet.splitlines()
    switchName = "" 
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName  = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "switchName":
            switchName = fieldValue
            
        if len(switchName) != 0 and switchName != "--":
            return (True, switchName)
        
    return (False, "--")

def getSwitchType(cli):
    '''
    @summary: 获取交换机类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True： 判断时正常
            False： 判断时异常
        ret: 
            flag为True时，交换机类型
            flag为False时，返回cli回显
        errMsg: 错误时的消息
    '''   
  
    cmd      = "switchshow"
    checkRet = excuteCmdInCliMode(cli, cmd, True)
    if checkRet[0] != True: 
        return (False, "--")
    
    cliRet     = checkRet[1]
    cliRetList = cliRet.splitlines()
    switchType = "" 
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName  = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "switchType":
            switchTypeInfo = fieldValue
            switchTypeNum  = switchTypeInfo.split(".")[0]
            switchModel    = SWITCH_TYPE_DICT.get(switchTypeNum,"")
            switchType     = SWITCH_MODEL_DICT.get(switchModel,"--")
            
        if len(switchType) != 0 and switchType != "--":
            return (True, switchType)
        
    return (False, "--")


def getSwitchVersion(cli):
    '''
    @summary: 获取交换机版本号
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True： 判断时正常
            False： 判断时异常
        ret: 
            flag为True时，交换机版本号
            flag为False时，返回cli回显
        errMsg: 错误时的消息
    '''   
  
    cmd      = "version"
    checkRet = excuteCmdInCliMode(cli, cmd, True)
    if checkRet[0] != True: 
        return (False, "--")
    
    cliRet        = checkRet[1]
    cliRetList    = cliRet.splitlines()
    switchVersion = "" 
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName  = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Fabric OS":
            switchVersion = fieldValue
            
        if len(switchVersion) != 0 and switchVersion != "--":
            return (True, switchVersion)
        
    return (False, "--")

def getSwitchWWN(cli):
    '''
    @summary: 获取交换机WWN号
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True： 判断时正常
            False： 判断时异常
        ret: 
            flag为True时，交换机WWN号
            flag为False时，返回cli回显
        errMsg: 错误时的消息
    '''   
  
    cmd      = "switchshow"
    checkRet = excuteCmdInCliMode(cli, cmd, True)
    if checkRet[0] != True: 
        return (False, "--")
    
    cliRet     = checkRet[1]
    cliRetList = cliRet.splitlines()
    switchWwn  = "" 
    for line in cliRetList:
			
        fields = line.split(":")
        if len(fields) < 2 :
            continue
        
        fieldName = fields[0].strip()      
        if fieldName == "switchWwn":
            switchWwn = line[line.index(":")+1:].strip()
            
        if len(switchWwn) != 0 and switchWwn != "--":
            return (True, switchWwn)
        
    return (False, "--")
