# -*- coding: UTF-8 -*-
import traceback
import re
from contextUtil import getLang, getDeveloperPwd


#########################################
#      异常回显判断脚本
# 检查回显是否异常
#########################################
#CLI回显结果异常，直接判断巡检不通过
CLI_RET_FAULT = -1
#CLI回显结果成功，直接判断巡检通过
CLI_RET_SUCCESS = 0
#CLI回显包含状态值，需要通过后续逻辑判断是否通过
CLI_RET_OK =1

PRODUCT_VESION = "V100R001C00"

#结束标志
CLI_EXEC_SUCCESS = "Command executed successfully"

NOT_SUPPORT_CMD_FLAG = "\^"


#模式标志
SUPER_USER_FLAG = "admin:/>"

DEVELOPER_MODEL_FLAG = "developer:/>"

CLI_RET_END_FLAG = ":/>"

KEY_WORD_VALID_LINES = 4

#重试次数，防止多次重试死循环
MAX_RETRYS = 5


#异常回显黑名单，针对捕捉到回显信息，判断回显是否正常，若是不正常，直接巡检不通过
blackDict = [
{"key_word":"-bash:",
 "msg_zh":u"\n统没有运行在正常CLI模式",
 "msg_en":"\nSystem is not in right CLI mode"},
          
{"key_word":"minisystem>",
 "msg_zh":u"\n系统没有运行在正常CLI模式",
 "msg_en":"\nSystem is not in right CLI mode"}, 
                            
{"key_word":"upgrade:/>",
 "msg_zh":u"\n系统处于升级模式",
 "msg_en":"\nSystem is in upgrade Mode"},
           
{"key_word":"The system is powering off",
 "msg_zh":u"\n系统不正常，处于下电模式",
 "msg_en":"\nSystem is abnormal, powering off"},
 
{"key_word":"system is busy",
 "msg_zh":u"\n系统忙,请稍后重试",
 "msg_en":"\nSystem is busy.Please try again later"},
           
{"key_word":"Socket connect failed",
 "msg_zh":u"\nSocket连接失败",
 "msg_en":"\nSocket connect failed"},
 
{"key_word":"Processing...",
 "msg_zh":u"\n正在处理",
 "msg_en":"\nSystem is busy"},
 
{"key_word":"Error: Operation timeout.",
 "msg_zh":u"\n执行超时",
 "msg_en":"\nOperation timeout"},
 
{"key_word":"Error:",
 "msg_zh":u"\n命令执行出错",
 "msg_en":"\nCmd execute Error"},
             
{"key_word":"Receive message time out",
 "msg_zh":u"\n信息查询超时",
 "msg_en":"\nReceive message time out"}]


#白名单，回显中捕捉到这列关键字时，巡检通过，巡检通过时无法添加详细信息，只需要列出关键字即可
whiteList = [NOT_SUPPORT_CMD_FLAG,CLI_EXEC_SUCCESS]

#行黑名单，针对捕捉到某行回显信息，判断当前行是否显示异常
blackLineDict = [
{"key_word":"Receive message time out",
 "msg_zh":u"\n信息查询超时",
 "msg_en":"\nReceive message time out"},
           
{"key_word":"Socket connect failed",
 "msg_zh":u"\nSocket连接失败",
 "msg_en":"\nSocket connect failed"}]

# **************************************************************************** #
# 函数名称: isChinese
# 功能说明: 判断当前语言是否为中文
# 输入参数: lang
# 输出参数: True or False
# **************************************************************************** # 
def isChinese(lang):
    if "zh"==lang:
        return True
    return False

# **************************************************************************** #
# 函数名称: isExecSuccess
# 功能说明: 为了加速判定，根据回显结果判定命令是否执行成功（一般针对无记录的情况）
# 输入参数: cliRet（CLI回显）
# 输出参数: True or False:
# **************************************************************************** # 
def isExecSuccess(cliRet):
    if re.search(CLI_EXEC_SUCCESS, cliRet, re.IGNORECASE):
        return True
    return False

# **************************************************************************** #
# 函数名称: isSuperUserRet
# 功能说明: 根据CLI回显判断是否为超级管理员用户
# 输入参数: cliRet
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def isSuperUserRet(cliRet):
    
    if re.search(SUPER_USER_FLAG, cliRet, re.IGNORECASE):
        return True
    return False

# **************************************************************************** #
# 函数名称: isSuperUserCli
# 功能说明: 根据传入的SSH连接执行CLI命令后判断是否为超级管理员用户
# 输入参数: cliRet
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def isSuperUserCli(cli):
    cliRet = cli.execCmdNoLog("show system general")
    lineList = cliRet.splitlines()
    
    #用户名一定包含在最后一行中
    if SUPER_USER_FLAG == lineList[-1].replace(" ",""):
        return True

    return False

# **************************************************************************** #
# 函数名称: isSuperUserObj
# 功能说明: 根据根据框架传入的参数判断是否为超级管理员用户
# 输入参数: cliRet
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def isSuperUserObj(devObj):
    if "admin" ==  devObj.get("devInfoMap").get("userName"):
        return True
    return False

# **************************************************************************** #
# 函数名称: isDeveloperModle
# 功能说明: 判断当前捕捉到的CLI回显是否在Developer模式下
# 输入参数: cliRet
# 输出参数: True:Developer模式，False:非Developer模式
# **************************************************************************** # 
def isDeveloperModle(cliRet):
    if re.search(DEVELOPER_MODEL_FLAG, cliRet, re.IGNORECASE):
        return True
    return False

# **************************************************************************** #
# 函数名称: getCurrentVesion
# 功能说明: 获取当前系统运行的版本号，精确到SPC
# 输入参数: cli
# 输出参数: currentVesion
# **************************************************************************** # 
def getCurrentVesion(cli):
    currentVesion = ""
    cliRet = cli.execCmdNoLog("show upgrade package")
    lineList = cliRet.splitlines()
    
    #直接从包含版本号的行开始
    for line in lineList[5:-1]:
        columnList = line.split()
        #找到关键字，直接返回当前版本号
        for column in columnList[2:-1]:
            if -1 != column.find(PRODUCT_VESION):
                currentVesion = column[0:17]
                return currentVesion           
    return currentVesion


# **************************************************************************** #
# 函数名称: isLessThanSupportVesion
# 功能说明: 获取当前系统运行的版本号，精确到SPC
# 输入参数: cli,supportVesion(命令开始支持的版本)
# 输出参数: currentVesion
# **************************************************************************** # 
def isLessThanSupportVesion(cli, supportVesion):
    currentVesion =  getCurrentVesion(cli)
    if ""==currentVesion:
        return False
    
    if currentVesion<supportVesion:
        return True
    return False

# **************************************************************************** #
# 函数名称: getUserName
# 功能说明: 获取当前用户名
# 输入参数: cli
# 输出参数: userName
# **************************************************************************** # 
def getUserName(cli):
    userName = ""
    cliRet = cli.execCmdNoLog("show system general")
    lineList = cliRet.splitlines()
    
    #用户名一定包含在最后一行中
    if lineList[-1].find(CLI_RET_END_FLAG):
        userName = lineList[-1].replace(" ","").replace(CLI_RET_END_FLAG,"")
    
    return userName

# **************************************************************************** #
# 函数名称: getUserName
# 功能说明: 获取登录用户名的权限级别
# 输入参数: cli
# 输出参数: super_admin,guest或admin
# **************************************************************************** # 
def getUserPrivilege(cli):
    
    userName = getUserName(cli) 
    if ""== userName:
        return ""
    
    cliRet = cli.execCmdNoLog("show user user_name="+userName)
    lineList = cliRet.splitlines()
    
    #用户名一定包含在最后一行中
    for line in lineList[4:-1]:
        columnList = line.split()
        if len(columnList)>2:
            if userName == columnList[0]:
                return columnList[1]
    return ""

# **************************************************************************** #
# 函数名称: change2cli
# 功能说明: 任何模式尝试切换到CLI模式        
# 输入参数: cliRet
# 输出参数: True:超级用户，False:非超级用户
# **************************************************************************** # 
def anyModel2Cli(cli):
    index =0 
    cliRet = cli.execCmdNoLog("show system general")
    
    #密码输入错误时及正常情况从developer模式下退出
    if -1 != cliRet.find("Password") or -1 != cliRet.find(DEVELOPER_MODEL_FLAG):
        cliRet = cli.execCmd("exit")
        while(-1 == cliRet.find(SUPER_USER_FLAG)):
            index+=1
            cliRet = cli.execCmd("exit")
            if -1!=cliRet.find("Are you sure to exit?(y/n):"):
                cliRet = cli.execCmd("n")
            if index>MAX_RETRYS:
                break
            
# **************************************************************************** #
# 函数名称: change2Developer
# 功能说明: 切换到developer模式
# 输入参数: cli,py_java_env
# 输出参数: True:，False:
# **************************************************************************** # 
def change2Developer(cli, dataDict):
    errMsg = ""
    flag = False
    lang = dataDict.getLang(dataDict)
    if None == cli:    
        return (flag,"",errMsg)
    
    #guest用户无法切换到developer模式
    userLevel = getUserPrivilege(cli)
    if "guest" ==userLevel or ""==userLevel:
        if isChinese(lang):
            errMsg = u"\n系统没有运行在admin模式，无法进入调试模式"
        else:
            errMsg = "\nSystem is not in admin model, can't enter to debug model"
        return (flag,"",errMsg)

    cliRet = cli.execCmd("change user_mode current_mode user_mode=developer")
    
    password = str(dataDict.getDeveloperPwd(dataDict))
    if(None == password or ""== password):
        if isChinese(lang):
            errMsg = u"\n获取developer密码为空，请确定是否已配置密码"
        else:
            errMsg = "\nThe obtained developer password is blank. Check whether the password has been configured."
        return (flag, cliRet, errMsg)
    
    cliRet = cli.execCmdNoLog(password)
    
    #已经成功进入Developer模式
    if isDeveloperModle(cliRet):
        flag = True
    else:
        anyModel2Cli(cli)
        if isChinese(lang):
            if -1!=cliRet.find("Password is wrong") or -1!=cliRet.find("Password:"):
                errMsg = u"\ndeveloper密码配置错误"
            else:
                errMsg = u"\n系统不能进入developer模式"
        else:
            if -1!=cliRet.find("Password is wrong") or -1!=cliRet.find("Password:"):     
                errMsg = "\nThe password of developer is wrong"
            else:
                errMsg = "\nIt failed to access the developer mode" 
    return (flag, cliRet, errMsg)


# **************************************************************************** #
# 函数名称: developer2Cli
# 功能说明: developer模式下退出到CLI模式
# 输入参数: cli
# 输出参数: 无
# **************************************************************************** # 
def developer2Cli(cli):
    cli.execCmd("exit")
   
# **************************************************************************** #
# 函数名称: isEndLine
# 功能说明: 判断此行是否是回显结束行
# 输入参数: cliLine（某行CLI回显）
# 输出参数: True or False
# **************************************************************************** #       
def isEndLine(cliLine):
    
    #空行，无效行 非结束标志
    if "" == cliLine or None == cliLine:
        return False
    
    if -1 != cliLine.find(CLI_RET_END_FLAG):
        return True     
    return False

# **************************************************************************** #
# 函数名称: isSkipLineWithStart
# 功能说明: 跳过以某关键字起头的行，空行，无效行，结束行也要跳过
# 输入参数: cliLine（某行CLI回显），startWordList（关键字列表）
# 输出参数: True or False
# **************************************************************************** #       
def isSkipLineWithStart(cliLine, startWordList):
    
    #未知对象，跳过
    if None == cliLine:
        return True
    
    #去掉头尾空格后为空，表示此行原始为空行，需要跳过
    line=cliLine.strip()
    if "" == line:
        return True
    
    #结束行也要跳过
    if -1 != line.find(CLI_RET_END_FLAG):
        return True
        
    #没传入任何关键字，不跳过
    if 0 == len(startWordList):
        return False
    

    for startWord in startWordList:
        #是以XXX起头的行，需要跳过
        if line.startswith(startWord):
            return True
           
    return False

# **************************************************************************** #
# 函数名称: isSkipLineWithContain
# 功能说明: 跳过包含某关键字起头的行，空行，无效行，结束行也要跳过
# 输入参数: cliLine（某行CLI回显），startWordList（关键字列表）
# 输出参数: True or False
# **************************************************************************** #       
def isSkipLineWithContain(cliLine, containWordList):
    
    #未知对象，跳过
    if None == cliLine:
        return True
    
    #去掉头尾空格后为空，表示此行原始为空行，需要跳过
    line=cliLine.strip()
    if "" == line:
        return True
    
    #结束行也要跳过
    if -1 != line.find(CLI_RET_END_FLAG):
        return True
        
    #没传入任何关键字，不跳过
    if 0 == len(containWordList):
        return False
    

    for containWord in containWordList:
        #包含XXX关键字的行，需要跳过
        if -1 != line.find(containWord):
            return True
           
    return False

# **************************************************************************** #
# 函数名称: isLineInBlackDict
# 功能说明: 判断某行回显是否在黑名单中，在黑名单中表示CLI回显存在异常，巡检不通过，
# 不用逻辑判断，最终应返回CLI_RET_FAULT
# 输入参数: cliLine（某行CLI回显）
# 输出参数: True or False
# **************************************************************************** #    
def isLineInBlackDict(cliLine, lang):
    
    errMsg = ""
    
    #空行，无效行，回显正常时会包含，不属于黑名单
    if None == cliLine or "" == cliLine:
        return (False, errMsg)
    
    for dictItems in blackDict:
        #黑名单中找到，表示回显无效，直接退出
        if -1 != cliLine.find(dictItems.get("key_word")):
            if isChinese(lang):
                errMsg = dictItems.get("msg_zh")
            else:
                errMsg = dictItems.get("msg_en")
            return (True, errMsg)
    return (False, errMsg)


# **************************************************************************** #
# 函数名称: isLineInWhiteList
# 功能说明: 判断某行回显是否在白名单中，在白名单中表示CLI执行完成，巡检通过，
# 不用逻辑判断，最终应返回CLI_RET_SUCCESS
# 输入参数: cliLine（某行CLI回显）
# 输出参数: True or False
# **************************************************************************** #    
def isLineInWhiteList(cliLine):
       
    keyWordMsg = ""
    
    #空行，无效行，回显正常时会包含，不属于白名单
    if None == cliLine or "" == cliLine:
        return (False, keyWordMsg)

    for keyWords in whiteList:
        #在白名单中找到关键字，表示命令执行成功，直接退出
        if re.search(keyWords, cliLine, re.IGNORECASE):
            keyWordMsg = keyWords
            return (True, keyWordMsg)
        
        #无license情况通过
        if re.search("license", cliLine, re.IGNORECASE):
            if -1 != cliLine.find("Error:") or -1 != cliLine.find("Suggestion:"):
                keyWordMsg = "license"
                return (True, keyWordMsg)
            
    return (False,keyWordMsg)

# **************************************************************************** #
# 函数名称: isNormalLine
# 功能说明: 判断某行回显是否正常
# 输入参数: cliLine（某行CLI回显）
# 输出参数: True or False
# **************************************************************************** #       
def isNormalLine(cliLine, lang):
    errMsg = ""
    
    #空行，无效行正常
    if None == cliLine or "" == cliLine:
        return (True, errMsg)
    
    #异常的关键字存在于前几行，防止全部判断时误判
    #某行是否因为超时或是其它原因，导致端口信息查询不到的情况，返回不正常行
    for dictItems in blackLineDict:
        if -1 != cliLine.find(dictItems.get("key_word")):
            if isChinese(lang):
                errMsg = dictItems.get("msg_zh")
            else:
                errMsg = dictItems.get("msg_en")
            return (False, errMsg)
    return (True, errMsg)

# **************************************************************************** #
# 函数名称: isCliExecRetInBlackList
# 功能说明: 传入待执行CLI命令，判断回显是否在黑名单中，带回结果供各个巡检项调用
#           1）黑名单回显，返回：True
#           2）其它情况，返回：False
# 输入参数: cli,cmd,lang,isHasLog（命令执行是否打印日志）
# 输出参数: True or False，同时带回异常回显对应的中英文提示信息
# **************************************************************************** # 
def isCliExecRetInBlackList(cli, cmd, lang, isHasLog):
    
    errMsg = ""
    
    #CLI命令执行是否打印日志
    if isHasLog:
        cliRet = cli.execCmd(cmd)
    else:
        cliRet = cli.execCmdNoLog(cmd)
        
    if None == cliRet or ""==cliRet:
        if isChinese(lang):
            errMsg =u"\nCLI回显结果为空"
        else:     
            errMsg = "\nThe display of cli result is empty"
        return (True, cliRet, errMsg)
    
    
    #异常，命令执行成功的关键字存在于前几行，防止全部判断时误判
    lineList = cliRet.splitlines()
    
    #行数大于5行，则回显正常，不在黑名单中
    if len(lineList)>KEY_WORD_VALID_LINES:
        return (False, cliRet, errMsg)
    
    #回显在白名单中直接返回，不带回key值
    for line in lineList[0:KEY_WORD_VALID_LINES]:
        checkRet = isLineInWhiteList(line)
        if checkRet[0]:
            return (False, cliRet, errMsg) 
    
    #是否在黑名单中，若是获取具体错误原因
    for line in lineList[0:KEY_WORD_VALID_LINES]:
        checkRet = isLineInBlackDict(line, lang)
        if checkRet[0]:
            errMsg = checkRet[1]
            return (True, cliRet, errMsg)  
    
    #不在黑名单中 
    return (False,cliRet,errMsg)

# **************************************************************************** #
# 函数名称: isCliExecRetInWriteList
# 功能说明: 传入待执行CLI命令，判断回显是否在白名单中，带回结果供各个巡检项调用
#           1）白名单回显，返回：True
#           2）其它情况，返回：False
# 输入参数: cli,cmd,lang,isHasLog（命令执行是否打印日志）
# 输出参数: True or False，同时带回捕捉到的白名单关键字信息
# **************************************************************************** # 
def isCliExecRetInWriteList(cli, cmd, lang, isHasLog):
    
    writeKeyWords = ""
    
    #CLI命令执行是否打印日志
    if isHasLog:
        cliRet = cli.execCmd(cmd)
    else:
        cliRet = cli.execCmdNoLog(cmd)
        
    if None == cliRet or ""==cliRet:
        return (True, cliRet, writeKeyWords)
    
    #异常，命令执行成功的关键字存在于前几行，防止全部判断时误判
    lineList = cliRet.splitlines()
    for line in lineList[0:KEY_WORD_VALID_LINES]:
               
        #是否在白名单中
        checkRet = isLineInWhiteList(line)
        if checkRet[0]:
            writeKeyWords = checkRet[1]
            return (True, cliRet, writeKeyWords)
    
    #不在白名单中
    return (False,cliRet,writeKeyWords)

# **************************************************************************** #
# 函数名称: cliExecRetJudgeWithDoubleList
# 功能说明: 传入待执行CLI命令，初步判断回显是否正常，带回结果供各个巡检项
#           1）黑名单回显，返回：CLI_RET_FAULT，巡检不通过
#           2）白名单回显，返回：CLI_RET_SUCCESS，巡检通过，注意巡检项中使用多个CLI命令的情况
#           2）其它情况，返回：CLI_RET_OK，需要根据后续的逻辑判断是否巡检通过
# 输入参数: cli,cmd,lang,isHasLog（命令执行是否打印日志）
# 输出参数: CLI_RET_FAULT，CLI_RET_SUCCESS or CLI_RET_OK
# **************************************************************************** # 
def cliExecRetJudgeWithDoubleList(cli, cmd, lang, isHasLog):
    
    errMsg = ""
    
    #CLI命令执行是否打印日志 
    if isHasLog:
        cliRet = cli.execCmd(cmd)
    else:
        cliRet = cli.execCmdNoLog(cmd)
        
    if None == cliRet or ""==cliRet:
        if isChinese(lang):
            errMsg =u"\nCLI回显结果为空"
        else:     
            errMsg = "\nThe display of cli result is empty"
        return (CLI_RET_FAULT, cliRet, errMsg)

    #异常，命令执行成功的关键字存在于前几行，防止全部判断时误判
    lineList = cliRet.splitlines()
    for line in lineList[0:KEY_WORD_VALID_LINES]:
         
        #在白名单中，巡检可以直接通过
        checkRet = isLineInWhiteList(line)
        if checkRet[0]:
            whiteKeyWord = checkRet[1]
            return (CLI_RET_SUCCESS, cliRet,whiteKeyWord)
        
    for line in lineList[0:KEY_WORD_VALID_LINES]:
         
        #在黑名单中，巡检直接不通过
        checkRet = isLineInBlackDict(line, lang)
        if checkRet[0]:
            errMsg = checkRet[1]
            return (CLI_RET_FAULT, cliRet, errMsg) 
    
    #其它情况，说明需要通过回显来判断此项巡检是否通过   
    return (CLI_RET_OK,cliRet,errMsg)


# **************************************************************************** #
# 函数名称: cliExecRetWithTimeout
# 功能说明: 传入待执行CLI命令，初步判断回显是否正常，带回结果供各个巡检项，本函数带超时处理机制
#           1）黑名单回显，返回：CLI_RET_FAULT，巡检不通过
#           2）白名单回显，返回：CLI_RET_SUCCESS，巡检通过，注意巡检项中使用多个CLI命令的情况
#           2）其它情况，返回：CLI_RET_OK，需要根据后续的逻辑判断是否巡检通过
# 输入参数: cli,cmd,lang,isHasLog（命令执行是否打印日志）
# 输出参数: CLI_RET_FAULT，CLI_RET_SUCCESS or CLI_RET_OK
# **************************************************************************** # 

def cliExecRetWithTimeout(cli, cmd, timeOut, lang):
    
    #执行带超时设置的CLI命令
    cliRet = cli.execCmdWithTimout(cmd,timeOut)
    
    errMsg = ""
    if None == cliRet or ""==cliRet:
        if isChinese(lang):
            errMsg =u"\nCLI回显结果为空"
        else:     
            errMsg = "\nThe display of cli result is empty"
        return (CLI_RET_FAULT, cliRet, errMsg)
    
    #异常，命令执行成功的关键字存在于前几行，防止全部判断时误判
    lineList = cliRet.splitlines()
    for line in lineList[0:KEY_WORD_VALID_LINES]:
         
        #在黑名单中，巡检直接不通过
        checkRet = isLineInBlackDict(line, lang)
        if checkRet[0]:
            errMsg = checkRet[1]
            return (CLI_RET_FAULT, cliRet, errMsg) 
        
        #在白名单中，巡检可以直接通过
        checkRet = isLineInWhiteList(line)
        if checkRet[0]:
            whiteKeyWord = checkRet[1]
            return (CLI_RET_SUCCESS, cliRet,whiteKeyWord)
    
    #其它情况，说明需要通过回显来判断此项巡检是否通过   
    return (CLI_RET_OK,cliRet,errMsg)


#####################################################
# 函数名称：getStdBBUNum
# 功能说明：获取标配BBU个数
# 传入参数：cli回显信息
# 传出参数：标配BBU个数
#####################################################
def getStdBBUNum(cliRet):
    ctrlNum = 0
    bbuNumDict = {}
    list = cliRet.splitlines()

    #获取控制器个数
    for field in list:
        field = field.replace(" ","")
        if field.startswith("Controller:"):
            ctrlNum = ctrlNum + 1
            continue
    
    #计算控制器BBU标准配置个数：每个控制器标配2个BBU
    standardBbuNum = ctrlNum * 2
    return standardBbuNum

#####################################################
# 函数名称：checkBbuNum
# 功能说明：检测BBU个数是否达标
# 传入参数：cli连接，语言lang，错误信息errMsg，BBU状态回文cliRet4Bbu
# 返 回  值：通过标识flag，错误信息errMsg，回显结果cliRet
#####################################################
def checkBbuNum(cli, lang, errMsg, cliRet4Bbu):
    flag = True
    
    #CLI命令执行后，通过接口初步判断命令是否执行成功，异常情况直接返回，巡检不通过，相关详细描述信息在公共接口里面设置
    cmd = "show controller general"
    checkRet = isCliExecRetInBlackList(cli, cmd, lang, True)
    cliRet4CtrlInfo = checkRet[1]
    errMsg4CtrlInfo = checkRet[2]
    if checkRet[0]:
        return (False, cliRet4CtrlInfo, errMsg4CtrlInfo)
    
    #根据产品型号获取标配BBU个数
    standardBbuNum = getStdBBUNum(cliRet4CtrlInfo)
    
    #未查到标配BBU个数
    if 0 == standardBbuNum:
        flag = False
        if "zh" == lang:
            errMsg += u"\n未查到BBU标配个数。"
        else:
            errMsg += "\nThe standard configuration number of BBU cannot be found."
        return (flag, cliRet4CtrlInfo, errMsg)
    
    #获取真实BBU个数
    cliRet4BbuList = cliRet4Bbu.splitlines()
    count4Bbu = 0
    isCounterSwitchOn = False
    for line in cliRet4BbuList:
        #如果找到列分隔符"---"，就打开计数开关，并在下一行开始计数
        if re.search("---", line, re.IGNORECASE):
            isCounterSwitchOn = True
            continue
        #如果找到“:/>”，就关闭计数开关，并退出循环
        if re.search(":/>", line, re.IGNORECASE):
            isCounterSwitchOn = False
            break
        #如果是空行，则不计数
        if "" == line.strip():
            continue
        #如果计数开关打开，则开始计数
        if isCounterSwitchOn:
            count4Bbu += 1
    
    #比较真实BBU个数和标配BBU个数是否一致
    if count4Bbu < standardBbuNum:
        flag = False
        if "zh" == lang:
            errMsg += u"\nBBU个数不足。"
        else:
            errMsg += "\nThe number of BBU is not enough."
    return (flag, cliRet4CtrlInfo, errMsg)
