# -*- coding: UTF-8 -*-
from com.huawei.ism.tlv import TLVUtils
from com.huawei.ism.tlv.bean import Param
from com.huawei.ism.tlv.docoder import ParamType
from com.huawei.ism.tlv.lang import UnsignedInt32
from com.huawei.ism.tlv.lang import UnsignedInt64
from com.huawei.ism.exception import IsmException
from frame.tlv import tlvDataType
from frame.tlv import tlvData
from frame.common.regex import Regex

RETRY_TIMES = 5 #建立连接时重试次数
TLV_CMD_TIMEOUT = 120
CMD_DEFAULT_TIMEOUT = 60*5 #执行TLV命令默认超时时间
TLV_RECORD_END_INDEX = 4095
ERR_CODE_SLAVE_NODE_HANDLE_MSG_IGNORE = "67108849"
    
def getParamValue(paramType, paramValue):
    '''
    @summary: 将参数转换为tlv参数
    '''
    if paramType == tlvDataType.DATA_TYPE_UNSIGN_INT or \
       paramType == tlvDataType.DATA_TYPE_SIGN_INT or \
       paramType == tlvDataType.DATA_TYPE_BOOL or \
       paramType == tlvDataType.DATA_TYPE_ENUM or \
       paramType == tlvDataType.DATA_TYPE_TIME:
        return UnsignedInt32(paramValue)
    
    elif paramType == tlvDataType.DATA_TYPE_U64LONG_TYPE:
        return UnsignedInt64(paramValue)
    
    elif paramType == tlvDataType.DATA_TYPE_STRING or \
         paramType == tlvDataType.DATA_TYPE_ARRAY or \
         paramType == tlvDataType.DATA_TYPE_JSON:
        return unicode(paramValue)
    
    return paramValue

def getParamList(params):
    '''
    @summary: 获取tlv参数列表
    '''
    paramList = []
    if params is None or len(params) == 0:
        return paramList
    for paramDefine, paramValue in params:
        paramIndex = paramDefine["index"]
        paramType = paramDefine["type"]
        paramValue = getParamValue(paramType, paramValue)
        paramList.append(Param(paramIndex, paramType, paramValue))
    return TLVUtils.paramList(paramList)

def getRecordValue(record, field):
    '''
    @summary: 获取tlv记录中的值
    @param record:
    @param field:
    '''
    fieldIndex = field["index"]
    fieldType = field["type"]
    
    if fieldType == tlvDataType.DATA_TYPE_UNSIGN_INT or \
       fieldType == tlvDataType.DATA_TYPE_SIGN_INT or \
       fieldType == tlvDataType.DATA_TYPE_ENUM or \
       fieldType == tlvDataType.DATA_TYPE_TIME:
        val = record.getParamIntValue(fieldIndex)
        if val:
            return val.intValue()
        else:
            return 0
    
    elif fieldType == tlvDataType.DATA_TYPE_U64LONG_TYPE:
        val = record.getParamIntValue(fieldIndex)
        if val:
            return val.longValue()
        else:
            return 0
        
    elif fieldType == tlvDataType.DATA_TYPE_STRING or \
         fieldType == tlvDataType.DATA_TYPE_ARRAY or \
         fieldType == tlvDataType.DATA_TYPE_BOOL or \
         fieldType == tlvDataType.DATA_TYPE_JSON:
        val = record.getParamStrValue(fieldIndex)
        if val:
            return unicode(val)
        else:
            return ""
        
    return record.getParamValue(fieldIndex)

def execCmd(tlv, cmd, params, isBatch=False, getBatch=False, timeOut=TLV_CMD_TIMEOUT):
    '''
    @summary: 执行tlv命令
    @param tlv: tlv对象
    @param cmd: 命令字
    @param params: 参数列表
    @param isBatch: 是否为批量处理（参考GET_BATCH_NEXT的实现的批量处理，批处理最后一条记录有结束符标记）
    @param getBatch: 是否为获取批量内容（参考非GET_BATCH_NEXT的实现的批量处理，批处理最后一条记录没有结束符标记）
    @param timeOut: 超时时间
    '''
    try:
        records = None
        paramList = getParamList(params)
        
        if isBatch:
            records = tlv.bacthNext(cmd, paramList, timeOut, RETRY_TIMES)
            if records is not None and len(records) > 1:
                lastRecord = records[-1]
                lastRecordParamList = lastRecord.getParams()
                if lastRecordParamList[0].getIndex() == TLV_RECORD_END_INDEX:
                    records.pop()
            return records
        
        if getBatch:
            return tlv.getBatch(cmd, paramList, timeOut, RETRY_TIMES)
        
        return tlv.invoke(cmd, paramList, timeOut, RETRY_TIMES)

    except IsmException, ismException:
        errCode = ismException.getErrorId()
        #往从节点下发消息失败，可以忽略。
        if ERR_CODE_SLAVE_NODE_HANDLE_MSG_IGNORE in unicode(errCode):
            return None
        
        errMsg = ismException.getErrorMessage()
        raise Exception(errCode, errMsg)
        return

def getHardwareRecords(tlv, hardwareType):
    '''
    @summary: 根据硬件形态获取该硬件的所有记录
    @param tlv: tlv对象
    @param hardwareType: 硬件形态
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E[hardwareType])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)

def getInterfaceModuleRecords(tlv):
    '''
    @summary: 获取当前集群所有接口卡的记录
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['INTF_MODULE'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)

def getEthPortRecords(tlv):
    '''
    @summary: 获取当前集群所有ETH端口的记录
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ETH_PORT'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)

def getMgmtPorts(tlv):
    '''
    @summary: 获取当前集群所有管理端口的记录
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ETH_PORT'])
    params.append(param)
    param = (tlvData.ETH_PORT['logicType'], tlvData.PORT_LOGIC_TYPE_E["MNGT"])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)

def getMntnPorts(tlv):
    '''
    @summary: 获取当前集群所有维护口的记录
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ETH_PORT'])
    params.append(param)
    param = (tlvData.ETH_PORT['logicType'], tlvData.PORT_LOGIC_TYPE_E["MAINTENANCE"])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)

def getSASPortsById(tlv, portId):
    '''
    @summary: 根据端口ID获取SAS端口信息
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['SAS_PORT'])
    params.append(param)
    param = (tlvData.PUB_ATTR['id'], portId)
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET"], params, isBatch=False)

def getEthPort(tlv, portId):
    '''
    @summary: 根据特定端口ID获取该端口的信息
    @param tlv: tlv对象
    @param portId: ETH端口ID
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ETH_PORT'])
    params.append(param)
    param = (tlvData.PUB_ATTR['id'], portId)
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET"], params, isBatch=False)

def setEthPortLinkStatus(tlv, portId, enumStatus):
    '''
    @summary: 设置特定端口link down
    @param tlv: tlv对象
    @param portId: ETH端口ID
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ETH_PORT'])
    params.append(param)
    param = (tlvData.PUB_ATTR['id'], portId)
    params.append(param)
    param = (tlvData.PUB_ATTR['runningStatus'], enumStatus)
    params.append(param)
    return execCmd(tlv, tlvData.CMD["MODIFY"], params, isBatch=False)

def getPCIePortRecords(tlv):
    '''
    @summary: 获取当前集群所有PCIe端口的记录
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['PCIE_PORT'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)

def getPciePortLocation(tlv, portId):
    '''
    @summary: 根据特定端口ID获取该端口的Location
    @param tlv: tlv对象
    @param portId: PCIe端口ID
    '''
    try:
        params = []
        param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['PCIE_PORT'])
        params.append(param)
        param = (tlvData.PUB_ATTR['id'], portId)
        params.append(param)
        record = execCmd(tlv, tlvData.CMD["GET"], params, isBatch=False)
        return getRecordValue(record, tlvData.PUB_ATTR["location"])
    except:
        return ""

def getSASPortRecords(tlv):
    '''
    @summary: 获取当前集群所有SAS端口的记录
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['SAS_PORT'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)

def getEnclosureRecords(tlv):
    '''
    @summary: 获取当前集群所有框信息的记录
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ENCLOSURE'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)

def getEnclosureInfo(tlv):
    '''
    @summary: 获取当前集群所有框信息的记录
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ENCLOSURE'])
    params.append(param)
    records = execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)
    
    enclosureInfoList =[]
    for record in records:
        infoDict = {}
        infoDict["id"] = getRecordValue(record, tlvData.PUB_ATTR["id"])
        infoDict["name"] = getRecordValue(record, tlvData.PUB_ATTR["name"])
        infoDict["location"] = getRecordValue(record, tlvData.PUB_ATTR["location"])
        enclosureInfoList.append(infoDict.copy())
    return enclosureInfoList

def getEnclosureRecord(tlv, enclosureID):
    '''
    @summary: 获取当前集群特定框信息的记录
    @param tlv: tlv对象
    @param enclosureID: 框ID
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['ENCLOSURE'])
    params.append(param)
    param = (tlvData.PUB_ATTR['id'], enclosureID)
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET"], params, isBatch=False)

def getCtrlEnclosureHeight(tlv):
    '''
    @summary: 获取当前集群框高
    @param tlv: tlv对象
    '''
    records = getEnclosureRecords(tlv)
    for record in records:
        logicType = getRecordValue(record, tlvData.ENCLOSURE['logicType'])
        if logicType == tlvData.ENCLOSURE_TYPE_E["CTRL"]:
            return getRecordValue(record, tlvData.ENCLOSURE['height'])
    return None

def getEngineNameList(tlv):
    '''
    @summary: 获取引擎
    @param tlv: tlv对象
    '''
    engineList = []
    records = getEnclosureRecords(tlv)
    for record in records:
        name = getRecordValue(record, tlvData.PUB_ATTR["name"])
        if "CTE" in name:
            engineList.append(name)
    return engineList
                
def getAllEnclosureSNs(tlv):
    '''
    @summary: 获取当前集群所有框SN集合
    @param tlv: tlv对象
    '''
    records = getEnclosureRecords(tlv)
    enclosureSNList = []
    for record in records:
        enclosureSNList.append(getRecordValue(record, tlvData.ENCLOSURE["serialNum"]))
    return enclosureSNList

def getEthPortsParentIDList(tlv):
    '''
    @summary: 获取当前集群正常的ETH端口对应的控制器ID列表
    @param tlv: tlv对象
    '''
    records = getEthPortRecords(tlv)
    ethPortsParentIDList = []
    for record in records:
        ethPortsID = getRecordValue(record, tlvData.PUB_ATTR['parentID'])
        ethPortsParentIDList.append(ethPortsID)
    return ethPortsParentIDList

def getControllerRecords(tlv):
    '''
    @summary: 获取当前集群正常的ETH端口对应的控制器ID列表
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['CONTROLLER'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)

def getEnclosureSNs(tlv):
    '''
    @summary: 获取当前集群正常的ETH端口对应的控制器ID列表
    @param tlv: tlv对象
    '''
    records = getControllerRecords(tlv)
    enclosureIdSet = set()
    for record in records:
        enclosureIdSet.add(getRecordValue(record, tlvData.PUB_ATTR['parentID']))
    currentEnclosureSNs = []
    for enclosureId in enclosureIdSet:
        record = getEnclosureRecord(tlv, enclosureId)
        currentEnclosureSNs.append(getRecordValue(record, tlvData.ENCLOSURE['serialNum']))
    return currentEnclosureSNs

def getControllersList(tlv):
    '''
    @summary: 获取当前集群所有的控制器ID列表
    @param tlv: tlv对象
    '''
    records = getControllerRecords(tlv)
    controllersList = []
    for record in records:
        controllerID = getRecordValue(record, tlvData.PUB_ATTR['id'])
        controllersList.append(controllerID)
    controllersList.sort()
    return controllersList

def getControllersNum(tlv):
    '''
    @summary: 获取当前集群的控制器数量
    @param tlv: tlv对象
    '''
    return len(getControllerRecords(tlv))

def getBayRecords(tlv):
    '''
    @summary: 获取当前集群的所有柜信息
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['BAY'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_BATCH_NEXT"], params, isBatch=True)

def getBayIds(bayRecords):
    '''
    @summary: 获取当前集群的所有柜ID列表
    @param tlv: tlv对象
    '''
    bayIds = []
    for record in bayRecords:
        bayId = getRecordValue(record, tlvData.BAY['id'])
        match = Regex.search("\d+$", bayId)
        if match:
            bayIds.append(match.group())
    return bayIds

def getExpansionBayIdRecord(tlv, bayId, isIpScaleOut):
    '''
    @summary: 获取当前集群的特定柜ID的记录
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['BAY'])
    params.append(param)
    param = (tlvData.PUB_ATTR['id'], bayId)
    params.append(param)
    if isIpScaleOut:
        param = (tlvData.BAY['isIpScaleOut'], True)
        params.append(param)
    return execCmd(tlv, tlvData.CMD["GET_EXPANSION_BAY_ID"], params)

def getBayConfigCtrlNum(tlv, bayId, isIpScaleOut=False):
    '''
    @summary: 获取当前集群的特定柜节点的控制器数量
    @param tlv: tlv对象
    '''
    record = getExpansionBayIdRecord(tlv, bayId, isIpScaleOut)
    return getRecordValue(record, tlvData.BAY['contrlTotal'])
    
def getBayConfigClustType(bayConfigRecord):
    '''
    @summary: 获取当前集群组网方式
    @param tlv: tlv对象
    '''
    bayConfigClustType = getRecordValue(bayConfigRecord, tlvData.IP_CLUST_BASE_INFO['netMode'])
    return bayConfigClustType

def getBayConfigBaseIpAddr(bayConfigRecord):
    '''
    @summary: 获取当前集群基地址IP
    @param tlv: tlv对象
    '''
    bayConfigBaseIpAddr = getRecordValue(bayConfigRecord, tlvData.IP_CLUST_BASE_INFO['baseIpAddr'])
    return bayConfigBaseIpAddr

def readXnetInfo(tlv):
    '''
    @summary: 获取Xnet信息
    @param tlv: tlv对象
    '''
    paramList = []
    return execCmd(tlv, tlvData.CMD["READ_IP_CLUST_BASE_INFO"], paramList)

def getSystemRecord(tlv):
    '''
    @summary: 获取系统信息
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['SYSTEM'])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["GET"], params)

def getProductVersion(tlv):
    '''
    @summary: 获取产品版本
    @param tlv: tlv对象
    '''
    record = getSystemRecord(tlv)
    return getRecordValue(record, tlvData.SYSTEM['productVersion'])

def getPdtModel(productModelEnum):
    '''
    @summary: 获取产品类型
    @param tlv: tlv对象
    '''
    return dict([(tlvData.PRODUCT_MODE_E.get(key), key) for key in tlvData.PRODUCT_MODE_E]).get(productModelEnum)

def getProductModel(tlv):
    '''
    @summary: 获取产品类型
    @param tlv: tlv对象
    '''
    record = getSystemRecord(tlv)
    productModelEnum = getRecordValue(record, tlvData.SYSTEM['productMode'])
    return getPdtModel(productModelEnum)

def isIBCPasswordChanged(tlv):
    '''
    @summary: 检查IBC密码是否为默认密码
    @param tlv: tlv对象
    '''
    params = []
    cmd = tlvData.CMD["CHECK_IBC_PASSWORD"]
    record = execCmd(tlv, cmd , params)
    return getRecordValue(record, tlvData.CHECK_IBC_PASSWORD['passwordChanged'])

def isIBCKeyPairChanged(tlv):
    '''
    @summary: 检查IBC公私钥对是否为默认值
    @param tlv: tlv对象
    '''
    params = []
    cmd = tlvData.CMD["CHECK_IBC_PASSWORD"]
    record = execCmd(tlv, cmd , params)
    return getRecordValue(record, tlvData.CHECK_IBC_PASSWORD['keyPairChanged'])

def getCpuInfoRecords(tlv):
    '''
    @summary: 获取当前集群的控制器CPU信息集合（CPU利用率、剩余内存等）
    @param tlv: tlv对象
    '''
    
    params = []
    param0 = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXE_UPD_MSGTYPE'], 0)
    params.append(param0)
    param12 = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXC_DIAG_CMD'], "upd sysresourcecheck")
    params.append(param12)
    param13 = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXC_NODE_CFG'], 2) #所有节点上执行
    params.append(param13)
    return execCmd(tlv, tlvData.CMD["OM_MSG_OP_UPD_EXECUTE"], params, getBatch=True)

def getAllCpuUsage(tlv):
    '''
    @summary: 获取当前集群的控制器CPU利用率
    @param tlv: tlv对象
    '''
    cpuUsageList = []
    records = getCpuInfoRecords(tlv)
    for record in records:
        ctrlId = record.getParamStrValue(0)
        cpuInfo = record.getParamStrValue(1)
        cpuUsage = cpuInfo.split(",")[0]
        cpuUsageList.append({"ctrlId":ctrlId, "cpuUsage":cpuUsage}.copy())
    return cpuUsageList

def modifyBayID(tlv, bayID):
    '''
    @summary: 修改柜ID
    @param tlv: tlv对象
    '''
    bayID = "%s%s" % ("SMB", bayID)
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['BAY'])
    params.append(param)
    param = (tlvData.PUB_ATTR['id'], bayID)
    params.append(param)
    return execCmd(tlv, tlvData.CMD["MODIFY_BAY_CONFIG"], params)

def modifyBayIDCtrlNum(tlv, bayId, ctrlNum):
    '''
    @summary: 修改柜ID和控制器数量
    @param tlv: tlv对象
    '''
    bayId = "%s%s" % ("SMB", bayId)
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['BAY'])
    params.append(param)
    param = (tlvData.PUB_ATTR['id'], bayId)
    params.append(param)
    param = (tlvData.BAY["contrlTotal"], int(ctrlNum))
    params.append(param)
    return execCmd(tlv, tlvData.CMD["MODIFY_BAY_CONFIG"], params)

def modifyCtrlNum(tlv, newConfigCtrlNum):
    '''
    @summary: 修改控制器数量
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['BAY'])
    params.append(param)
    param = (tlvData.BAY["contrlTotal"], int(newConfigCtrlNum))
    params.append(param)
    return execCmd(tlv, tlvData.CMD["MODIFY_BAY_CONFIG"], params)

def changeBMC(tlv):
    '''
    @summary: 修改BMC（6U2C扩容至6U4C时，需将配置信息先写到BMC）
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.BMC_INFO_DEFINE["flag"], tlvData.BMC_CHANGE_MODE_E["change"])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["CHANGE_BMC_INFO"], params)

def deleteBMC(tlv):
    '''
    @summary: 清除BMC（6U2C扩容至6U4C时，配置信息先写到BMC后需清除）
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.BMC_INFO_DEFINE["flag"], tlvData.BMC_CHANGE_MODE_E["delete"])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["CHANGE_BMC_INFO"], params)

def syncBMC(tlv):
    '''
    @summary: 同步BMC（2U环境将配置信息直接写入到disk，3U和6U环境将配置信息同步到BMC）
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.BMC_INFO_DEFINE["flag"], tlvData.BMC_CHANGE_MODE_E["sync"])
    params.append(param)
    return execCmd(tlv, tlvData.CMD["CHANGE_BMC_INFO"], params)

def expandCtrlCapacity(tlv, isIpScaleOut=True):
    '''
    @summary: 执行扩容
    @param tlv: tlv对象
    '''
    params = []
    param = (tlvData.PUB_ATTR['type'], tlvData.OM_OBJ_E['BAY'])
    params.append(param)
    if isIpScaleOut:
        param = (tlvData.BAY['isIpScaleOut'], True)
        params.append(param)
    return execCmd(tlv, tlvData.CMD["CAPACITY_EXPAND"], params)

def setEnclosureSwitchOn(tlv):
    '''
    @summary: 打开框的定位灯
    @param tlv: tlv对象
    @return: 
        True: 设置成功
        Flase: 设置失败
    '''
    return setEnclosureSwitch(tlv, 1)

def setEnclosureSwitchOff(tlv):
    '''
    @summary: 关闭框的定位灯
    @param tlv: tlv对象
    @return: 
        True: 设置成功
        Flase: 设置失败
    '''
    return setEnclosureSwitch(tlv, 2)

def setCurrentEnclosureSwitch(tlv, enclosureId, switch):
    '''
    @summary: 设置当前框的定位灯状态
    @param tlv: tlv对象
    @param enclosureId: 框ID
    @param switch: 定位灯状态（ON：1，OFF：2）
    @return: 
        True: 设置成功
        Flase: 设置失败
    '''
    try:
        params = []
        param = (tlvData.PUB_ATTR["type"], tlvData.OM_OBJ_E["ENCLOSURE"])
        params.append(param)
        param = (tlvData.PUB_ATTR["id"], enclosureId)
        params.append(param)
        param = (tlvData.ENCLOSURE["switch"], switch)
        params.append(param)
        
        execCmd(tlv, tlvData.CMD["MODIFY"], params)
        
    except Exception, exception:
        return False
    
    return True

def setEnclosureSwitch(tlv, switch):
    '''
    @summary: 设置框的定位灯状态
    @param tlv: tlv对象
    @param switch: 定位灯状态（ON：1，OFF：2）
    @return: 
        True: 设置成功
        Flase: 设置失败
    '''
    records = getEnclosureRecords(tlv)
    for record in records:
        enclosureId = getRecordValue(record, tlvData.PUB_ATTR["id"])
        setSwitchRet = setCurrentEnclosureSwitch(tlv, enclosureId, switch)
        if not setSwitchRet:
            return False
    
    return True

def getUpgradeRecords(tlv, checkItem):
    '''
    @summary: 执行升级检查的tlv命令
    @param tlv: tlv对象
    @param checkItem:升级检查项索引
    '''
    params = []
    param = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXE_UPD_MSGTYPE'], tlvData.UPD_MSG_E['EXE_UPD_MSGTYPE'])
    params.append(param)
    param = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXE_UPD_ACTIVETYPE'], tlvData.UPD_MSG_E['EXE_UPD_ACTIVETYPE'])
    params.append(param)
    param = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXC_DIAG_CMD'], tlvData.UPD_CMD_DEFINE[checkItem]['cmd'])
    params.append(param)
    param = (tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE['CMO_EXC_NODE_CFG'], tlvData.UPD_CMD_DEFINE[checkItem]['id'])
    params.append(param)
    
    return execCmd(tlv, tlvData.CMD["OM_MSG_OP_UPD_EXECUTE"], params, getBatch=True, timeOut=CMD_DEFAULT_TIMEOUT)

def getTLVParams(paramDict):
    """@summary: 解析字典获得参数列表
       @param paramDict：包含参数信息的字典 
       @return: 参数排序后返回
    """
    tempDict = dict()
    for (type, value) in paramDict.items():
        paramInfo = tlvData.CMO_NOTIFY_EXC_UPGRADE_DEFINE[type]
        tempDict[paramInfo["index"]] = (paramInfo, value)
    indexList = tempDict.keys()
    indexList.sort()
    paramList = [tempDict[item] for item in indexList]
    return paramList

def getAllKindsTLVRecords(tlv, checkItem, logger):
    """
    @summary: 获取tlv消息的通用方法
    """
    paramDict = tlvData.UPD_CMD_DEFINE[checkItem]
    params = getTLVParams(paramDict)
    logger.logInfo("params=%s" % unicode(params))
    return execCmd(tlv, tlvData.UPD_SPECIAL_CMD[checkItem], params, getBatch=True, timeOut=CMD_DEFAULT_TIMEOUT)
    

