# -*- coding: UTF-8 -*-
from frame.cli import cliUtil
from frame.common import common
from frame.common import config
from frame.common.regex import Regex
from frame.context import contextUtil
from frame.tlv import tlvUtil
from frame.tlv import tlvData

INVOVLED_DEVICE_TYPE = ['5600 V3','5800 V3','6800 V3','6900 V3','18500 V3','18800 V3',
                        'S5600T','S5800T','S6800T','18500','18800','18800F','HVS85T','HVS88T',]
#onetrack代码，830之后共用
INVOVLED_VERSION = ['V100R001','V200R002','V300R001','V300R002','V300R003C00','V300R003C00SPC100','V300R003C10']

DEVICE_PORT_MAP = {
                   "CTEX.A0.P0": ["DAEX00.A.PRI","DAEX00.A.PRI0"],
                   "CTEX.R0.IOM0.P0": ["DAEX00.A.PRI","DAEX00.A.PRI0"],
                   "CTEX.R0.IOM1.P0": ["DAEX80.A.PRI","DAEX80.A.PRI0"],
                   "CTEX.R5.P0": ["DAEX00.A.PRI","DAEX00.A.PRI0"],
                   "ENGX.A0.P0": ["DAEX00.A.PRI","DAEX00.A.PRI0"],
                   
                   "CTEX.B0.P0": ["DAEX00.B.PRI","DAEX00.B.PRI0"],
                   "CTEX.L0.IOM0.P0": ["DAEX00.B.PRI","DAEX00.B.PRI0"],
                   "CTEX.L0.IOM1.P0": ["DAEX80.B.PRI","DAEX80.B.PRI0"],
                   "CTEX.L5.P0": ["DAEX00.B.PRI","DAEX00.B.PRI0"],
                   "ENGX.B0.P0": ["DAEX00.B.PRI","DAEX00.B.PRI0"],
                   }

DEVICE_DOUBLE_PORT_MAP = {
                   "CTEX.A0.P1": ["DAEX00.A.PRI1"],
                   "CTEX.R0.IOM0.P1": ["DAEX00.A.PRI1"],
                   "CTEX.R0.IOM1.P1": ["DAEX80.A.PRI1"],
                   "CTEX.R5.P1": ["DAEX00.A.PRI1"],
                   "ENGX.A0.P1": ["DAEX00.A.PRI1"],
                   
                   "CTEX.B0.P1": ["DAEX00.B.PRI1"],
                   "CTEX.L0.IOM0.P1": ["DAEX00.B.PRI1"],
                   "CTEX.L0.IOM1.P1": ["DAEX80.B.PRI1"],
                   "CTEX.L5.P1": ["DAEX00.B.PRI1"],
                   "ENGX.B0.P1": ["DAEX00.B.PRI1"],
                   }

CONF_DISK_LIST = ['DAEX00','DAEX80']

def execute(context):
    
    '''
    @summary：检查独立机头设备保险框环路组网，要求正正接
    ''' 
    lang = contextUtil.getLang(context)
    logger = common.getLogger(contextUtil.getLogger(context), __file__)
    tlv = contextUtil.getTlvObj(context)
    cliRet = ""
    errMsg = ""
    flag = True
    
    try:
        deviceType = contextUtil.getDevType(context)
        #只检查独立机头的设备
        if deviceType not in INVOVLED_DEVICE_TYPE:
            return common.getUpgEvaluationRs(cliUtil.RESULT_NOSUPPORT, cliRet, errMsg)
        
        #判断涉及版本
        devVersion = contextUtil.getCurVersion(context)
        VRVersion = devVersion[:8]
        if (devVersion not in INVOVLED_VERSION) and (VRVersion not in INVOVLED_VERSION):
            SPCVersion = common.getSpcVersion(devVersion)
            if SPCVersion >= 'V300R003C10SPC100':
                cli = contextUtil.getSSH(context)
                return checkConfCableAlarm(cli, logger, lang)
            return common.getUpgEvaluationRs(cliUtil.RESULT_NOSUPPORT, cliRet, errMsg)
        
        #取得引擎个数 和sas端口信息
        engineNum = getEngineNum(tlv, logger)
        SASPortDict = getSasPortsDict(tlv, logger)
        confDiskEncList = getHighConfDiskEncList(tlv, logger)
        logger.logInfo("get high disk enclosure: %s" % str(confDiskEncList))
        
        #多个引擎再判断版本，V3R1及之前的版本只检查0号引擎；其余都检查所有引擎
        if engineNum > 1 and VRVersion <= "V300R001":
            #多引擎，并且不能准确检查接线的版本
            logger.logInfo("Multiple %s Engine: Only Check Zero Engine!!" % engineNum)
            flag = cliUtil.RESULT_NOCHECK
            (checkFlag, cliRetList, errPortList) = checkEncloLink(SASPortDict, confDiskEncList, logger, True)
            if checkFlag:
                errMsg += common.getMsg(lang, "zero.engine.check.pass")
            else:
                errMsg += getErrMsg(errPortList, common, lang)
            errMsg += common.getMsg(lang, "multi.engine.not.check")
            
        else:
            logger.logInfo("Check All Engine!!")
            (flag, cliRetList, errPortList) = checkEncloLink(SASPortDict, confDiskEncList, logger)
            
            errMsg += getErrMsg(errPortList, common, lang)
       
        cliRet = '\n'.join(cliRetList)
        return common.getUpgEvaluationRs(flag, cliRet, errMsg)
    
    
    except Exception, exception:
        logger.logException(exception)
        return common.getUpgEvaluationRs(False, cliRet, common.getMsg(lang, "query.result.abnormal"))

def getHighConfDiskEncList(tlv, logger):
    '''
    @summary: 获取DAEX00,DAEX80硬盘框id,name,model
    @param:
        tlv: tlv连接
        logger: 日志打印
    @return confDiskEncList: id、name、model的字典列表
    '''
    encRecords = tlvUtil.getEnclosureRecords(tlv)
    logger.logInfo("enclosure Tlv Records:%s" % unicode(encRecords))
    
    confDiskEncList = []
    for record in encRecords:
        name = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["name"])
        model = tlvUtil.getRecordValue(record, tlvData.ENCLOSURE["model"])
        
        if cliUtil.isEngineId(name):
            continue
        #DAEX00,DAEX80
        xName = 'DAEX' + name[-2:]
        logger.logInfo("enclosure xname:%s" % unicode(xName))
        if str(model) in config.EXPSAS4U_75_LIST and xName in CONF_DISK_LIST:
            confDiskEncList.append(name)
        
    return confDiskEncList

def getEngineNum(tlv, logger):
    '''
    @summary: 获取组网中引擎框个数
    @param 
        tlv: tlv连接
        logger: 日志打印
    @return engineNum: 引擎框个数
    '''
    enclosureRecords = tlvUtil.getEnclosureRecords(tlv)
    logger.logInfo("Enclosure Tlv Records:%s" % unicode(enclosureRecords))
        
    engineNum = 0
    for record in enclosureRecords:
        name = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["name"])
        if cliUtil.isEngineId(name):
            engineNum += 1
    return engineNum

def getSasPortsDict (tlv, logger):
    '''
    @summary: 获取sas端口id、location、当前连接的对端端口id
    @param:
        tlv: tlv连接
        logger: 日志打印
    @return SASPortList: id、location、当前连接的对端端口id构成的字典列表
    '''
    SASPortRecords = tlvUtil.getSASPortRecords(tlv)
    logger.logInfo("SAS Port Tlv Records:%s" % unicode(SASPortRecords))
    
    SASPortDict = {}
    for record in SASPortRecords:
        Id = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["id"])
        location = tlvUtil.getRecordValue(record, tlvData.PUB_ATTR["location"])
        curentPeerPortId = tlvUtil.getRecordValue(record, tlvData.SAS_PORT["current_peer_port_id"])
        if not location:
            continue
        
        infoDict = {
                    "id":Id,
                    "location":location,
                    "curentPeerPortId":curentPeerPortId,
                    }
        SASPortDict[location] = infoDict
        
    return SASPortDict



def getPortLocationById(portId, SASPortDict):
    '''
    @summary: 根据端口id获取端口的location
    @param 
        portId: 端口id
        SASPortList: 所有sas端口的查询结果
    @return location: 端口位置信息
    '''
    location = "--"
    for saslocation in SASPortDict.keys():
        infoDict = SASPortDict[saslocation]
        Id = infoDict["id"]
        if portId == Id:
            return saslocation
    return location


def isPeerPortRight(curentPeerPortLocation, confEngPortList, xValue):
    '''
    @summary: 判断对端端口是否在配置的引擎端口范围内，从而判断端口是否正确连接
    @param 
        curentPeerPortLocation: 当前连接的对端端口位置
        confEngPortList: 配置的可连的引擎端口位置信息
        xValue: 当前引擎号（配置中X位置的值）
    @return: True 正确连接，False连接错误
    '''
    newList = []
    for enginePort in confEngPortList:
        newList.append(Regex.replaceAll('X', enginePort, xValue))

    if curentPeerPortLocation in newList:
        return True
    return False

def getXLocation(location):
    
    if location is None or len(location) < 5:
        return location
    
    return location[:3] + 'X' + location[4:]      

def getErrMsg(errPortList, common, lang):
    '''
    @summary: 获取错误消息
    @param errPortList: 错误消息的参数列表
    @return errMsg: 错误提示
    '''
    errMsg = ""
    for err in errPortList:
        errMsg += common.getMsg(lang, "engine.port.check.notpass", err)
    return errMsg

def getSugDAEsasPortsLoc(confDAELocations, engId):
    '''
    @summary: 获取对应引擎下对应的硬盘框连线端口
    '''
    suggestDAELocations = []
    
    engIdStr = str(engId) 
    for loc in confDAELocations:
        suggestDAELocations.append(Regex.replaceAll('X', loc, engIdStr))
        
    return suggestDAELocations

def getP1PortByP0(location):   
    return location[:-1] + "1"

def needCheckLinkForP1(location, highConfEncList):
    '''
    @summary: 是否需要检查P1端口
    '''
    enclosure = location.split('.')[0].strip()
    if enclosure in highConfEncList:
        return True
    return False

def checkEncloLink(SASPortDict, highConfEncList, logger, zeroEngFlag = False):
    '''
    @summary: 检查保险箱框连接正确性，高密框的PRI0口连接错误则检查PRI1端口
    '''
    flag = True
    cliRetList = []
    errPortList = []
    engPortList = DEVICE_PORT_MAP.keys()
    
    for location in SASPortDict.keys():
        infoDict = SASPortDict[location]
        if not cliUtil.isEngineId(location):
            continue
        
        xValue = location[3]
        if zeroEngFlag and xValue != '0':
            continue
        
        xLocation = getXLocation(location)
        if xLocation in engPortList:
            curentPeerPortId = infoDict["curentPeerPortId"]
            curentPeerPortLocation = getPortLocationById(curentPeerPortId, SASPortDict)
            cliRetList.append("port: %s, current peer port: %s." % (location, curentPeerPortLocation))
            
            confDAELocations = DEVICE_PORT_MAP.get(xLocation)
            suggestDAELocations = getSugDAEsasPortsLoc(confDAELocations, xValue)

            #检查P0端口是否连接正确
            if curentPeerPortLocation not in suggestDAELocations:
                logger.logInfo("Engine [%s]: Port %s is not correctly connected to the configuration disk enclosure (current connection on the peer device: %s). " % 
                               (xValue, location, curentPeerPortLocation))
                flag = False
                errPort = (xValue, location, curentPeerPortLocation)
                errPortList.append(errPort) 
            
            #如果为高密框，则检查P1端口
            needCheckFlag = needCheckLinkForP1(suggestDAELocations[0], highConfEncList)
            logger.logInfo("check P1 port flag: %s, location:%s, conf enclosure:%s"%(needCheckFlag, location, str(highConfEncList)))
            if needCheckFlag:
                p1location = getP1PortByP0(location)
                logger.logInfo("Engine [%s]: begin check port %s" %(xValue, p1location))
                checkFlag, errInfo, checkCliRetList = checkEncloLinkForP1(p1location, SASPortDict, highConfEncList, logger)
                cliRetList.extend(checkCliRetList)
                if checkFlag:
                    continue
                else:
                    flag = False
                    errPortList.append(errInfo)
                
    sortErrPortList = sorted(errPortList, key=lambda x:x[1])
    return (flag, cliRetList, sortErrPortList)

def checkEncloLinkForP1(location, SASPortDict, highConfEnclist, logger):
    '''
    @summary: 检查P1端口连接正确性
    '''
    cliRetList = []
    xLocation = getXLocation(location)
    engPortList = DEVICE_DOUBLE_PORT_MAP.keys()
    xValue = location[3]
    infoDict = SASPortDict[location]
    curentPeerPortId = infoDict["curentPeerPortId"]
    curentPeerPortLocation = getPortLocationById(curentPeerPortId, SASPortDict)
    cliRetList.append("port: %s, current peer port: %s." % (location, curentPeerPortLocation))
    
    if xLocation not in engPortList:
        return (False, (xValue, location, curentPeerPortLocation))
    
    confDAELocations = DEVICE_DOUBLE_PORT_MAP.get(xLocation)
    suggestDAELocations = getSugDAEsasPortsLoc(confDAELocations, xValue)
    if curentPeerPortLocation not in suggestDAELocations:
        logger.logInfo("Engine [%s]: Port %s is not correctly connected to the configuration disk enclosure (current connection on the peer device: %s). " % 
                       (xValue, location, curentPeerPortLocation))
        return (False, (xValue, location, curentPeerPortLocation),cliRetList)
    
    return (True, [],cliRetList)
            
def checkConfCableAlarm(cli,logger,lang):
    '''
    @summary: V3R3C10SPC100及以后的版本查询告警：0xF00CE0033
    '''    
    alarmId = "0xF00CE0033"
    cmd = "show alarm |filterRow column=ID predict=equal_to value=%s" %alarmId 
    (flag, cliRet, errMsg) = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if flag != True:
        return common.getUpgEvaluationRs(flag, cliRet, errMsg)
    
    infoList = cliUtil.getHorizontalCliRet(cliRet)
    if len(infoList) > 0:
        errMsg = common.getMsg(lang, "conf.enclosure.cable.alarm.error", alarmId)
        return common.getUpgEvaluationRs(False, cliRet, errMsg)
    return common.getUpgEvaluationRs(True, cliRet, '')
    
