# -*- coding: UTF-8 -*-
from frame.cli.cliFactory import CliConnection
from frame.tlv.tlvFactory import TlvConnection
from frame.tlv.tlvFactory import TlvConnectionSVP
from frame.rest.restFactory import RestConnection
from frame.rest import restData
from frame.common import jsonUtil
from frame.cli import cliUtil
from frame.common import common
from frame.base import config, baseUtil
from cbb.frame.util.common import wrapAllExceptionLogged
import cbb.frame.base.config as CBB_CONFIG
import os
import traceback
import time

SCRIPT_DEF_CONTEXT = "SCRIPT_DEF_CONTEXT"
RESOURCE_DEF_CONTEXT = "RESOURCE_DEF_CONTEXT"

CLI_CONNECTION = "CLI_CONNECTION"
TLV_CONNECTION = "TLV_CONNECTION"
SVP_CONNECTION = "SVP_CONNECTION"
REST_CONNECTION = "REST_CONNECTION"

ERROR_CODE_DEFINE = {
    "16797697":
        {
            "errMsg_zh":u"通信异常或系统繁忙。",
            "errMsg_en":"The communication is abnormal or the system is busy.",
            "suggestion_zh":u"",
            "suggestion_en":"",
        },
    "1073949185":
        {
            "errMsg_zh":u"与设备通信异常。",
            "errMsg_en":"Communicating with the device failed.",
            "suggestion_zh":u"请检查网络连接或设备状态是否正常。如果有任何疑问，请联系技术支持工程师协助处理。",
            "suggestion_en":"Please check that the network connection or the system is normal. If you have any problems, please contact technical support engineers for help.",
        },
    "TLV_CAN_NOT_CONNECT_CODE_SVP":
        {
            "errMsg_zh":u"与SVP通信异常。",
            "errMsg_en":"Communicating with the SVP failed.",
            "suggestion_zh":u"请检查SVP状态是否正常。如果有任何疑问，请联系技术支持工程师协助处理。",
            "suggestion_en":"Please check that the status of SVP is normal. If you have any problems, please contact technical support engineers for help.",
        },
    "1077936891":
        {
            "errMsg_zh":u"命令执行失败。",
            "errMsg_en":"Command execute failed.",
            "suggestion_zh":u"",
            "suggestion_en":"",
        },
    "1077949083":
        {
            "errMsg_zh":u"引擎无法监控。",
            "errMsg_en":"The engine can not be monitored.",
            "suggestion_zh":u"请查看系统中告警并按照告警修复建议进行处理。",
            "suggestion_en":"Please rectify the faults by referring to recommended actions.",
        },
    "1077949058":
        {
            "errMsg_zh":u"用户权限不足。",
            "errMsg_en":"The user does not have the permission.",
            "suggestion_zh":u"请联系管理员。",
            "suggestion_en": "Contact the administrator.",
        },                                    
}
        
def getLang(context):
    '''
    @summary: 获取语言
    @param context: 上下文对象
    '''
    return context.get("lang")

def getLogger(context):
    '''
    @summary: 获取日志打印对象
    @param context: 上下文对象
    '''
    return context.get("logger")

def getSSH(context):
    '''
    @summary: 获取工具与阵列的SSH连接对象
    '''
    return getCLI(context)

def getCLI(context):
    '''
    @summary: 获取工具与阵列的SSH连接对象,并判断是否可用,不可用进行重连
    '''
    cli = context.get('ssh')
    cmd = "show system general"
    for _ in range(3):
        
        try:
            cliRet = cli.execCmd(cmd)
            if "Health Status" in cliRet:
                return cli
            raise  
        except:
                try:
                    time.sleep(5)
                    cli.close()
                    cli.reConnect()
                    return cli
                except:
                    continue
    return cli

def getSFTP(context):
    '''
    @summary: 获取工具与阵列的SFTP连接对象
    '''
    return context.get('sftp')

def getDevType(context):
    '''
    @summary: 获取设备类型
    '''
    return getDevObj(context).getDeviceModel()

def getDevSN(context):
    '''
    @summary: 获取设备序列号Serial Number
    '''
    return getDevObj(context).getDeviceSerialNumber()

def getCurVersion(context):
    '''
    @summary: 获取系统当前版本
    @param context: 上下文对象
    '''
    return getDevObj(context).getProductVersion()

def getTargetVersion(context):
    '''
    @summary: 获取目标版本号
    @param context: 上下文对象
    '''
    return getDevObj(context).getDestVersion()

def getScriptDefDict(context):
    '''
    @summary: 获取上下文自定义字典
    @param context: 上下文对象
    '''
    if isinstance(context, dict):
        if not context.has_key(SCRIPT_DEF_CONTEXT):
            context.setdefault(SCRIPT_DEF_CONTEXT, {})
    else:
        if not context.containsKey(SCRIPT_DEF_CONTEXT):
            context.put(SCRIPT_DEF_CONTEXT, {})
    return context.get(SCRIPT_DEF_CONTEXT)

def setItem(context, key, value):
    '''
    @summary: 往自定义字典中添加项
    @param context: 上下文对象
    @param key: 添加项的键
    @param value: 添加项的值
    '''
    scriptDefDict = getScriptDefDict(context)
    scriptDefDict[key] = value
    
    if isinstance(context, dict):
        context[SCRIPT_DEF_CONTEXT] = scriptDefDict.copy()
    else:
        context.put(SCRIPT_DEF_CONTEXT, scriptDefDict.copy())
    return

def removeItem(context, key):
    '''
    @summary: 删除自定义字典中的项
    @param context: 上下文对象
    @param key: 删除项的键
    '''
    scriptDefDict = getScriptDefDict(context)
    if scriptDefDict.has_key(key):
        scriptDefDict.pop(key)
    return

def getItem(context, key):
    '''
    @summary: 获取自定义字典中的项
    @param context: 上下文对象
    @param key: 项的键
    '''
    scriptDefDict = getScriptDefDict(context)
    return scriptDefDict.get(key, None)

def getConnectorFactoryObj(context):
    '''
    @summary: 获取上下文对象中创建连接的对象
    @param context: 上下文对象
    '''
    return context.get("sshFactory")

def getNewDevObj(context):
    '''
    @summary: 获取上下文对象中dev_rest对象
    @param context: 上下文对象
    '''
    return context.get("newDev")

def getDevObj(context):
    '''
    @summary: 获取上下文对象中dev对象
    @param context: 上下文对象
    '''
    return context.get("dev")

def getLoginUserName(context):
    '''
    @summary: 获取登录用户名
    '''
    return getDevObj(context).getLoginUser().getUserName() 

def getDevLoginUserPwd(context):
    '''
    @summary: 获取上下文对象中dev对象中的的登录密码
    @param context: 上下文对象
    '''
    dev = getDevObj(context)
    return dev.getLoginUser().getPassword()

def getDeveloperPwd(context):
    '''
    @summary: 获取developer密码
    @param context: 上下文对象
    '''
    return getDevObj(context).getDeveloperPwd()

def getDevPort(context):
    '''
    @summary: 获取上下文对象中dev对象中的的登录端口
    @param context: 上下文对象
    '''
    dev = getDevObj(context)
    return dev.getPort()

def getTmpDir(context):
    '''
    @summary: 获取子工具运行根目录下的临时目录
    '''
    return context.get('tmpDir')

def getTlvObj(context):
    '''
    @summary: 获取上下文对象中tlv对象
    @param context: 上下文对象
    '''
    return getRest(context)

def getUpgradeModel(context):
    '''
    @summary: 获取升级模式（在线或者离线），其值为"ONLINE"或"OFFLINE"
    @param context: 上下文对象
    '''
    return str(getDevObj(context).getUpgradeMode())

def initCliConnection(context):
    '''
    @summary: 初始化cli连接
    @param context: 上下文对象
    '''
    connectorFactoryObj = getConnectorFactoryObj(context)
    devObj = getDevObj(context)
    
    ip = devObj.get("ip")
    sshPort = int(devObj.get("sshPort"))
    
    conn = CliConnection(connectorFactoryObj, ip, sshPort)
    conn.create(devObj.get("user"), devObj.get("pawd"))
    
    setItem(context, CLI_CONNECTION, conn)
    return

def getCli(context):
    '''
    @summary: 获取cli连接
    @param context: 上下文对象
    '''
    conn = getItem(context, CLI_CONNECTION)
    if conn is None:
        initCliConnection(context)
        conn = getItem(context, CLI_CONNECTION)
        
    devObj = getDevObj(context)
    cli = conn.getCli(devObj.get("user"), devObj.get("pawd"))
    return cli


@wrapAllExceptionLogged(logger=None)
def destroyCliConnection(context):
    '''
    @summary: 销毁cli连接
    @param context: 上下文对象
    '''
    conn = getItem(context, CLI_CONNECTION)
    if conn is not None:
        conn.close()
        removeItem(context, CLI_CONNECTION)


@wrapAllExceptionLogged(logger=None)
def destroyCliConn(cliConn):
    '''
    @summary: 销毁cli连接
    @param: cliConn:cli连接
    '''
    cliConn.close()


@wrapAllExceptionLogged(logger=None)
def closeCliConnection(cliCon):
    """关闭CLI连接。

    :param cliCon:
    :return:
    """
    if not cliCon:
        cliCon.close()

def initTlvConnection(context):
    '''
    @summary: 初始化tlv连接
    @param context: 上下文对象
    '''
    connectorFactoryObj = getConnectorFactoryObj(context)
    devObj = getDevObj(context)
    
    ip = devObj.get("ip")
    tlvPort = devObj.get("tlvPort")
    
    conn = TlvConnection(connectorFactoryObj, ip, tlvPort, innerIps = getItem(context, "innerIps"))
    conn.create(devObj.get("user"), devObj.get("pawd"))
    
    setItem(context, TLV_CONNECTION, conn)
    return

def getTlv(context):
    '''
    @summary: 获取tlv连接
    @param context: 上下文对象
    '''
    return getRest(context)


@wrapAllExceptionLogged(logger=None)
def destroyTlvConnection(context):
    '''
    @summary: 销毁tlv连接
    @param context: 上下文对象
    '''
    conn = getItem(context, TLV_CONNECTION)
    if conn is not None:
        conn.close()
        removeItem(context, TLV_CONNECTION)

def initTlvConnectionSVP(context):
    '''
    @summary: 初始化SVP的tlv连接
    @param context: 上下文对象
    '''
    connectorFactoryObj = getConnectorFactoryObj(context)
    devObj = getDevObj(context)
    
    conn = TlvConnectionSVP(connectorFactoryObj, deviceType=devObj.get("type"), sn=devObj.get("sn"))
    conn.create()
    
    setItem(context, SVP_CONNECTION, conn)
    return

def getTlvSVP(context):
    '''
    @summary: 获取SVP的tlv连接
    @param context: 上下文对象
    '''
    conn = getItem(context, SVP_CONNECTION)
    if conn is None:
        initTlvConnectionSVP(context)
        conn = getItem(context, SVP_CONNECTION)
    
    tlv = conn.getTlv()
    return tlv


@wrapAllExceptionLogged(logger=None)
def destroyTlvConnectionSVP(context):
    '''
    @summary: 销毁SVP的tlv连接
    @param context: 上下文对象
    '''
    conn = getItem(context, SVP_CONNECTION)
    if conn is not None:
        conn.close()
        removeItem(context, SVP_CONNECTION)

def getCurCtrlIp(context):
    '''
    @summary: 获取设备当前控制器IP
    @param context: 上下文对象
    '''
    return getDevObj(context).getIp()

def getImportRootDir(context):
    '''
    @summary: 获取脚本的根目录
    @param context: 上下文对象
    '''
    return os.path.dirname(context.get("importRootDir"))

def getResourceDict(context):
    '''
    @summary: 将property资源文件转换为键值对字典
    @param context: 上下文对象
    '''
    scriptDefDict = getScriptDefDict(context)
    if not scriptDefDict.has_key(RESOURCE_DEF_CONTEXT):
        resourceFilePath = getResourecePath(context)
        resourceFile = open(resourceFilePath)
        contents = resourceFile.read().splitlines()
        resourceDict = {}
        for line in contents:
            fields = line.split("=")
            if len(fields) < 2:
                continue
            key = fields[0].strip()
            value = "=".join(fields[1:]).strip()
            resourceDict.setdefault(key, value)
        scriptDefDict.setdefault(RESOURCE_DEF_CONTEXT, resourceDict)
        
    return scriptDefDict.get(RESOURCE_DEF_CONTEXT)

def getResource(context, key):
    '''
    @summary: 根据property资源文件中的键，获取对应的值
    @param context: 上下文对象
    @param key: property资源文件中的键
    '''
    resourceDict = getResourceDict(context)
    return resourceDict.get(key, "--")

def getResourecePath(context):
    '''
    @summary: 返回property文件所在的路径
    @param context: 上下文对象
    '''
    lang = getLang(context)
    return os.path.join(getImportRootDir(context), "res", "language", "common_%s.properties" % lang)

def createCliConnection(context, ip):
    '''
    @summary: 根据ip创建cli连接，支持跳转
    @param context: 上下文对象
    @param ip:创建cli连接的ip
    @return: 创建成功返回cli连接，否则为None 
    '''
    logger = getLogger(context)
    dev = getDevObj(context)
    
    #先保存原有ip，SN号，使用新ip建立连接后进行恢复
    originalIp = dev.getIp()    
    originalSN = dev.getDeviceSerialNumber()
    
    try:
        sn = originalSN + originalIp + ip    #生成临时SN用于建立cli连接，一个SN号建立一个连接
        
        dev.setIp(ip)
        dev.setDeviceSerialNumber(sn)
        connectorFactoryObj = getConnectorFactoryObj(context)
        sshPort = int(getDevPort(context))
        conn = CliConnection(connectorFactoryObj, ip, sshPort)
        cliConn = conn.createForwardConnetor(dev)
        return cliConn
    except:
        logger.info(unicode(traceback.format_exc()))
        return None
    finally:
        #恢复为原有ip，SN号
        dev.setIp(originalIp)
        dev.setDeviceSerialNumber(originalSN)

def getCollectTmpDir(context):
    '''
    @summary: 获取当前设备存放文件临时目录
    '''
    tmpDir = getTmpDir(context)
    tmpFileName = getItem(context, common.TMP_FILE_NAME)
    if not tmpFileName:
        return ""
    TmpDir = os.path.join(tmpDir, tmpFileName)
    return TmpDir           
 

def createRest(context, newIp):
    '''
    @summary: 使用ip创建rest连接
    @param context: 上下文对象
    @param newIp: 创建连接的ip
    '''
    try:
        connectorFactoryObj = getConnectorFactoryObj(context)
        devObj = getNewDevObj(context)
        
        devSN = devObj.get("sn")
        restConn = RestConnection(connectorFactoryObj, newIp, devSN)
        result = restConn.create(devObj.get("user"), devObj.get("pawd"), restData.REST_SCOPE_DEFAULT)
        return (result, restConn)
    except:
        return (False, None)

def initRestConnection(context):
    '''
    @summary: 初始化rest连接
    @param context: 上下文对象
    '''
    connectorFactoryObj = getConnectorFactoryObj(context)
    devObj = getNewDevObj(context)
    
    ip = getRestIp(context)
    devSN = devObj.get("sn")
    conn = RestConnection(connectorFactoryObj, ip, devSN)
    conn.create(devObj.get("user"), devObj.get("pawd"), restData.REST_SCOPE_DEFAULT)    
    setItem(context, REST_CONNECTION, conn)
    return
    
def getRest(context):
    '''
    @summary: 获取rest连接
    @param context: 上下文对象
    '''
    restConn = getItem(context,REST_CONNECTION)
    if restConn is None:
        initRestConnection(context)
        restConn = getItem(context,REST_CONNECTION)
        
    devObj = getNewDevObj(context)
    restConn.getRestConnection(devObj.get("user"), devObj.get("pawd"),restData.REST_SCOPE_DEFAULT)
    return restConn
    

def getRestIp(context):
    """
    针对V5R7C60 版本SVP选配场景，兼容老工具箱+新子工具箱场景
    @param context: 上下文
    @return:Rest 连接IP
    """
    oldDevObj = getDevObj(context)
    devObj = getNewDevObj(context)
    devType = getDevType(context)
    # 老工具箱场景
    if oldDevObj.getHighDevSVPModuleInfo() == CBB_CONFIG.SvpResult.OLD_TOOLBOX:
        if baseUtil.isHighEndDev(str(devType)):
            return config.SVP_REST_IP
        return devObj.get("ip")
    # 新工具箱 新子工具 有SVP模块场景
    elif oldDevObj.getHighDevSVPModuleInfo() == \
            CBB_CONFIG.SvpResult.NEW_TOOLBOX_HAS_SVP:
        return config.SVP_REST_IP
    else:
        return devObj.get("ip")

def is_dev_of_12GB_SAS_Share_Expansion(context, productVersion):
    '''
    @summary: Check whether the device is of 12 GB SAS Share Expansion('Big Card') or not.
    @param pyJavaEnv: Java python evrionment variable.
    @param productVersion: Product version.
    @return: True is it is else False.
    @attention: For V5 version, please confirm with product manager or SE.
    '''
    
    devType = getDevType(context)
    return ('18' in str(devType) and productVersion > 'V300R003')

def isDoradoDev(context):
    devType = getDevType(context)
    if devType in config.DORADO_DEVS:
        return True
    return False
