﻿# -*- coding: UTF-8 -*-

import os
from frame.common import common
from frame.context import contextUtil
import defusedxml.ElementTree as ET
from frame.base import config
from frame.base import baseUtil


##受限检查项配置配置文档
CHECKITEM_LIMIT_FILE = "check_item_limit.xml"

#根据版本号确定检查项的配置文件，配置字典
DEFAULT_XML_FILE = "checklist_innerversion_base.xml"
DEFAULT_NAS_XML_FILE = "checklist_innerversion_nas.xml"
SPECAIL_CHECKLIST_XML = {
    "Dorado": {
        "V300R001C00": "checklist_innerversion_lessthanv3r1c20.xml",
        "V300R001C01": "checklist_innerversion_lessthanv3r1c20.xml",
    },
    "V3": {
        "V300R003C00": "checklist_innerversion_v3r3c00.xml",
        "V300R003C20": "checklist_innerversion_v3r3c20.xml",
        "V300R006C00": "checklist_innerversion_v3r6c0x.xml",
        "V300R006C01": "checklist_innerversion_v3r6c0x.xml",
    }
}
VERSION_CHECKLIST_XML = {
    "Dorado": {

    },
    "V3": {
        "V300R005": "checklist_innerversion_v3r5.xml",
        "V300R006": "checklist_innerversion_v3r6.xml",  # 不包括V3R6C00/01
        "V500R007": "checklist_innerversion_v3r6.xml",
    }
}

def execute(context):
    '''
    @summary      : the entrance of main method, filter item according to the rules
    @param context: context
    @return       : flag, cliRet, errMsg
    '''
    flag = True
    errMsg = ''
    logger = common.getLogger(contextUtil.getLogger(context), __file__)
    
    curentVersion = contextUtil.getCurVersion(context)
    targetVersion = contextUtil.getTargetVersion(context)
    DeviceType = contextUtil.getDevType(context)
    logger.logInfo("current Version=[%s],target version=[%s],product model=[%s]" % (curentVersion, targetVersion, DeviceType))
    
    #获取升级模式
    upgradeMode = contextUtil.getUpgradeModel(context)
    logger.logInfo('upgradeMode:' + upgradeMode)
    
    filePath = getXmlFilePath(DeviceType, common.getVrcVersion(curentVersion))
    logger.logInfo('filePath:' + filePath)
    filterList = ET.parse(filePath)
    
    #根据受限配置文件过滤检查项
    filterList = getVerUpgLimitLst(context, curentVersion, targetVersion, DeviceType, filterList, logger)
    
    #根据升级模式过滤
    filterUpgradeManner(filterList, upgradeMode)
    
    #删除无关属性
    filterList=filterItemName(filterList)
    
    #删除没有孩子节点“item”的“module”节点
    filterNoChildModule(filterList)
    
    #node类型转化为str
    cliRet = getCheckList(filterList)
    logger.logInfo('check list:' + str(cliRet))
    return (flag, cliRet, errMsg)


def getXmlFilePath(DeviceType, VRCVersion):
    '''
    @summary: 获取当前版本调用的检查项xml文件
    @param curentVersion: 当前版本的VRC版本
    @return: 检查项配置文件的绝对路径
    '''
    curDir = os.path.dirname(__file__)
    VRVersion = VRCVersion[:8]
    if baseUtil.isDoradoNasDev(DeviceType):
        return os.path.join(curDir, DEFAULT_NAS_XML_FILE)

    productSeriers = "Dorado" if baseUtil.isDoradoDev(DeviceType) else "V3"
    checkListName = SPECAIL_CHECKLIST_XML.get(productSeriers).get(VRCVersion)
    if not checkListName:
        checkListName = VERSION_CHECKLIST_XML.get(productSeriers).get(VRVersion, DEFAULT_XML_FILE)
    
    return os.path.join(curDir, checkListName)

def getCheckList(node):
    '''
    @Function name      : getCheckList
    @Function describe  : 将节点转化为字符串型checklist
    @Input              : node
    @Return             : checklist
    '''
    root=node.getroot() 
    checklist= ET.tostring(root,encoding="utf-8")
    checklist='''<?xml version="1.0" encoding="UTF-8"?>\n'''+checklist
    return checklist

def filterUpgradeManner(checklist, method):
    '''
    @Function name      : filterUpgradeManner(checklist)
    @Function describe  : 根据升级方式过滤checklist
    @Input              : checklist, method
    @Return             : checklist
    '''
    for node in checklist.getiterator("module"):
        items=node.getchildren()
        iFlag=0
        lenItem=len(node.getchildren())
        while iFlag<lenItem:
            if items[iFlag].attrib.has_key("upgrademode") and items[iFlag].attrib["upgrademode"].lower()!='all':
                if items[iFlag].attrib["upgrademode"].lower() != method.lower():
                    node.remove(items[iFlag])
                    iFlag-=1
                    lenItem-=1
                iFlag+=1     
            #未配置upgrademode和upgrademode=ALL的情况，默认都为upgrademode=ALL
            else:
                iFlag+=1

def filterNoChildModule(checklist):
    '''
    @summary: filter the node named 'module' which has no child node
    @param context: the dictionary of data which provided by tool framework
    @param checklist: an object which stand for the XML configuration file is parsed as a tree
    '''
    rootNode = checklist.getroot()
    for nodeModule in rootNode.getchildren():
        curNodeItemList = nodeModule.getchildren()
        #删除空节点
        if not curNodeItemList:
            rootNode.remove(nodeModule)

def filterItemName(checklist):
    '''
    @Function name      : filterItemName(checklist)
    @Function describe  : 输出无关属性
    @Input              : checklist
    @Return             : checklist
    '''
    for node in checklist.getiterator("module"):
        if node is not None:
            del node.attrib["name"]
            for item in node.getchildren():
                del item.attrib["name"]
                if item.attrib.has_key("aftersecure"):
                    del item.attrib["aftersecure"]
                if item.attrib.has_key("upgralimit"):
                    del item.attrib["upgralimit"]
                if item.attrib.has_key("upgrademode"):#兼容upgrademode未配置的情况
                    del item.attrib["upgrademode"]
    return checklist

def getVerUpgLimitLst(context, curVer,targetVer,productModel,checklist, logger):
    '''
    @Function describe  : 根据源版本、目标版本、设备类型获取受限的检查项
    @Input              : curVer,targetVer,productModel,checklist
    @Return             : checklist过滤后的检查项列表
    '''
    curDir = os.path.dirname(__file__)
    cfgFilePath = os.path.join(curDir, CHECKITEM_LIMIT_FILE)
    filterList = ET.parse(cfgFilePath)
    
    for product in filterList.getiterator("product"):
        version = product.attrib.get('version')
        target = product.attrib.get('target')
        modelStr = product.attrib.get('model')
        oper = product.attrib.get('oper')
        
        if (not version or curVer.startswith(version)) and (not target or targetVer.startswith(target)) and (not modelStr or productModel in modelStr.split(',')):
            if oper == "add":
                checklist = addCheckItems(product, checklist)
            if oper == "delete":
                checklist = deleteCheckItems(product, checklist)
            if oper == 'preUpgradeCheckScene' and context.get("isPreUpgradeCheckScene"):
                checklist = addCheckItems(product, checklist)
            if oper == 'upgradeScene' and context.get("isUpgradeScene"):
                checklist = addCheckItems(product, checklist)
            if oper == 'onlyPreUpd' and context.get("isUpgradeScene"):
                checklist = deleteCheckItems(product, checklist)
            if oper == 'onlyPreUpd' and not context.get("isUpgradeScene"):
                checkFlag = False
                for tmpVersion in common.LOG_CHECK_SUPPORT_VERSION_LIST:
                    if curVer.startswith(tmpVersion):
                        checkFlag = True
                        break
                if not checkFlag:
                    checklist = deleteCheckItems(product, checklist)   
    return checklist  

def addCheckItems(product, checklist):
    '''
    @summary: 找到指定的检查分类，新增配置的检查项
    '''
    rootNode = checklist.getroot()
    for module in product.getchildren():
        index = int(module.attrib["index"])
        for item in module.getchildren():
            rootNode.getchildren()[index].append(item)
    return checklist

def deleteCheckItems(product, checklist):
    '''
    @summary: 如果当前配置存在删除项id，删除该检查项
    '''
    deleteIds = [item.attrib["id"] for item in product.getchildren()]
    for module in checklist.getiterator("module"):
        tempModule = list(module.getchildren())
        for item in tempModule:
            if item.attrib["id"] in deleteIds:
                module.remove(item)
    return checklist





