# -*- coding: UTF-8 -*-
from frame.cli import cliUtil
from frame.common import common
from frame.context import contextUtil

def execute(context):
    '''
    SAN双活状态检查
        1 如果双活仲裁链路状态不为Link Up，表示不正常；
        2 如果双活仲裁服务器运行状态不为Online，表示不正常；
        3 如果双活域的运行状态不为Normal表示不正常；
        4 如果双活Pair的健康状态不为Normal或者链路状态不为Link Up或者运行状态不为Normal/Synchronizing/Paused中的一种表示不正常；
        5 如果双活一致性组的健康状态不为Normal或者运行状态不为Normal/Synchronizing/Paused中的一种表示不正常；
        6 其它情况均属于正常状态。

    '''
    lang = contextUtil.getLang(context)
    logger = common.getLogger(contextUtil.getLogger(context), __file__)
    cli = contextUtil.getSSH(context)
    
    flag = True
    cliRet = ""
    errMsg = ""
    
    try:
        #先执行命令确认特性支持，再查询后面处理
        checkCmd = "show hyper_metro_domain general"
        checkFlag, checkCliRet = cliUtil.isSupportFeatureBySSH(cli, checkCmd)
        if not checkFlag:
            return common.getUpgEvaluationRs(cliUtil.RESULT_NOSUPPORT, checkCliRet, errMsg)

        checkResults = (coordinatorPointServerLink(cli,logger,lang), coordinatorPointServer(cli,logger,lang), hyperClusterDomain(cli,logger,lang),hyperConsistGroup(cli,logger,lang), hyperClusterPair(cli,logger,lang))
        flagList = [ret[0] for ret in checkResults]
        flag = common.getCheckResult(flagList)
        
        for ret in checkResults:
            cliRet += ret[1] + "\n"
            errMsg += ret[2]
        
        return common.getUpgEvaluationRs(flag, cliRet, errMsg)
        
    except Exception, exception:
        logger.logException(exception)
        return common.getUpgEvaluationRs(False, cliRet, common.getMsg(lang, "query.result.abnormal"))   
    

def coordinatorPointServerLink(cli,logger,lang):
    '''
    @summary: 检查双活仲裁服务器链路状态是否正常
    @param cli: cli对象
    @return: (flag, cliRet, errMsg)
        flag: 
            True: 状态健康
            False: 状态不健康
        cliRet: 
            Flag为True时，返回设备上的信息
            Flag为False时，返回cli回显结果
        errMsg: 错误消息
        suggestionInfo:修改建议（15365）
    '''
    errMsg = ""
    flag = True
     
    cmd = "show quorum_server_link general"
    logger.logExecCmd(cmd)
    checkRet = cliUtil.excuteCmdInCliMode4Privilege(cli, cmd, True, lang)
    if checkRet[0] != True: 
        logger.logSysAbnormal()
        return checkRet
    
    cliRet = checkRet[1]
    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)
    
    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
    if len(cliRetLinesList) == 0:
        errMsg = common.getMsg(lang, "cannot.get.quorum.server.link.info")
        logger.logNoPass("Failed to obtain the status of active-active quorum links.")
        return (False, cliRet, errMsg)

    for line in cliRetLinesList:
        linkId = line["Link ID"]
        linkStatus = line["Link Status"]
        if linkStatus != common.STATUS_LINK_UP:
            flag = False
            errMsg += common.getMsg(lang, "quorum.server.link.status.abnormal", (linkId, linkStatus))
            logger.logNoPass("The status of the links to active-active quorum server [%s] is abnormal (link status: %s)." 
                 % (linkId, linkStatus))
     
    if not flag:
        return (flag, cliRet, errMsg)  
        
    return (flag, cliRet, errMsg)

def coordinatorPointServer(cli,logger,lang):
    '''
    @summary: 查看双活仲裁服务器运行状态是否正常
    @param cli: cli对象
    @return: (flag, cliRet, errMsg)
        flag: 
            True: 状态健康
            False: 状态不健康
        cliRet: 
            Flag为True时，返回设备上的信息
            Flag为False时，返回cli回显结果
        errMsg: 错误消息
        suggestionInfo:修改建议（15363）
    '''
    errMsg = ""
    flag = True
    
    cmd = "show quorum_server general|filterColumn include columnList=Server\sID,Running\sStatus"
    logger.logExecCmd(cmd)
    checkRet = cliUtil.excuteCmdInCliMode4Privilege(cli, cmd, True, lang)
    if checkRet[0] != True: 
        logger.logSysAbnormal()
        return checkRet
    
    cliRet = checkRet[1]
    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)
    
    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
    if len(cliRetLinesList) == 0:
        errMsg = common.getMsg(lang, "cannot.get.quorum.server.info")
        logger.logNoPass("Failed to obtain the running status of the active-active quorum server.")
        return (False, cliRet, errMsg)

    for line in cliRetLinesList:
        serverId = line["Server ID"]
        runningStatus = line["Running Status"]
        if runningStatus != common.STATUS_ONLINE:
            flag = False
            errMsg += common.getMsg(lang, "quorum.server.running.status.abnormal", (serverId, runningStatus))
            logger.logNoPass("The running status of active-active quorum server [%s] is abnormal (running status: %s)." 
                 % (serverId, runningStatus))
      
    if not flag:
        return (flag, cliRet, errMsg) 
      
    return (flag, cliRet, errMsg)
    
def hyperClusterDomain(cli,logger,lang):
    '''
    @summary: 检查双活域运行状态是否正常
    @param cli: cli对象
    @return: (flag, cliRet, errMsg)
        flag: 
            True: 状态健康
            False: 状态不健康
        cliRet: 
            Flag为True时，返回设备上的信息
            Flag为False时，返回cli回显结果
        errMsg: 错误消息
        suggestionInfo:修改建议（15362）
    '''
    errMsg = ""
    flag = True
    
    cmd = "show hyper_metro_domain general"
    logger.logExecCmd(cmd)
    checkRet = cliUtil.excuteCmdInCliMode4Privilege(cli, cmd, True, lang)
    if checkRet[0] != True: 
        logger.logSysAbnormal()
        return checkRet
    
    cliRet = checkRet[1]
    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)
    
    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
    if len(cliRetLinesList) == 0:
        errMsg = common.getMsg(lang, "cannot.get.hyper.metro.domain.info")
        logger.logNoPass("Failed to obtain the running status of the active-active domain.")
        return (False, cliRet, errMsg)

    for line in cliRetLinesList:
        Id = line["ID"]
        runningStatus = line["Running Status"]
        if runningStatus != common.STATUS_NORMAL:
            flag = False
            errMsg += common.getMsg(lang, "hyper.metro.domain.running.status.abnormal", (Id, runningStatus))
            logger.logNoPass("The Health Status of hyper cluster domain [%s] is abnormal (Running Status:%s)" 
                 % (Id, runningStatus))
    if not flag:
        return (flag, cliRet, errMsg)  
        
    return (flag, cliRet, errMsg)    


def hyperConsistGroup(cli,logger,lang):
    '''
    @summary: 检查双活一致性组状态是否正常
    @param cli: cli对象
    @return: (flag, cliRet, errMsg)
        flag: 
            True: 状态健康
            False: 状态不健康
        cliRet: 
            Flag为True时，返回设备上的信息
            Flag为False时，返回cli回显结果
        errMsg: 错误消息
        suggestionInfo:修改建议（15364）
    '''
    errMsg = ""
    flag = True
    
    cmd = "show hyper_metro_consistency_group general"
    logger.logExecCmd(cmd)
    checkRet = cliUtil.excuteCmdInCliMode4Privilege(cli, cmd, True, lang)
    
    cliRet = checkRet[1]
    if not cliUtil.hasCliExecPrivilege(cliRet):
        return (True, "", errMsg)
    
    if checkRet[0] != True: 
        logger.logSysAbnormal()
        return checkRet
    
    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)
    
    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
    if len(cliRetLinesList) == 0:
        errMsg = common.getMsg(lang, "cannot.get.hyper.consist.group.info")
        logger.logNoPass("Failed to obtain the information about the status of the active-active consistency groups.")
        return (False, cliRet, errMsg)

    for line in cliRetLinesList:
        Id = line["ID"]
        healthStatus = line["Health Status"]
        runningStatus = line["Running Status"]
        
        if healthStatus != common.STATUS_NORMAL:
            flag = False
            errMsg += common.getMsg(lang, "hyper.consist.group.health.status.abnormal", (Id, healthStatus))
            logger.logNoPass("The health status of active-active consistency group [%s] is abnormal (health status: %s)." 
                 % (Id, healthStatus))
        
        if runningStatus not in (common.STATUS_NORMAL, common.STATUS_SYNCHRONIZING, common.STATUS_PAUSED):
            flag = False
            errMsg += common.getMsg(lang, "hyper.consist.group.running.status.abnormal", (Id, runningStatus))
            logger.logNoPass("The running status of active-active consistency group [%s] is abnormal (running status: %s)." 
                 % (Id, runningStatus))
    
    if not flag:
        return (flag, cliRet, errMsg)  
        
    return (flag, cliRet, errMsg)      
    
    
def hyperClusterPair(cli,logger,lang):
    '''
    @summary: 检查双活Pair状态是否正常
    @param cli: cli对象
    @return: (flag, cliRet, errMsg)
        flag: 
            True: 状态健康
            False: 状态不健康
        cliRet: 
            Flag为True时，返回设备上的信息
            Flag为False时，返回cli回显结果
        errMsg: 错误消息
        suggestionInfo:修改建议（15361）
    '''
    errMsg = ""
    flag = True
    
    cmd = "show hyper_metro_pair general|filterColumn include columnList=ID,Health\sStatus,Link\sStatus,Running\sStatus"
    logger.logExecCmd(cmd)
    checkRet = cliUtil.excuteCmdInCliMode4Privilege(cli, cmd, True, lang)
    if checkRet[0] != True: 
        logger.logSysAbnormal()
        return checkRet
    
    cliRet = checkRet[1]
    if cliUtil.queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)

    cliRetLinesList = cliUtil.getHorizontalCliRet(cliRet)
    if len(cliRetLinesList) == 0:
        errMsg = common.getMsg(lang, "cannot.get.hyper.metro.pair.info")
        logger.logNoPass("Failed to obtain the status of the active-active pair.")
        return (False, cliRet, errMsg)

    for line in cliRetLinesList:
        Id = line["ID"]
        healthStatus = line["Health Status"]
        linkStatus = line["Link Status"]
        runningStatus = line["Running Status"]
        
        if healthStatus != common.STATUS_NORMAL:
            flag = False
            errMsg += common.getMsg(lang, "hyper.metro.pair.health.status.abnormal", (Id, healthStatus))
            logger.logNoPass("The health status of active-active pair [%s] is abnormal (health status: %s)." 
                 % (Id, healthStatus))
        
        if linkStatus not in (common.STATUS_LINK_UP, "Linkup"):
            flag = False
            errMsg += common.getMsg(lang, "hyper.metro.pair.link.status.abnormal", (Id, linkStatus))
            logger.logNoPass("The link status of active-active pair [%s] is abnormal (link status: %s)." 
                 % (Id, linkStatus))
        
        if runningStatus not in (common.STATUS_NORMAL, common.STATUS_SYNCHRONIZING, common.STATUS_PAUSED):
            flag = False
            errMsg += common.getMsg(lang, "hyper.metro.pair.running.status.abnormal", (Id, runningStatus))
            logger.logNoPass("The running status of active-active pair [%s] is abnormal (running status: %s)." 
                 % (Id, runningStatus))
    
    if not flag:
        return (flag, cliRet, errMsg)  
        
    return (flag, cliRet, errMsg)        
    
