# -*- coding: UTF-8 -*-
import re
from frame.cli import cliUtil
from frame.common import common
from frame.context import contextUtil
import traceback
from com.huawei.ism.tool.upgradeevalu.parser import XmlItemParser

# 主机HBA黑名单检查
def execute(context):
    logger = common.getLogger(contextUtil.getLogger(context), __file__)
    lang = contextUtil.getLang(context)
    devNode = contextUtil.getDevObj(context)
    try:
        status, cliRet, errMsg = HBACheck(context).check()
        if status != CheckStatus.PASS:
            devNode.setHostCompatible(False)  # 标记当前设备无法通过阵列侧检查
        return common.getUpgEvaluationRs(status, cliRet, errMsg)
    except:
        logger.logInfo("check hostHbaInfo exception[%s]" % traceback.format_exc())
        errMsg = common.getMsg(lang, "excute.cmd.error")
        devNode.setHostCompatible(False)  # 标记当前设备无法通过阵列侧检查
        return common.getUpgEvaluationRs(cliUtil.RESULT_NOCHECK, "", errMsg)

class CheckStatus():
    PASS = True
    WARNING = cliUtil.RESULT_WARNING
    NOCHECK = cliUtil.RESULT_NOCHECK

class HBACheck():
    isFirst = True
    blackRecords = []
    hbaNodes = []
    def __init__(self, context):
        # 从上一个检查项获取HBA信息
        self.allHBAInfo = context.get("allHBAInfo")
        self.lang = contextUtil.getLang(context)
        self.logger = common.getLogger(contextUtil.getLogger(context), __file__)
        self.allCliRet = ""

    def check(self):
        if self.allHBAInfo is None or len(self.allHBAInfo) == 0:
            self.logger.logInfo("host not exist.the hba info is empty.")
            return CheckStatus.PASS, "", ""
        self.xmlParser = XmlItemParser.getInstance()
        allBlackHbaInfo = []
        notEvaluBlackHbaInfo = []
        for hostName in self.allHBAInfo.keys():
            oneHostHbainfo = self.allHBAInfo.get(hostName)
            blackWwpnList = []
            notEvaluWwpnList = []
            for tempHbaInfo in oneHostHbainfo:
                wwpn = tempHbaInfo.get("wwpn")
                hbaInfo = tempHbaInfo.get("hbaInfo")
                ok, hbaFormat = self._parseHBAInfo(hbaInfo)
                if not ok:
                    # 解析出错，报无法评估
                    self.logger.logInfo("the hbaInfo=%s parse error." % hbaInfo)
                    notEvaluWwpnList.append(wwpn)
                    self.allCliRet += "\nthe hba %s parse error." % hbaInfo
                    continue
                isInBlack, suggestion = self._isInCompatibilityBlackList(hbaFormat)
                if isInBlack == "ERROR":
                    self.logger.logInfo("hba=%s check in blacklist excpet." % hbaInfo)
                    self.allCliRet += "\nthe hba %s check error." % hbaInfo
                    notEvaluWwpnList.append(wwpn)
                elif isInBlack:
                    self.logger.logInfo("hba=%s is in blacklist" % hbaInfo)
                    blackWwpnList.append(wwpn)
                    self.allCliRet += "\nthe hba %s check in blacklist." % hbaInfo

            if len(blackWwpnList) > 0:
                allBlackHbaInfo.append(dict(hostName=hostName, wwpn=blackWwpnList))

            if len(notEvaluWwpnList) > 0:
                notEvaluBlackHbaInfo.append(dict(hostName=hostName, wwpn=notEvaluWwpnList))

        errMsg = ""
        status = CheckStatus.PASS
        if len(notEvaluBlackHbaInfo) > 0:
            status = CheckStatus.NOCHECK
            errMsg += common.getMsg(self.lang, "hostHbaInfo.nocheck", self.formatOutput(notEvaluBlackHbaInfo))
            errMsg += "\n"
        if len(allBlackHbaInfo) > 0:
            errMsg += common.getMsg(self.lang, "hostHbaInfo.warning", self.formatOutput(allBlackHbaInfo))
            status = CheckStatus.WARNING
        return status, self.allCliRet, errMsg

    # 将主机按每5个换行显示
    def formatOutput(self, results):
        ret = ""
        for i in range(len(results)):
            if i % 5 == 0:
                ret += "\n    "
            temp = results[i]
            ret += temp.get("hostName")
            if temp.has_key("wwpn"):
                ret += common.getMsg(self.lang, "hostcompatibility.result.wwpn", ','.join(temp.get("wwpn")))
            ret += ", "
        ret = ret[:-2] if ret.endswith(", ") else ret
        return ret

    def _parseHBAInfo(self, hbaInfo):
        '''
        博科：
            Emulex 42D0494 FV1.11A5  DV2.74.214.004 VEEAMBACKUP
            QLE2672 FW:v7.03.00 DVR:v2.1.30.0
        思科：
            Emulex LPe11002 FV2.71X15 DV8.2.0.128.3p
            Emulex LPe1250-F8 FV2.01A13 DV11.1.0.6 HN:localhost OS:VMware ESXi 6.5.0
            QLE2462 FW:v5.03.01 DVR:v8.03.01.12.10.3-k
            QLE2672 FW:v6.02.00 DVR:v2.1.53.0
            QLE2560 FW:v8.02.00 DVR:v2.1.53.0
        阵列：
            Emulex LPe16002-M6 FV10.6.144.21 DV11.1.145.16 HN:WIN2008-74 OS:Windows 2008
            QLE2562 FW:v5.08.00 DVR:8.04.00.13.11.3-k
        :return: 
        '''
        EmulexRegex = "^(Emulex\s\S+)\s+FV(\S+)\s+DV(\S+)"
        QlogicRegex = "^(QLE\S+)\s+FW:v(\S+)\s+DVR:v?(\S+)"
        rec = re.match(EmulexRegex, hbaInfo, re.IGNORECASE)
        if not rec:
            rec = re.match(QlogicRegex, hbaInfo)
        if rec:
            hba = dict()
            hba["adapter"] = rec.group(1).strip()
            hba["firmware"] = rec.group(2).strip()
            hba["driver"] = rec.group(3).strip()
            return True, hba
        return False, ""

    def _isInCompatibilityBlackList(self, hba):
        adapter = hba.get("adapter")
        firmware = hba.get("firmware")
        driver = hba.get("driver")
        self.logger.logInfo("call java checkInHbaBlackList.adapter=%s, firmware=%s, driver=%s" % (adapter, firmware, driver))

        try:
            record = self.xmlParser.checkHbaInBlackList(adapter, driver, firmware, None)
            if record is None or record.get("ID") == "2":
                return False, ""
            else:
                self.logger.logInfo("the blacklist record=%s" % record.get("ID"))
                return True, ""
        except:
            return "ERROR", ""



