# -*- coding: UTF-8 -*-
#  Copyright (c) Huawei Technologies Co., Ltd. 2024-2024. All rights reserved.

from common import common
from common import context_util
from cbb.frame.cli import cliUtil
from cbb.frame.base import baseUtil
from cbb.business.checkitems import ssh_port_check
import time
from utils import Products
from com.huawei.ism.tool.obase.utils import ManualConfigItemMgr


def execute(context):
    """
    @summary: the entrance of main method, do preparing check before evaluation
    @param context: the dictionary of data which provided by tool framework
    @return: (pass status, CLI information, error message) as (boolean, string, string)
    """
    logger = baseUtil.getLogger(context_util.get_logger(context), __file__)
    lang = context_util.get_lang(context)
    cli = cliUtil.getAvaliableCli(context_util.get_cli(context))
    item = "pre_check"
    try:
        flag = common.RESULT_PASS
        cli_ret = ""
        err_msg = ""
        flag_lst = []
        err_msg_lst = []
        # 检查设备状态
        flag, err_msg = is_device_normal(context)
        if common.RESULT_NOTPASS == flag:
            logger.logNoPass('check if the device is not normal!')
            return flag, cli_ret, err_msg

        # 检查设备是否为单控模式
        fun_flag = cliUtil.isSigleModel(cli, lang)
        if fun_flag and context_util.get_upgrade_model(context) == common.MODEL_ONLINE_UPGRADE:
            logger.logNoPass('Single-Controller does not support online-upgrade evaluation!')
            return common.RESULT_NOTPASS, cli_ret, common.get_msg(lang, "unsupport.single.online")

        # 检查SSH默认端口是否为22.
        ssh_port_flag, err_msg = ssh_port_check.execute(context)
        if not ssh_port_flag:
            logger.logNoPass('[precheck]ssh port is not 22.')
            return common.RESULT_NOTPASS, "", err_msg

        common.set_process(context, common.PROGRESS_NUM_1, item)

        err_msg_lst = pre_check_item(context, err_msg_lst, flag_lst, item)

        common.set_process(context, common.PROGRESS_NUM_60, item)
        # 获取集群所有控制器IP，并存入上下文中
        ctrls_ips_ret = cliUtil.getCtrlIps(cli, logger, lang)
        logger.logInfo('get all ctrollers ip result:%s' % str(ctrls_ips_ret))
        context_util.set_item(context, "allCtrlsIp", ctrls_ips_ret[1])
        common.set_process(context, common.PROGRESS_NUM_80, item)
        if common.RESULT_NOTPASS in flag_lst:
            flag = common.RESULT_NOTPASS

        for index, msg in enumerate(err_msg_lst, start=1):
            err_msg += common.get_msg(lang, "show.problems", (index, msg))

        if not flag:
            create_and_save_timestamp(context)
        return flag, cli_ret, err_msg
    finally:
        logger.logInfo("the progress is 100")
        common.set_process(context, common.PROGRESS_NUM_MAX, item)


def pre_check_item(context, err_msg_lst, flag_lst, item):
    logger = baseUtil.getLogger(context_util.get_logger(context), __file__)
    lang = context_util.get_lang(context)
    cli = cliUtil.getAvaliableCli(context_util.get_cli(context))
    fun_flag, fun_err_msg = is_admin_user(cli, lang)
    flag_lst.append(fun_flag)
    if common.RESULT_NOTPASS == fun_flag:
        err_msg_lst += fun_err_msg
        logger.logNoPass('check if it is admin user failed!')
    common.set_process(context, common.PROGRESS_NUM_20, item)
    fun_flag, fun_err_msg = is_device_ver_valid(context)
    flag_lst.append(fun_flag)
    if common.RESULT_NOTPASS == fun_flag:
        err_msg_lst += fun_err_msg
        logger.logNoPass('check if device version is valid failed!')
    common.set_process(context, common.PROGRESS_NUM_40, item)
    fun_flag, fun_err_msg = is_version_valid(context)
    flag_lst.append(fun_flag)
    if common.RESULT_NOTPASS == fun_flag:
        err_msg_lst += fun_err_msg
        logger.logNoPass('check if destination version is valid failed!')
    return err_msg_lst


def is_admin_user(cli, lang):
    """
    @Function name      : isAdminUser
    @Function describe  : 判断当前用户权限级别是否是Super_admin登录
    @Input              : cli, lang
    @Return             : funFlag,funErrMsg
    """
    fun_flag = common.RESULT_NOTPASS
    fun_err_msg = []

    check_ret = cliUtil.getUserPrivilege(cli, lang)
    if check_ret[0] is not True:
        return check_ret[0], check_ret[2]

    if check_ret[1].upper() == 'Super_admin'.upper():
        fun_flag = common.RESULT_PASS
    else:
        fun_err_msg.append(common.get_msg(lang, "user.authority.failure"))
    return fun_flag, fun_err_msg


def is_device_normal(context):
    """
    @summary: judge whether the device status is normal
    @param context: the context object provided by tool framework
    @return: the status of device which is normal or abnormal, as boolean
    """
    fun_flag = common.RESULT_PASS
    fun_err_msg = ""
    err_msg = ""
    cli_ret = ""
    cli = cliUtil.getAvaliableCli(context_util.get_cli(context))
    lang = context_util.get_lang(context)
    logger = baseUtil.getLogger(context_util.get_logger(context), __file__)
    cmd = "show system general"
    check_ret = cliUtil.excuteCmdInCliMode(cli, cmd, True, lang)
    if check_ret[0] is not True:
        logger.logNoPass("show system general error.")
        return common.RESULT_NOTPASS, check_ret[2]

    cli_ret = check_ret[1]
    cli_ret_lines_list = cliUtil.getVerticalCliRet(cli_ret)
    if len(cli_ret_lines_list) == 0:
        return common.RESULT_NOTPASS, err_msg

    for ret_dict in cli_ret_lines_list:
        sys_running_status = ret_dict["Running Status"]
        sys_health_status = ret_dict["Health Status"]
        if sys_running_status != common.STATUS_NORMAL or sys_health_status != common.STATUS_NORMAL:
            fun_flag = common.RESULT_NOTPASS
            logger.logNoPass('[pre_check]System status is not normal.')
            fun_err_msg = common.get_msg(lang, "system.status.abnormal")

    return fun_flag, fun_err_msg


def is_version_valid(context):
    """
    @Function name      : isVersionValid
    @Function describe  : 判断版本是否有效
    @Input              : context
    @Return             : funFlag,funErrMsg,devType
    """

    fun_flag = common.RESULT_PASS
    fun_err_msg = []
    # 源版本
    cur_ver = context_util.get_cur_version(context)
    # 目标版本
    dest_ver = context_util.get_target_version(context)
    # 语言环境
    lang = context_util.get_lang(context)
    if ManualConfigItemMgr.getInstance().expected("permanent.support.downgrade", "yes"):
        return fun_flag, fun_err_msg
    # 向框架提需求，目标版本不能为空
    # 有效性检查（1）：输入的当前版本号不能为空
    if not dest_ver:
        fun_flag = common.RESULT_NOTPASS
        fun_err_msg.append(common.get_msg(lang, "target.version.not.exist"))

    # 有效性检查（2）：当前版本号与目标版本号不允许相同
    up_mode = str(context.get('upgradeMode'))
    if cur_ver == dest_ver and up_mode != "APOLLO":
        fun_flag = common.RESULT_NOTPASS
        fun_err_msg.append(common.get_msg(lang, "original.equal.to.target"))
    dev = context.get('dev')
    upgrade_mode = str(dev.getUpgradeModeUI())
    # 6.1.99版本仅支持通过快速升级和滚动升级到6.1.7
    if cur_ver.startswith("6.1.99") and "6.1.7" in dest_ver and upgrade_mode in ("ROLL", "FAST"):
        return fun_flag, fun_err_msg

    # 有效性检查（3）：判断是否为回退
    # 非回退检查
    # 当前C版本大于目标C版本则提示不支持回退检查
    if Products.compareVersion(cur_ver, dest_ver) > 0:
        fun_flag = common.RESULT_NOTPASS
        fun_err_msg.append(common.get_msg(lang, "original.less.to.target"))
    return fun_flag, fun_err_msg


def is_device_ver_valid(context, is_contain_spc=True):
    """
    @summary: 判断上下文中的版本信息是否有效
    @param context: 上下文
    @param is_contain_spc: 比较版本信息时是否包含SPC版本的标记，默认包含SPC即比较时要精确到SPC版本版本。如果不包含SPC，则比较时，只比较V、R、C是否一致
    @return:
        True: 版本一致
        False: 版本不一致
    """
    lang = context_util.get_lang(context)
    cli = cliUtil.getAvaliableCli(context_util.get_cli(context))
    logger = baseUtil.getLogger(context_util.get_logger(context), __file__)

    fun_flag = common.RESULT_NOTPASS
    fun_err_msg = []

    # 从上下文中获取当前版本并转化为标准格式
    cur_version = baseUtil.formatVersion(context_util.get_cur_version(context))

    # 从设备上获取当前版本
    flag, device_version, cli_ret = cliUtil.getDevVersion(cli, lang, logger)
    if not flag:
        fun_err_msg.append(common.get_msg(lang, "cannot.get.version"))
        return fun_flag, fun_err_msg

    # 标准化设备版本
    device_version = baseUtil.formatVersion(device_version)

    if not is_contain_spc:
        cur_version = baseUtil.getVRCVersion(cur_version)
        device_version = baseUtil.getVRCVersion(device_version)

    if cur_version == device_version:
        fun_flag = common.RESULT_PASS
    else:
        fun_err_msg.append(common.get_msg(lang, "need.refresh.device", (device_version, cur_version)))
    return fun_flag, fun_err_msg


def create_and_save_timestamp(context):
    """
    @summary: 设置当前设备的存放文件的临时目录，存放在上下文中
    """
    cur_time = time.strftime('%Y%m%d%H%M%S', time.localtime(time.time()))
    device_sn = context_util.get_dev_sn(context)
    tmp_file_name = device_sn + "_" + cur_time

    context_util.set_item(context, common.TMP_FILE_NAME, tmp_file_name)
    return
