﻿# -*- coding: UTF-8 -*-

import os
from common import contextUtil
import defusedxml.ElementTree as ET
from cbb.frame.base import baseUtil


# 受限检查项配置配置文档
DEFAULT_XML_FILE = "checklist_innerversion_base.xml"

# 根据版本号确定检查项的配置文件，配置字典


def execute(context):
    """
    @summary      : the entrance of main method, filter item according to the rules
    @param context: context
    @return       : flag, cliRet, errMsg
    """
    flag = True
    errMsg = ""
    logger = baseUtil.getLogger(contextUtil.getLogger(context), __file__)

    curentVersion = contextUtil.getCurVersion(context)
    targetVersion = contextUtil.getTargetVersion(context)
    DeviceType = contextUtil.getDevType(context)
    logger.logInfo(
        "current Version=[%s],target version=[%s],product model=[%s]"
        % (curentVersion, targetVersion, DeviceType)
    )

    # 获取升级模式
    upgradeMode = contextUtil.getUpgradeModel(context)
    logger.logInfo("upgradeMode:" + upgradeMode)

    filePath = getXmlFilePath(DeviceType, curentVersion)
    logger.logInfo("filePath:" + filePath)
    filterList = ET.parse(filePath)

    # 根据升级模式过滤
    filterUpgradeManner(filterList, upgradeMode)

    # 删除无关属性
    filterList = filterItemName(filterList)

    # 删除没有孩子节点“item”的“module”节点
    filterNoChildModule(filterList)

    # node类型转化为str
    cliRet = getCheckList(filterList)
    logger.logInfo("check list:" + str(cliRet))
    return (flag, cliRet, errMsg)


def getXmlFilePath(DeviceType, VRCVersion):
    """
    @summary: 获取当前版本调用的检查项xml文件
    @param curentVersion: 当前版本的VRC版本
    @return: 检查项配置文件的绝对路径
    """
    curDir = os.path.dirname(__file__)
    return os.path.join(curDir, DEFAULT_XML_FILE)


def getCheckList(node):
    """
    @Function name      : getCheckList
    @Function describe  : 将节点转化为字符串型checklist
    @Input              : node
    @Return             : checklist
    """
    root = node.getroot()
    checklist = ET.tostring(root, encoding="utf-8")
    checklist = """<?xml version="1.0" encoding="UTF-8"?>\n""" + checklist
    return checklist


def filterUpgradeManner(checklist, method):
    """
    @Function name      : filterUpgradeManner(checklist)
    @Function describe  : 根据升级方式过滤checklist
    @Input              : checklist, method
    @Return             : checklist
    """
    for node in checklist.getiterator("module"):
        items = node.getchildren()
        iFlag = 0
        lenItem = len(node.getchildren())
        check_upgrade_type(iFlag, items, lenItem, method, node)


def check_upgrade_type(iFlag, items, lenItem, method, node):
    while iFlag < lenItem:
        if items[iFlag].attrib.get("upgrademode", "").lower() != "all":
            if items[iFlag].attrib.get("upgrademode",
                                       "").lower() != method.lower():
                node.remove(items[iFlag])
                iFlag -= 1
                lenItem -= 1
            iFlag += 1
        # 未配置upgrademode和upgrademode=ALL的情况，默认都为upgrademode=ALL
        else:
            iFlag += 1


def filterNoChildModule(checklist):
    """
    @summary: filter the node named 'module' which has no child node
    @param context: the dictionary of data which provided by tool framework
    @param checklist: an object which stand for the XML configuration file is parsed as a tree
    """
    rootNode = checklist.getroot()
    for nodeModule in rootNode.getchildren():
        curNodeItemList = nodeModule.getchildren()
        # 删除空节点
        if not curNodeItemList:
            rootNode.remove(nodeModule)


def filterItemName(checklist):
    """
    @Function name      : filterItemName(checklist)
    @Function describe  : 输出无关属性
    @Input              : checklist
    @Return             : checklist
    """
    for node in checklist.getiterator("module"):
        if node is None:
            continue
        del node.attrib["name"]
        for item in node.getchildren():
            del item.attrib["name"]
            if "aftersecure" in item.attrib:
                del item.attrib["aftersecure"]
            if "upgralimit" in item.attrib:
                del item.attrib["upgralimit"]
            if "upgrademode" in item.attrib:  # 兼容upgrademode未配置的情况
                del item.attrib["upgrademode"]
    return checklist


def addCheckItems(product, checklist):
    """
    @summary: 找到指定的检查分类，新增配置的检查项
    """
    rootNode = checklist.getroot()
    for module in product.getchildren():
        index = int(module.attrib["index"])
        for item in module.getchildren():
            rootNode.getchildren()[index].append(item)
    return checklist


def deleteCheckItems(product, checklist):
    """
    @summary: 如果当前配置存在删除项id，删除该检查项
    """
    deleteIds = [item.attrib["id"] for item in product.getchildren()]
    for module in checklist.getiterator("module"):
        tempModule = list(module.getchildren())
        for item in tempModule:
            if item.attrib["id"] in deleteIds:
                module.remove(item)
    return checklist
