# -*- coding: UTF-8 -*-

import os
import re
import shutil
import traceback
import time
import cmdManager
from constant import CheckedResult, DeviceType
from cmdRetManager import getCliRet
from contextUtil import getLang, getLogger, getSshObj, getSftpObj, getPackMgr, getTimeStamp, getTmpDir


#一键收集日志命令超时时间（25分钟）
COLLECT_CMD_TIMEOUT = 25 * 60

def getDataCollectDecompressDir(dataDict):
    '''
    @summary: 获取一键收集日志的解压目录
    @param dataDict: the context object provided by tool framework
    @return: flag, dataCollectRet, errMsg, deCompressDestDir
    '''
    ssh = getSshObj(dataDict)
    fileHandle = getPackMgr(dataDict)
    log = getLogger(dataDict)
    lang = getLang(dataDict)

    flag = CheckedResult.PASS
    dataCollectRet = ""
    errMsg = ''
    deCompressDestDir = ""
    
    #本地保存一键解压日志的文件夹名称
    timeStamp = getTimeStamp(dataDict)
    localResultDirName = "Datacollect_" + timeStamp

    #获取数据解压临时文件夹
    dataCollectTmpDir = getCollectTmpDir(dataDict)
    log.info("collect temp file dir is:" + dataCollectTmpDir)
    
    #步骤1：查看是否此次检查前面的检查下已经保存了
    if os.path.exists(dataCollectTmpDir) and os.path.isdir(dataCollectTmpDir):
        #获取解压目录
        deCompressDestDir =  dataCollectTmpDir + os.path.sep + localResultDirName
        if os.path.exists(dataCollectTmpDir):
            log.info("Decompress datacollect file directory already exists: " + deCompressDestDir)
            return (flag, dataCollectRet, errMsg, deCompressDestDir)

    #步骤2：需要执行一键收集日志（设置超时时间25分钟）
    dataCollectRet = cmdManager.execCmdWithTimout(dataDict, "datacollect", COLLECT_CMD_TIMEOUT)
    if re.search("The system is busy", dataCollectRet, re.IGNORECASE):
        flag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"当前系统繁忙，收集系统日志失败，请稍后重试。"
        else:
            errMsg = "The system is busy, failed to collect system log. Please wait for a while then check again."
        return (flag, dataCollectRet, errMsg, deCompressDestDir)
    
    if re.search("Part of", dataCollectRet, re.IGNORECASE):
        flag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"系统日志部分收集成功。"
        else:
            errMsg = "Part of the system log export successfully."
        return (flag, dataCollectRet, errMsg, deCompressDestDir)
    
    #命令执行失败
    if not re.search("Path :", dataCollectRet, re.IGNORECASE):
        flag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"收集系统日志失败。"
        else:
            errMsg = "Failed to collect system log."
        return (flag, dataCollectRet, errMsg, deCompressDestDir)
        
    #步骤3：获取远程文件路径
    remotePath = ""
    remoteFileName = ""
    lineList = dataCollectRet.splitlines()
    
    #获取远端阵列上的临时文件全路径
    for line in lineList:
        if line.find("Path :") != -1:
            lineSub = line[line.index("Path :"):]
            remotePath = lineSub.replace("Path :", "")
            break
    if remotePath == "":
        flag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"获取日志收集文件失败。"
        else:
            errMsg = "Getting system log file failed."
        return (flag, dataCollectRet, errMsg, deCompressDestDir)
    
    #步骤4：通过SFTP下载文件
    #获取远端文件名
    remoteFileName = ""
    remoteFileName = remotePath.split("/")[-1]
    #通过SFTP下载文件
    sftp = getSftpObj(dataDict)
    remoteFileDir = remotePath.replace(remoteFileName, "")
    
    #创建临时文件夹
    if not os.path.exists(dataCollectTmpDir):
        os.makedirs(dataCollectTmpDir)
    
    #指定文件名称为Datacollect（避免文件名过长）
    localFileName = localResultDirName + ".tgz"
    localFile = dataCollectTmpDir + os.sep + localFileName
    
    #通过sftp下载文件
    try:
        sftp.getFile(remotePath, localFile, None)
    except Exception, exception:
        #打印错误信息
        log.error("catch except when download log file over!")
        log.error("except messages:" + unicode(exception))
        log.error("except trace back:" + unicode(traceback.format_exc()))

        flag = CheckedResult.WARN
        if lang == "zh":
            errMsg = u"通过sftp下载文件失败。"
        else:
            errMsg = "Downloading the file by sftp failed."
        return (flag, dataCollectRet, errMsg, deCompressDestDir)
    finally:
        #删除设备上临时文件
        deleteRemoteFile(dataDict, remotePath)
    
    #步骤5：解压本地日志文件
    deCompressCmds = ['', 'OceanspaceS5000T', 'OceanStorS5100']
    deCompressDestDir = fileHandle.deCompressPackage(localFile, deCompressCmds)
    log.info("Decompress datacollect file directory is: " + deCompressDestDir)
    
    #获取结果返回
    if not (os.path.exists(deCompressDestDir) and os.path.isdir(deCompressDestDir)):
        flag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"获取一键收集日志结果文件失败。"
        else:
            errMsg = "Getting the datacollect result file failed."
    return (flag, dataCollectRet, errMsg, deCompressDestDir)


def getCacheInfoStrByDatacollect(dataDict):
    '''
    @summary: 通过一键收集日志获取Cache信息
    @param dataDict: the context object provided by tool framework
    @return: flag, cacheInfoStr, errMsg
    '''
    ssh = getSshObj(dataDict)
    fileHandle = getPackMgr(dataDict)
    log = getLogger(dataDict)
    lang = getLang(dataDict)

    flag = CheckedResult.PASS
    dataCollectRet = ""
    errMsg = ''
    cacheInfoStr = ""
    deCompressDestDir = ""
    
    #获取一键收集日志解压文件夹
    flag, dataCollectRet, errMsg, deCompressDestDir = getDataCollectDecompressDir(dataDict)
    cacheInfoStr += "\n" + dataCollectRet
    if flag != CheckedResult.PASS:
        return (flag, cacheInfoStr, errMsg)

    #查看Config.txt文件中的Cache信息
    configFile = getConfigFileName(deCompressDestDir)
    if not configFile:
        flag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"获取配置文件失败。"
        else:
            errMsg = "Getting configure file failed."
        return (flag, cacheInfoStr, errMsg)
    
    fileHandle = open(configFile)
    fileInfoStr = fileHandle.read()
    fileHandle.close()
    configInfo = fileInfoStr.splitlines()
    
    infoStart = False
    cacheInfoStr += "\n" + "[Configure file cache information context]:"
    for line in configInfo:
        #查看Cache信息起始位置
        if "Cache Info---" in line:
            infoStart = True
        elif "-----------" in line:
            infoStart = False
        if infoStart:
            cacheInfoStr += "\n" + line
    
    #获取信息无效
    if "Controller Id:" not in cacheInfoStr:
        flag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"获取Cache信息失败。"
        else:
            errMsg = "Getting cache information failed."
    
    #获取信息返回
    return (flag, cacheInfoStr, errMsg)


def getConfigFileName(deCompressDestDir):
    '''
    @summary: 获取配置文件的路径
    @param deCompressDestDir: the destination dir of compress
    @return: configFile
    '''
    configFile = ""
    
    #文件夹名称匹配
    for fileName in os.listdir(deCompressDestDir):
        if not fileName.startswith("DebugLog"):
            subPath = (fileName + os.sep) * 2 + "Config" + os.sep + "Config.txt"
            configFile = deCompressDestDir + os.sep + subPath
            break
    
    #确认文件是否存在
    if not configFile:
        if os.path.exists(configFile) and os.path.isfile(configFile):
            return configFile
        else:
            return ""
    
    #返回信息
    return configFile


def deleteCollectTmpDir(dataDict):
    '''
    @summary: 清除一键收集日志临时解压文件夹
    @param dataDict: the context object provided by tool framework
    @return: None
    '''
    log = getLogger(dataDict)
    #获取临时目录
    dataCollectTmpDir = getCollectTmpDir(dataDict)
    
    #查看是否此次检查已经保存了一键收集日志
    try:
        shutil.rmtree(dataCollectTmpDir, ignore_errors = True)
        log.info("Delete temp alarm file succeed.")
    except:
        log.error("[deleteCollectTmpDir]except trace back:" + unicode(traceback.format_exc()))
    
    return


def getCollectTmpDir(dataDict):
    '''
    @summary: 获取一键收集日志临时解压文件夹
    @param dataDict: the context object provided by tool framework
    @return: dataCollectTmpDir
    '''
    log = getLogger(dataDict)
    
    #获取设备唯一标志
    timeStamp = getTimeStamp(dataDict)
    
    dataCollectTmpDir = getTmpDir(dataDict) + os.sep + timeStamp
    
    return dataCollectTmpDir


def deleteRemoteFile(dataDict, collectRemotePath):
    '''
    @summary: 清理设备临时文件
    @param dataDict: the context object provided by tool framework
    @param collectRemotePath: the  path of remote array which is collected
    @return: True, False
    '''
    try:
        sftp = getSftpObj(dataDict)
        logger = getLogger(dataDict)
        #使用sftp自带接口删除远端临时文件
        if not collectRemotePath:
            logger.error("[deleteRemoteFile] delete file is:" + unicode(collectRemotePath))
            return False
        else:
            sftp.deleteFile(collectRemotePath)
            return True
    except:
        logger.error("[deleteRemoteFile] except trace back:" + unicode(traceback.format_exc()))
        return False


def deleteRemoteFiles(dataDict, collectRemotePath, collectRemoteFile):
    '''
    @summary: 清理设备collectRemotePath 目录下的 collectRemoteFile 文件
    @param dataDict: the context object provided by tool framework
    @param collectRemotePath: the  path of remote array which is collected
    @param collectRemoteFile: 文件名支持正则表达式
    @return: True, False
    '''
    try:
        sftp = getSftpObj(dataDict)
        logger = getLogger(dataDict)

        logger.info("[deleteRemoteFiles] begin to delete %s in dir %s" % (unicode(collectRemoteFile), unicode(collectRemotePath)))
        #使用sftp自带接口删除远端文件
        if not collectRemotePath or not collectRemoteFile:
            return False

        lst = sftp.listFiles(collectRemotePath)
        for tmpfile in lst:
            if re.search(collectRemoteFile, tmpfile):
                try:
                    sftp.deleteFile(collectRemotePath + tmpfile)
                    logger.info("[deleteRemoteFiles] delete remote file : " + unicode(tmpfile))
                except:
                    logger.error("[deleteRemoteFiles] delete remote file failed :" + unicode(traceback.format_exc()))
        return True
    except:
        logger.error("[deleteRemoteFiles] except trace back:" + unicode(traceback.format_exc()))
        return False