# -*- coding: UTF-8 -*-

import os
import traceback
from common.sysInfoManager import getCurDeviceType
from common.constant import CheckedResult, DeviceType
from common.cmdRetManager import getCliRet, checkCliInfoValid
from common.contextUtil import getLang, getLogger, getCurSysVer, getRet4Datacollect,getMainOrSlaveAndCtrlIdMap


def getMemInfoFileName(deCompressDestDir, fileName):
    '''
    @summary: 通过控制器路径名称，获取一键收集日志中的指定meminfo文件路径
    @param deCompressDestDir: 已解压的日志目录
    @param fileName: IP开头的文件夹名称
    @return: meminfo文件的绝对路径
    '''
    #文件夹名称匹配
    ioStatFilePath = os.sep.join([deCompressDestDir, fileName, 'msg_other', 'msg_other', 'Other', 'meminfo'])
    
    #返回信息
    return ioStatFilePath


# **************************************************************************** #
# 函数名称: checkFreeMemByFile
# 功能说明: 解析meminfo文件，查看空闲内存
# 输入参数: deCompressDestDir, fileName, context
# 输出参数: flag
# **************************************************************************** # 
def checkFreeMemByFile(deCompressDestDir, fileName, context):
    
    flag = CheckedResult.PASS
    errMsg = ""
    checkStr = ""
    lang = getLang(context)
    log = getLogger(context)
    
    freeMemStand = 220
    
    #获取控制器ID信息
    ctrlId = ''
    if -1 != fileName.find('_MAIN'):#主控
        ctrlId = getMainOrSlaveAndCtrlIdMap(context).get('Primary', "Primary")
    else:
        ctrlId = getMainOrSlaveAndCtrlIdMap(context).get('Secondary', "Secondary")

    #通过控制器Ip查找meminfo文件位置
    memInfoFilePath = getMemInfoFileName(deCompressDestDir, fileName)
    log.info("Getting memory information file path of controller( ID: "  + unicode(ctrlId) + ") is: "+ unicode(memInfoFilePath))
    
    #判断文件是否存在
    if not os.path.exists(memInfoFilePath):
        flag = CheckedResult.WARN
        if lang == "zh":
            errMsg = u"控制器%s：无法找到文件meminfo，可能原因是日志收集失败或日志解压失败。" % unicode(ctrlId)
        else:
            errMsg = "Controller %s: Failed to find the meminfo file. The possible cause is log collection or decompression failure." % unicode(ctrlId)  
        return (flag, checkStr, errMsg)

    #解析mml信息文件
    file = open(memInfoFilePath)
    memInfo = file.read()
    lineList = memInfo.splitlines()
    file.close()

    checkStr += "[Controller " + unicode(ctrlId) + "'s memory information]:\n" + unicode(memInfo)
    
    #获取空闲内存信息
    freeMemValue = -1
    for line in lineList:
        if line.startswith("MemFree:"):
            field = line.split()
            if len(field) == 3:
                if field[1].strip().isdigit():
                    #计算空闲内存（单位：MB）
                    freeMemValue = int(field[1].strip()) / 1024
                    break
            log.error("Free memory information failed:" + unicode(line))
            break
    
    log.info("Check Free memory standard:" + unicode(freeMemStand))
    log.info("Current Free memory standard:" + unicode(freeMemValue))
    #空闲内存（单位MB）
    if freeMemValue == -1:
        flag = CheckedResult.WARN
        if lang == "zh":
            errMsg = u"控制器" + ctrlId + u"：获取内存信息失败。"
        else:
            errMsg = "Controller " + ctrlId + ": Getting memory information failed."
        return (flag, checkStr, errMsg)
    #小于内存阈值
    if freeMemValue < freeMemStand:
        flag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"控制器" + ctrlId + u"：空闲内存不足：空闲内存为" + unicode(freeMemValue) + u"MB"
        else:
            errMsg = "Controller " + ctrlId + ": The free memory is not enough: " + unicode(freeMemValue) + "MB"
    
    return (flag, checkStr, errMsg)


def execute(dataDict):
    '''
    @summary: the entrance of main method, this check item is used to check free memory
    @param dataDict: the dictionary of data which provided by tool framework
    @return: (pass status, CLI information, error message) as (boolean, string, string)
    '''
    
    log = getLogger(dataDict)
    lang = getLang(dataDict)

    flag = CheckedResult.PASS
    cliRet = ''
    errMsg = ''
    
    #获取设备类型
    iRet = getCurDeviceType(dataDict)
    cliRet = iRet[0]
    deviceType = iRet[1]
    if deviceType == DeviceType.UNKNOWN:
        flag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"获取设备类型失败。"
        else:
            errMsg = "Getting device type failed."
        return (flag, cliRet, errMsg)   
        
    #获取一键收集日志解压文件夹
    flag4Datacollect, dataCollectRet, errMsg4Datacollect, deCompressDestDir, isDoubleCtrlLostOne = getRet4Datacollect(dataDict)
    cliRet += "\n" + dataCollectRet
    #解压路径为空或者不存在说明一定收集不成功(包括解压失败)；双控只收集到一个控制器的日志说明部分收集成功
    if not (deCompressDestDir and os.path.exists(deCompressDestDir)) or isDoubleCtrlLostOne:
        return (flag4Datacollect, cliRet, errMsg4Datacollect)

    #查看所有控制器的iostat_1_10文件，确认io利用率
    #文件夹名称匹配
    for fileName in os.listdir(deCompressDestDir):
        if not fileName.startswith("DebugLog"):
            #解析meminfo文件，查看空闲内存
            iRet = checkFreeMemByFile(deCompressDestDir, fileName, dataDict)
            #判断回文信息
            checkFlag = iRet[0]
            cliRet += "\n" + iRet[1]
            if checkFlag != CheckedResult.PASS:
                flag = checkFlag
                errMsg += "\n" + iRet[2]
    
    #检查结果返回
    return (flag, cliRet, errMsg.strip()) 
