# -*- coding: UTF-8 -*-

from common.constant import CheckedResult, DeviceType
from common.cmdRetManager import getCliRet, checkCliInfoValid
from common.contextUtil import getLang, getSshObj, getCliRet4UpgradePkgInfo, getCurSysVer, getDevType,getLogger
from common.modelManager import changeCli2Developer
from common.cTV1R1 import cHandleCliRet
from common.sysInfoManager import switchDeviceType

class CommandType():
    '''
    @summary: Cli命令的类型
    '''
    CONTROLLER = 1
    BMC = 2
    BBU = 3
    SES = 4
    SAS = 5

def isCliInfoValidByClitype(cliInfo, lang, cliType):
    '''
    @summary: 验证CLI命令是否有效
    @param cliInfo: CLI命令回文
    @param lang: 系统语言环境
    @param cliType: 命令类型
    @return: Boolean，errMsg
    '''
    flag = True
    errMsg = ''
    if not checkCliInfoValid(cliInfo):
        flag = False
        if cliType == CommandType.CONTROLLER:    
            if lang == "zh":
                errMsg = u"\n查询控制器版本信息无效。"
            else:
                errMsg = "\nThe Controller version is invalid." 
        elif cliType == CommandType.BMC:    
            if lang == "zh":
                errMsg = u"\n查询BMC版本信息无效。"
            else:
                errMsg = "\nThe BMC version is invalid." 
        elif cliType == CommandType.BBU:    
            if lang == "zh":
                errMsg = u"\n查询BBU版本信息无效。"
            else:
                errMsg = "\nThe BBU version is invalid." 
        elif cliType == CommandType.SES:    
            if lang == "zh":
                errMsg = u"\n查询SES版本信息无效。"
            else:
                errMsg = "\nThe SES version is invalid." 
        else:
            if lang == "zh":
                errMsg = u"\n查询SAS版本信息无效。"
            else:
                errMsg = "\nThe SAS version is invalid." 
    return (flag, errMsg)

def isBMCVerNeedCheck(dataDict):
    '''
    @summary: 验证当前设备型号和版本是否需要验证BMC版本的一致性
    @param dataDict: the dictionary of data which provided by tool framework
    @return: BMC版本需要验证返回True，不需要验证返回False
    '''
    currentDevType = getDevType(dataDict)
    #Dorado 5100 系列型号所有版本需要检查, 其他版本默认通过
    if currentDevType == "Dorado5100":
        return True
    return False

def checkFWVer(cliRet, keyword):
    '''
    @summary: 根据回文检查关键字keyword对应的固件版本信息是否一致
    @param cliRet: 命令回文
    @param keyword: 固件版本对应的关键字
    @return: 固件版本一致返回True，其他返回False
    '''
    formatFunction = cHandleCliRet(cliRet)
    cliInfoList = formatFunction.handle()
    versionList = []
    for cliInfo in cliInfoList:
        cliVersion = cliInfo.get(keyword)
        if cliVersion and len(cliVersion) > 0:
            versionList.append(cliVersion)
        else:
            return False
    versionList = list(set(versionList))
    if len(versionList) <= 1:
        return True
    return False
    
def checkFWVerForCtrl(cliRet, keyword, logger):
    '''
    @summary: 根据回文检查关键字keyword对应的固件版本信息是否一致，控制器需要考虑两种情况：导入包的情况和Version情况
    @param cliRet: 命令回文
    @param keyword: 固件版本对应的关键字
    @return: 固件版本一致返回True，其他返回False
    '''
    keyForStatus = "Package Status"
    
    #正常版本：携带Package Status字段
    if keyForStatus in cliRet:
        formatFunction = cHandleCliRet(cliRet)
        cliInfoList = formatFunction.handle()
        versionList = []
        for cliInfo in cliInfoList:
            cliVersion = cliInfo.get(keyword)
            if cliVersion and len(cliVersion) > 0:
                if "Running" == cliInfo.get(keyForStatus):
                    versionList.append(cliVersion)
            else:
                return False
        versionList = list(set(versionList))
        logger.info("VersionList in checkFWVerForCtrl is:" + unicode(versionList))
        if len(versionList) <= 1:
            return True
    else:
        #特殊版本V1R1C00SPC013没有Package Status
        cliList = cliRet.splitlines()
        versionForA = ""
        versionForB = ""
        for field in cliList[6:]:
            fieldList = field.split()
            if len(fieldList) < 2:
                continue
            if fieldList[0] == "A" and not versionForA:
                versionForA = fieldList[1]
            if fieldList[0] == "B" and not versionForB:
                versionForB = fieldList[1]
        
        logger.info("CheckFWVerForCtrl versionForA:%s, versionForB:%s." % (unicode(versionForA),unicode(versionForB)))
        if versionForA == versionForB:
            return True
        
    return False
    
def checkPartFWVer(cliRet, keyword, partitionKey):
    '''
    @summary: 根据回文检查各个partitionKey组内，关键字keyword对应的固件版本信息是否一致
    @param cliRet: 命令回文
    @param keyword: 固件版本对应的关键字
    @return: 固件版本一致返回True，其他返回False
    '''
    formatFunction = cHandleCliRet(cliRet)
    cliInfoList = formatFunction.handle()
    partionIds = list(set([cliInfo.get(partitionKey) for cliInfo in cliInfoList]))
    if None in partionIds:
        return False

    flag = True
    for partitionId in partionIds:
        versionList = []
        for cliInfo in cliInfoList:
            cliVersion = cliInfo.get(keyword)
            if cliInfo.get(partitionKey) == partitionId:
                if cliVersion and len(cliVersion) > 0:
                    versionList.append(cliVersion)
                else:
                    flag = False
        versionList = list(set(versionList))
        if len(versionList) > 1:
            flag = False
         
    return flag

    
def execute(dataDict):
    '''
    @summary: the entrance of main method, this check item is used to check LUN Status (Online)
    @param dataDict: the dictionary of data which provided by tool framework
    @return: (pass status, CLI information, error message) as (boolean, string, string)
    '''
    
    flag = CheckedResult.PASS
    lang = getLang(dataDict)
    ssh = getSshObj(dataDict)
    errMsg = ""
    allRet = ""
    logger = getLogger(dataDict)
    
    #步骤1:查看控制器版本(Package Version）
    cliRet = getCliRet4UpgradePkgInfo(dataDict)
    allRet += cliRet
    cliValidflag, errMsg1 = isCliInfoValidByClitype(cliRet, lang, CommandType.CONTROLLER)       
    if not cliValidflag:
        flag = CheckedResult.NOTPASS 
        errMsg += errMsg1 
    else:
        if not checkFWVerForCtrl(cliRet, "Package Version", logger) and not checkFWVerForCtrl(cliRet, "Version", logger):
            flag = CheckedResult.NOTPASS
            if lang == "zh":
                errMsg += u"\n控制器版本不一致。"
            else:
                errMsg +=  "\nController versions are inconsistent."
                    
    #步骤2：查看BMC版本(Version)
    if isBMCVerNeedCheck(dataDict):
        cliRet = getCliRet(dataDict, "showbmcver")
        allRet += cliRet
        cliValidflag, errMsg1 = isCliInfoValidByClitype(cliRet, lang, CommandType.BMC)       
        if not cliValidflag:
            flag = CheckedResult.NOTPASS  
            errMsg += errMsg1 
        else:
            #判断BMC版本是否一致
            if not checkFWVer(cliRet, "Version"):
                flag = CheckedResult.NOTPASS
                if lang == "zh":
                    errMsg += u"\nBMC版本不一致。"
                else:
                    errMsg +=  "\nBMC versions are inconsistent."               
             
    #步骤3：切换developer模式（切换失败直接返回）
    if not changeCli2Developer(dataDict):
        if flag == CheckedResult.PASS:
            flag = CheckedResult.WARN
        if errMsg:
            errMsg += '\n'
        if lang == "zh":
            errMsg += u"进入developer模式失败，查询BBU/SES/SAS版本信息失败。失败的原因可能为：\n" \
                    + u"（1）添加设备时未输入developer密码。\n（2）添加设备时输入的developer密码无效。"
        else:
            errMsg += "Login to developer model failed, can not get BBU/SES/SAS version. The reason of failure may be:\n" \
                    + "(1) Did not enter a developer password when adding the device.\n(2) The developer password entered is incorrect."
        return (flag, allRet, errMsg)
    
    #步骤4：查看BBU版本(Firmware Version)
    cliRet = getCliRet(dataDict, "showbbuver")
    allRet += cliRet
    cliValidflag, errMsg1 = isCliInfoValidByClitype(cliRet, lang, CommandType.BBU)       
    if not cliValidflag:
        flag = CheckedResult.NOTPASS  
        errMsg += errMsg1 
    else:
        #判断BBU版本是否一致
        if not checkFWVer(cliRet, "Firmware Version"):
            flag = CheckedResult.NOTPASS
            if lang == "zh":
                errMsg += u"\nBBU版本不一致。"
            else:
                errMsg +=  "\nBBU versions are inconsistent."
    
    #步骤5：查看同一个框的SES版本(Version)
    cliRet = getCliRet(dataDict, "showsesver")
    allRet += cliRet
    cliValidflag, errMsg1 = isCliInfoValidByClitype(cliRet, lang, CommandType.SES)       
    if not cliValidflag:
        flag = CheckedResult.NOTPASS  
        errMsg += errMsg1 
    else:
        #判断SES版本是否一致
        if not checkPartFWVer(cliRet, "Version", "Enclosure Id"):
            flag = CheckedResult.NOTPASS
            if lang == "zh":
                errMsg += u"\nSES版本不一致。"
            else:
                errMsg +=  "\nSES versions are inconsistent."
    
    #步骤6：查看SAS版本(Firmware Version)
    cliRet = getCliRet(dataDict, "showsasfirmwarever")
    allRet += cliRet
    cliValidflag, errMsg1 = isCliInfoValidByClitype(cliRet, lang, CommandType.SAS)       
    if not cliValidflag:
        flag = CheckedResult.NOTPASS  
        errMsg += errMsg1 
    else:
        #判断SAS版本是否一致
        if not checkFWVer(cliRet, "Firmware Version"):
            flag = CheckedResult.NOTPASS
            if lang == "zh":
                errMsg += u"\nSAS版本不一致。"
            else:
                errMsg +=  "\nSAS versions are inconsistent."
    
    #退出developer模式
    ssh.execCmd("exit") 
    
    return (flag, allRet, errMsg)
