# -*- coding: UTF-8 -*-

from common.constant import CheckedResult
from common.cmdRetManager import getCliRet, checkCliInfoValid
from common.contextUtil import getLang, getLogger
from common.cTV1R1 import cliRetHorizontal

#特定Model值和FW version对照字典
BLACK_EQUALS_DICT = {'HUS156060VLS600':['a510','a760'],
                       'HUS156060VLF400':['ft02','f510'],
                       }
#FW版本在特定范围内的MODEL值对照字典
BLACK_RANGE_MODEL_FW = {
                        'HSSD-D3J2JAM5400':('8701','9105'),
                        'HSSD-D3J2JAM5600':('8701','9105'),
                        'HSSD-D3J2LAM5400':('8701','9105'),
                        'HSSD-D3J2LAM5600':('8701','9105')
                        }

DISK_LOCATION = "Disk Location"
DISK_MODEL = "Model"
DISK_FW_VER = "Fw Version"

def execute(dataDict):
    '''
    @summary: 硬盘固件版本检查
    @param dataDict: the dictionary of data which provided by tool framework
    @return: (pass status, CLI information, error message) as (boolean, string, string)
    '''

    flag = CheckedResult.PASS
    lang = getLang(dataDict)
    logger = getLogger(dataDict)
    
    errMsg = ""
    cliRet = ""
    DiskLocationList = []#检测到不通过硬盘的位置信息
    errInfoDiskList = []#获取Model和FW失败的硬盘

    cliRet = getCliRet(dataDict, "showdisk -physic", False)
    printCliRet(cliRet, 20, logger)
    list = cliRet.splitlines()
    if len(list) < 7:
        #判断cli信息是否有效
        if not checkCliInfoValid(cliRet):
            flag = CheckedResult.NOTPASS
            if lang == "zh":
                errMsg = u"\nCli信息无效。"
            else:
                errMsg = "\nCli information invalid." 
        return (flag, cliRet, errMsg)
    
    #转成字典列表
    cliHorizontal = cliRetHorizontal(cliRet)
    cliRetLinesList = cliHorizontal.handle()
       
    for cliRetLine in cliRetLinesList:
        diskLocation = cliRetLine.get(DISK_LOCATION,"")
        model = cliRetLine.get(DISK_MODEL,"")
        FWVersion = cliRetLine.get(DISK_FW_VER,"")
        if model == "" or model == "--" or FWVersion == "" or FWVersion == "--":
            errInfoDiskList.append(diskLocation)
        if isDiskEqualsModelAndVer(cliRetLine) or isDiskInSpecRange(cliRetLine):
            DiskLocationList.append(diskLocation)
                
    if DiskLocationList:
        flag = CheckedResult.WARN
        #对硬盘列表排序
        DiskLocationList.sort()
        locationStr = ", ".join(DiskLocationList)
        message = {"zh":u"\n固件版本过低的硬盘位置如下：\n%s",
                   "en":"\nPositions of disks whose firmware versions are too early are as follows: \n%s",
                   }
        errMsg = message.get(lang) % locationStr
    
    if errInfoDiskList:
        flag = CheckedResult.NOTPASS
        #对硬盘列表排序
        errInfoDiskList.sort()
        locationStr = ", ".join(errInfoDiskList)
        message = {"zh":u"\n无法获取固件版本和Model的硬盘如下：\n%s",
                   "en":"\nDisks whose firmware version and Model cannot be obtained are as follows: \n%s",
                   }
        errMsg += message.get(lang) % locationStr

    return (flag, cliRet, errMsg)
    
def isDiskInSpecRange(cliRetDict):
    Model = cliRetDict.get(DISK_MODEL)
    fwVersion = cliRetDict.get(DISK_FW_VER).lower()
    blackFWRange = BLACK_RANGE_MODEL_FW.get(Model,None)
    if not blackFWRange:
        return False
    if fwVersion>=blackFWRange[0] and fwVersion <= blackFWRange[1]:
        return True
    return False

def isDiskEqualsModelAndVer(cliRetDict):
    Model = cliRetDict.get(DISK_MODEL)
    fwVersion = cliRetDict.get(DISK_FW_VER).lower()
    
    blackFWList = BLACK_EQUALS_DICT.get(Model,None)
    if not blackFWList:
        return False
    if fwVersion in blackFWList:
        return True
    return False

def printCliRet(cliRet, num, logger):
    '''
    @summary: 以num作为规格划分打印日志
    '''
    lines = cliRet.splitlines()
    length = len(lines)
    for i in range(1, length):
        mul = i%num
        if mul == 0:
            logger.info("\n%s" % "\n".join(lines[i-num:i]))
            continue
        if i == length-1:
            logger.info("\n%s" % "\n".join(lines[i-mul:length]))
    return
