# -*- coding: UTF-8 -*-

import re
import string
from common.utils import safeSleep
from common.constant import CheckedResult
from common.cmdRetManager import getCliRet, checkCliInfoValid
from common.contextUtil import getLang, getLogger, getSshObj
from common.modelManager import *

# **************************************************************************** #
# 函数名称: getPeerHeartBeatIP
# 功能说明: 解析数据，获取远端控制器的心跳IP
# 输入参数: 
# 输出参数:
# 返 回 值: 
# **************************************************************************** # 
def getPeerHeartBeatIP(heartBeatIPMsg):
    rowList = string.split(heartBeatIPMsg, '\n')
    rowCount = len(rowList)
    for iter in range(rowCount):
        localHeartBeatIP = ''
        if bool(re.search('inet addr:', rowList[iter])):
            columnList = string.split(rowList[iter])
            localHeartBeatIP = string.split(columnList[1], ':')[1]
        if '127.127.127.10' == localHeartBeatIP:
            return '127.127.127.11'
        elif '127.127.127.11' == localHeartBeatIP:
            return '127.127.127.10'
    return ''

# **************************************************************************** #
# 函数名称: LinkCtrlByHeartBeat
# 功能说明: 通过心跳来切换命令发送的控制器
# 输入参数: dataDict
# 输出参数: NA
# 返 回 值: True,False
# **************************************************************************** # 
def LinkCtrlByHeartBeat(ssh):
    #执行此操作前确保切换到Storage命令模式
    heartBeatIPMsgOld = ssh.execCmd('ifconfig bond0')
    peerHeartBeatIPOld = getPeerHeartBeatIP(heartBeatIPMsgOld)
    ssh.execCmd('ssh ibc_os_hs@' + peerHeartBeatIPOld)
    heartBeatIPMsgNew = ssh.execCmd('ifconfig bond0')
    peerHeartBeatIPNew = getPeerHeartBeatIP(heartBeatIPMsgNew)
    if bool(re.search('127.127.127', peerHeartBeatIPOld)) \
    and bool(re.search('127.127.127', peerHeartBeatIPNew)) \
    and (peerHeartBeatIPOld != peerHeartBeatIPNew):
        return True
    else:
        return False


# **************************************************************************** #
# 函数名称: getTGTIoInfo
# 功能说明: 获取TGT的io数据
# 输入参数: tgtIoInfo
# 输出参数: NA
# 返 回 值: Start和not back的Io数据
# 其 他   :  函数传入时ssh为cli命令模式，返回时为debug模式
# **************************************************************************** # 
def getTGTIoInfo(tgtIoInfo):
    startIoData = ""
    notbackIoData = ""
    
    field = tgtIoInfo.split()
    for temp in field:
        if temp.startswith("Start:"):
            startIoData = temp.replace("Start:", "").replace(",", "")
        elif temp.startswith("back:"):
            notbackIoData = temp.replace("back:", "")
        else:
            pass
        if startIoData != "" and notbackIoData != "":
            break
    
    return (startIoData, notbackIoData)

# **************************************************************************** #
# 函数名称: getCurCtrlTGTStartIoInfo
# 功能说明: 获取当前控制器的io数据
# 输入参数: tgtIoInfo
# 输出参数: NA
# 返 回 值: Start和not back的Io数据
# 其 他   :  函数传入时ssh为cli命令模式，返回时为debug模式
# **************************************************************************** # 
def getCurCtrlTGTStartIoInfo(ddioInfo, lang):
    
    flag = True
    errMsg = ""
    tgtIoInfo = ()
    
    readIoTgtStart = ""
    readIoTgtNotback = ""
    writeIoTgtStart = ""
    writeIoTgtNotback = ""

    writeIoDataFlag = False
    
    #判断mml回文是否正确返回
    if not re.search("Read IOs:", ddioInfo, re.IGNORECASE):
        flag = False
        if lang == "zh":
            errMsg = u"mml命令执行异常。"
        else:
            errMsg = "The command execute error."

        return (flag, errMsg, ())
        
    #解析正确返回的命令回文
    lineList = ddioInfo.splitlines()
    for line in lineList:
        #查看是否为Write IOS数据
        if re.search("Write IOs:", line, re.IGNORECASE):
            writeIoDataFlag = True
            continue
        elif line.startswith("TGT"):
            iRet = getTGTIoInfo(line)
            #判断是读IO还是写IO
            if not writeIoDataFlag:
                readIoTgtStart = iRet[0]
                readIoTgtNotback = iRet[1]
            else:
                writeIoTgtStart = iRet[0]
                writeIoTgtNotback = iRet[1]
        else:
            pass
    
        #获取读IO和写IO的数据成功，则退出循环
        if readIoTgtStart != "" and writeIoTgtStart != "":
            break

    #判断所有值是否有效
    tgtIoInfo = (readIoTgtStart, readIoTgtNotback, writeIoTgtStart, writeIoTgtNotback)
    
    #如果其中任意一项为空，则报错
    for temp in tgtIoInfo:
        if temp == "":
            flag = False
            if lang == "zh":
                errMsg = u"获取TGT的io数据失败。"
            else:
                errMsg = "Getting io data of TGT failed."
            return (flag, errMsg, ())

    #如果not back的IO数据不为0，则检查不通过
    if readIoTgtNotback != "0" or writeIoTgtNotback != "0":
        flag = False
        if lang == "zh":
            errMsg = u"获取读写IO的not back值不为0，存在未停止的业务。"
        else:
            errMsg = "The not back value of the read IO or write IO is not 0, some services are running."
        return (flag, errMsg, ())
    
    #正常情况下，notback数据已经检查，只需要返回start IO信息
    return (flag, errMsg, (readIoTgtStart, writeIoTgtStart))
    
# **************************************************************************** #
# 函数名称: checkCurCtrlServiceStop
# 功能说明: 查询当前控制器的业务是否完全停止
# 输入参数: ssh,lang
# 输出参数: NA
# 返 回 值: True,False
# 其 他   :  在MML模式下执行处理
# **************************************************************************** # 
def checkCurCtrlServiceStop(ssh, lang, log):
    #进入此函数执行前，需保证ssh在mml命令模式下
    allRet = ''
    checkFlag = CheckedResult.PASS
    errMsg = ''

    tgtioInfo = ()
    tgtioInfoTmp = ()
    
    #查询IO信息
    mmlRet1 = ssh.execCmd("dd io")
    allRet += mmlRet1 + '\n'

    #获取一次TGT的io信息
    iRet0 = getCurCtrlTGTStartIoInfo(mmlRet1, lang)
    if not iRet0[0]:
        checkFlag = CheckedResult.NOTPASS
        errMsg = iRet0[1]
        return (checkFlag, allRet, errMsg)
    
    #获取tgt的io信息
    tgtioInfo = iRet0[2]

    #睡眠1分钟后再发送ddio命令
    safeSleep(60)
    mmlRet2 = ssh.execCmd("dd io")
    allRet += mmlRet2 + '\n'

    #获取一分钟后的TGT的io信息
    iRet1 = getCurCtrlTGTStartIoInfo(mmlRet2, lang)
    if not iRet1[0]:
        checkFlag = CheckedResult.NOTPASS
        errMsg = iRet1[1]
        return (checkFlag, allRet, errMsg)

    #获取tgt的io信息
    tgtioInfoTmp = iRet1[2]
    
    #判断当前控制器业务是否完全停止
    if tgtioInfo != tgtioInfoTmp:
        checkFlag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"两次查询的IO数据不相同，存在未停止的业务。"
        else:
            errMsg = "The IO data is different between two queries, some services are running."
        return (checkFlag, allRet, errMsg)
    
    #检查通过，返回
    return (checkFlag, allRet, errMsg)


def execute(dataDict):
    '''
    @summary: the entrance of main method, this check item is used to check Service Stopped
    @param dataDict: the dictionary of data which provided by tool framework
    @return: (pass status, CLI information, error message) as (boolean, string, string)
    '''
    allRet = ''
    checkFlag = CheckedResult.PASS
    errMsg = ''
    checkRetTmp1 = None
    checkRetTmp2 = None
    
    ssh = getSshObj(dataDict)
    log = getLogger(dataDict)
    lang = getLang(dataDict)
    
    #切换到MML模式
    iRet = changeCli2Mml(dataDict)
    if True == iRet:
        #在mml模式下：查询当前控制器的业务压力
        checkRetTmp1 = checkCurCtrlServiceStop(ssh, lang, log)
        #退出到debug模式
        ssh.execCmd("exit")       
        #在debug模式下：心跳到对端，查询对端控制器的业务压力
        ret = LinkCtrlByHeartBeat(ssh)
        if True == ret:
            #切换到mml模式
            ssh.execCmd("mml")
            checkRetTmp2 = checkCurCtrlServiceStop(ssh, lang, log)
            #退出到debug模式
            ssh.execCmd("exit")
            #退出心跳
            ssh.execCmd("exit")
        else:
            pass
            
        #从debug模式退回到cli模式，执行两次exit
        ssh.execCmd("exit")
        ssh.execCmd("exit")
        
        #确认本端检查结果
        if checkRetTmp1 != None:
            if lang == 'zh':
                allRet += u"本端控制器命令结果：\n" + checkRetTmp1[1]
            else:
                allRet += u"Local Controller command results：\n" + checkRetTmp1[1]

            if checkRetTmp1[0] == CheckedResult.PASS:
                pass
            else:
                checkFlag = CheckedResult.NOTPASS
                if lang == 'zh':
                    errMsg += u"本端控制器检查结果：\n" + checkRetTmp1[2]
                else:
                    errMsg += u"Local Controller check results：\n" + checkRetTmp1[2]
        #确认对端检查结果
        if checkRetTmp2 != None:
            if lang == 'zh':
                allRet += u"\n\n对端控制器命令结果：\n" + checkRetTmp2[1]
            else:
                allRet += u"\n\nPeer Controller command results：\n" + checkRetTmp2[1]

            if checkRetTmp2[0] == CheckedResult.PASS:
                pass
            else:
                checkFlag = CheckedResult.NOTPASS
                if lang == 'zh':
                    errMsg += u"\n\n对端控制器检查结果：\n" + checkRetTmp2[2]
                else:
                    errMsg += u"\n\nPeer Controller check results：\n" + checkRetTmp2[2]     
        
    else:
        checkFlag = CheckedResult.WARN
        # 设备登陆时输入developer密码无效 modified 20131130 Begin
        if lang == "zh":
            errMsg = u"进入MML模式失败，业务停止检查失败。失败的原因可能为：\n" \
                    + u"（1）添加设备时未输入developer密码。\n（2）添加设备时输入的developer密码无效。"
        else:
            errMsg = "Login to MML model failed, can not execute service stopped check.The reason of failure may be:\n" \
                    + "(1) Did not enter a developer password when adding the device.\n(2) The developer password entered is incorrect."
        # 设备登陆时输入developer密码无效 modified 20131130 End
        
    return (checkFlag, allRet, errMsg)
