# -*- coding: UTF-8 -*-
from regex import Regex
import config
import logger
import os
import resource
import time
import re
from cbb.frame.util.common import wrapAllExceptionLogged

IPV4 = "v4"
IPV6 = "v6"
#IPv4正则表达式定义
IPV4_REGEX_DEFINE = "^((25[0-5]|2[0-4]\d|[0-1]?\d\d?)\.){3}(25[0-5]|2[0-4]\d|[0-1]?\d\d?)$"
#IPv6正则表达式定义
IPV6_REGEX_DEFINE = "^\s*((([0-9A-Fa-f]{1,4}:){7}(([0-9A-Fa-f]{1,4})|:))|(([0-9A-Fa-f]{1,4}:){6}(:|((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})|(:[0-9A-Fa-f]{1,4})))|(([0-9A-Fa-f]{1,4}:){5}((:((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|(([0-9A-Fa-f]{1,4}:){4}(:[0-9A-Fa-f]{1,4}){0,1}((:((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|(([0-9A-Fa-f]{1,4}:){3}(:[0-9A-Fa-f]{1,4}){0,2}((:((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|(([0-9A-Fa-f]{1,4}:){2}(:[0-9A-Fa-f]{1,4}){0,3}((:((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|(([0-9A-Fa-f]{1,4}:)(:[0-9A-Fa-f]{1,4}){0,4}((:((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|(:(:[0-9A-Fa-f]{1,4}){0,5}((:((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})?)|((:[0-9A-Fa-f]{1,4}){1,2})))|(((25[0-5]|2[0-4]\d|[01]?\d{1,2})(\.(25[0-5]|2[0-4]\d|[01]?\d{1,2})){3})))(%.+)?\s*$"

def getRes(lang, res, args="", resDict=resource.RESOURCE_DICT):
    '''
    @summary: 资源国际化
    @param lang: 语言lang
    @param res: 资源
    @param args: 资源对应的参数
    @param resDict: 资源字典
    @return: 经过国际化处理后的资源
    '''
    try:
        key = "%s_%s" % (res, lang)
        if not resDict.has_key(key):
            return ("--", "")
        
        context = resDict.get(key)
        if "%s" in context or "%i" in context:
            context = context % args
        
        return context
        
    except:
        return ("--", "")
    
def getMsg(lang, msg, errMsgArgs="", suggestionArgs="", msgDict=resource.MESSAGES_DICT):
    '''
    @summary: 错误消息和修复建议国际化
    @param lang: 语言lang
    @param msg: 消息
    @param errMsgArgs: 错误对应的参数
    @param suggestionArgs: 修复建议对应的参数，有多个参数时，以元组方式传入，单个参数时，以字符串方式传入
    @param msgDict: 消息和修复建议字典，有多个参数时，以元组方式传入，单个参数时，以字符串方式传入
    @return: 经过国际化处理后的消息和修复建议
    '''
    try:
        if not msgDict.has_key(msg):
            return ("--", "")
        
        localeDict = msgDict.get(msg)
        errMsg = localeDict.get("errMsg_%s" % lang, "--")
        suggestion = localeDict.get("suggestion_%s" % lang , "")
        
        if "%s" in errMsg or "%i" in errMsg:
            errMsg = errMsg % errMsgArgs
        if "%s" in suggestion or "%i" in suggestion:
            suggestion = suggestion % suggestionArgs
        
        return (errMsg, suggestion)
        
    except:
        return ("--", "")

def getBaseName(filePath):
    '''
    @summary: 返回文件路径的文件名，不包含后缀
    @param filePath: 文件路径
    @return: 返回不包含后缀的文件名字符串
    '''
    baseName, ext = os.path.splitext(os.path.basename(filePath))
    return baseName

def getLogger(loggerInstance, pyFilePath):
    '''
    @summary: 获取日志类
    @param loggerInstance: logger实例
    @param pyFilePath: py文件路径
    '''
    pyFileName = getBaseName(pyFilePath)
    return logger.Logger(loggerInstance, pyFileName)

def cmpPorts(port1, port2):
    '''
    @summary: 端口字典比较
    @param portDict1: 端口字典1
    @param portDict2: 端口字典2
    @return: 根据location进行比较的结果
    '''
    if port1["location"] < port2["location"]:
        return -1
    elif port1["location"] > port2["location"]:
        return 1
    return 0

def getPortSlot(portInfo):
    '''
    @summary: 根据端口信息，获取该端口的槽位号
    @param portInfo: 端口信息
    @return: 端口的槽位号
    '''
    return portInfo["location"].split(".")[-1]

def getVRCVersion(version):
    '''
    @summary: 获取产品的C版本信息
    @param version: 产品版本
    @return: 产品C版本
    '''
    if len(version) <= 11:
        return version
    else:
        return version[0:11]

def bit2GB(bit):
    '''
    @summary: 比特容量转换为GB容量
    '''
    return bit * 1.0 / 1024 / 1024 /1024

def isPureDigit(digitStr):
    '''
    @summary: 判断字符串是否为数字（如果字符串以0开头，不认为是数字）
    '''
    if not digitStr.isdigit():
        return False
    
    if len(digitStr) != len(str(int(digitStr))):
        return False
    
    return True
    
def getIpVer(accessIP):
    '''
    @summary: 获取IP地址版本号
    @param baseIpAddr: IP
    @return: 
        v4: IPv4地址
        v6: IPv6地址
    '''
    if ":" in accessIP:
        return IPV6
    return IPV4

def isIllegalManIpAddr(manIpAddr, ipVer):
    '''
    @summary: 校验管理IP地址是否合法（要求为合法的主机IP地址）
    @param manIpAddr: 管理IP
    @return: 
        True: 管理IP合法
        False: 管理IP不合法
    '''
    if ipVer is None:
        #IPv4地址
        if not ":" in manIpAddr:
            return isIllegalManIpv4Addr(manIpAddr)
        #考虑IPv6地址有兼容IPv4的情况
        return isIllegalManIpv4Addr(manIpAddr) or isIllegalManIpv6Addr(manIpAddr)
    
    if ipVer == IPV4:
        return isIllegalManIpv4Addr(manIpAddr)
    
    if ipVer == IPV6:
        return isIllegalManIpv6Addr(manIpAddr)
    
    return False

def isIllegalManIpv4Addr(manIpAddr):
    '''
    @summary: 校验管理IP地址是否为合法的IPv4地址
    @param manIpAddr: 管理IP
    @return: 
        True: 合法的IPv4地址
        False: 不合法的IPv4地址
    '''
    if not isIpV4(manIpAddr):
        return False
    
    addrs = manIpAddr.split(".")
        
    if int(addrs[0]) == 0:
        return False
    
    return True

def isIllegalManIpv6Addr(manIpAddr):
    '''
    @summary: 校验管理IP地址是否为合法的IPv6地址
    @param manIpAddr: 管理IP
    @return: 
        True: 合法的IPv6地址
        False: 不合法的IPv6地址
    '''
    return isIpV6(manIpAddr)
    
def isIllegalMask(mask, ipVer):
    '''
    @summary: 校验IPV4子网掩码/IPV6前缀是否合法
    @param mask: 子网掩码/前缀
    @return: 
        True: 子网掩码/前缀合法
        False: 子网掩码/前缀不合法
    '''
    if ipVer is None:
        return isIllegalSubnetMask(mask) or isIllegalPrefix(mask)
    
    if ipVer == IPV4:
        return isIllegalSubnetMask(mask)
    
    if ipVer == IPV6:
        return isIllegalPrefix(mask)
    
    return False

def isIllegalSubnetMask(mask):
    '''
    @summary: 校验IPV4子网掩码是否合法
    @param mask: IPV4子网掩码
    @return: 
        True: IPV4子网掩码合法
        False: IPV4子网掩码不合法
    '''
    if not isIpV4(mask):
        return False
    
    addrs = mask.split(".")
        
    if int(addrs[0]) == 0:
        return False
    
    return True

def isIllegalPrefix(mask):
    '''
    @summary: 校验IPV6前缀是否合法
    @param mask: IPV6前缀
    @return: 
        True: IPV6前缀合法
        False: IPV6前缀不合法
    '''
    if not isPureDigit(mask):
        return False
    
    if 0 <= int(mask) <= 128:
        return True
    else:
        return False

def isIllegalGateway(gateway, ipVer):
    '''
    @summary: 校验IPV4网关地址/IPV6网关地址是否合法
    @param mask: 网关地址
    @return: 
        True: 网关地址
        False: 网关地址
    '''
    if ipVer is None:
        return isIllegalIPv4Gateway(gateway) or isIllegalIPv6Gateway(gateway)
    
    if ipVer == IPV4:
        return isIllegalIPv4Gateway(gateway)
    
    if ipVer == IPV6:
        return isIllegalIPv6Gateway(gateway)
    
    return False

def isIllegalIPv4Gateway(gateway):
    '''
    @summary: 校验IPv4网关地址是否合法
    @param mask: IPv4网关地址
    @return: 
        True: IPv4网关地址合法
        False: IPv4网关地址不合法
    '''
    return isIpV4(gateway)

def isIllegalIPv6Gateway(gateway):
    '''
    @summary: 校验IPv6网关地址是否合法
    @param mask: IPv6网关地址
    @return: 
        True: IPv6网关地址合法
        False: IPv6网关地址不合法
    '''
    return isIpV6(gateway)


#---------New Frame---------
def isIpV4(ip):
    '''
    @summary: 检测IP地址是否为通用的IPv4地址
    '''
    return Regex.find(IPV4_REGEX_DEFINE, ip)
    
def isIpV6(ip):
    '''
    @summary: 检测IP地址是否为通用的IPv6地址
    '''
    return Regex.find(IPV6_REGEX_DEFINE, ip)

def getBaseUri(ip, devSN, port=config.REST_DEFAULT_PORT):
    '''
    @summary: 获取rest的基础链接（https://${ip}:${port}/deviceManager/rest/${deviceId}/）
    @param ip: 设备ip
    @param devSN: 设备SN
    @param port: rest使用的端口
    @return: rest及基础链接
    '''   
    return config.REST_BASE_URI % (ip, port, devSN)


def isHighEndDev(productModel):
    '''
    @summary: 判读是否为高端设备
    @param productModel: 产品型号
    @return: 
        True: 产品型号为高端
        False: 产品型号不为高端
    '''
    return productModel in config.HIGH_END_DEVS

def isArmDev(productModel):
    '''
    @summary: 判读是否为ARM设备
    @param productModel: 产品型号
    @return: 
        True: 产品型号为ARM设备
        False: 产品型号不为ARM设备
    '''
    return productModel in config.ARM_DEVS

def isArmDev2U(productModel):
    '''
    @summary: 判读是否为2U ARM设备
    @param productModel: 产品型号
    @return: 
        True: 产品型号为2U ARM设备
        False: 产品型号不为2U ARM设备
    '''
    return productModel in config.ARM_DEVS_2U

def isDoradoDev(productModel):
    '''
    @summary: 判读是否为Dorado设备
    @param productModel: 产品型号
    @return: 
        True: 产品型号为Dorado设备
        False: 产品型号不为Dorado设备
    '''
    return productModel in config.DORADO_DEVS

def isDoradoNasDev(productModel):
    return productModel in config.DORADO_NAS_DEVS


@wrapAllExceptionLogged(logger=None)
def safeSleep(seconds):
    '''
    @summary: 睡眠
    @param seconds: 睡眠时间
    '''
    time.sleep(seconds)
