# -*- coding: UTF-8 -*-
import cliResource
import traceback
from frame.common.config import STOR_DEV_INFO_DICT
from frame.common.regex import Regex
from com.huawei.ism.exception import IsmException

CLI_EXECUTE_CMD_SUCCESS = "Command executed successfully"
DEBUG_MODEL_FLAG = ":/diagnose>"
DEVELOPER_MODEL_FLAG = "developer:/>"
CLI_RET_END_FLAG = ":/>"
DIAGNOSE_RET_END_FLAG = "diagnose>"
MINISYSTEM_MODEL_FLAG = "minisystem>"
STORAGE_MODEL_FLAG = "Storage:~ #"

DIAGNOSE_SYS_SHOW_CLS_CMD = 'sys showcls'

RESULT_NOCHECK = "NOCHECK"
RESULT_NOSUPPORT = "NOSUPPORT"
RESULT_WARNING = "WARNING"
RESULT_DICT = {
   True:"0",
   False:"1",
   RESULT_NOCHECK:"2",
   RESULT_NOSUPPORT:"3",
   RESULT_WARNING:"4",
}

class CheckStatus():
    ERROR = 'EEROR'
    
#优先级从高到底
RESULT_LIST = [False, RESULT_WARNING, RESULT_NOCHECK, RESULT_NOSUPPORT, True]

def getMsg(lang, msg, args="", resource=cliResource.MESSAGES_DICT):
    '''
    @summary: 消息国际化
    @param lang: 语言lang
    @param msg: 消息
    @param args: 消息参数
    @param resource: 消息字典
    @return: 经过国际化处理后的消息
    '''
    errMsg = "\n--"
    
    try:
        if not resource.has_key(msg):
            return errMsg
        
        localeDict = resource.get(msg)
        if not localeDict.has_key(lang):
            return errMsg
        
        localeMsg = localeDict.get(lang)
        if "%s" in localeMsg or "%i" in localeMsg:
            return localeMsg % args
        else:
            return localeMsg

    except:
        return errMsg
    
def isChinese(lang):
    '''
    @summary: 判断lang是否为中文
    @param lang: 语言lang
    @return: 
        True: 中文
        False: 非中文
    '''
    if lang == "zh":
        return True
    return False

def isNoneLicense(cliRet):
    '''
    @summary: 判断回显结果里面是否包含需要license
    @param cliRet: cli回显
    @return: 
        True: cli回显包含需要license
        False: cli回显不包含需要license
    '''
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "license" in lowerLine and ("error" in lowerLine or "suggestion" in lowerLine):
            return True
    
    return False

def hasCliExecPrivilege(cliRet):
    '''
    @summary: 判断是否具有执行cli命令的权限
    @param cliRet: cli回显
    @return: 
        True: 具有执行cli命令的权限
        False: 不具有执行cli命令的权限
    '''
    for line in cliRet.splitlines():
        if line.strip() == "^":
            return False
        if "use the tab key to check that the command format is correct and then try again" in line.lower():
            return False
        if "does not have the permission" in line:
            return False
    return True

def queryResultWithNoRecord(cliRet):
    '''
    @summary: 判断回显是否为Command executed successfully
    @param cliRet: cli回显
    @return: 
        True: cli回显中包含Command executed successfully
        False: cli回显中不包含Command executed successfully
    '''
    if Regex.find(CLI_EXECUTE_CMD_SUCCESS, cliRet, Regex.IGNORECASE):
        return True
    return False

def isInternalError(cliRet):
    '''
    @summary: 判断回显结果是否包含内部错误信息
    @param cliRet: cli回显
    @return: 
        True: cli回显包含内部错误信息
        False: cli回显不包含内部错误信息
    '''
    lowerCliRet = cliRet.lower()
    if "internal error" in lowerCliRet:
        return True
    else:
        return False
    
def getUserName(cli, lang):
    '''
    @summary: 获取设备上登录的用户名
    @param cli: cli对象
    @param cli: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，返回设备上登录的用户名
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    cliRet = ""
    errMsg = ""
    userName = ""

    try:
        cmd = "show system general"
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if checkRet[0] != True: 
            return checkRet
        
        cliRet = checkRet[1]
        lineList = cliRet.splitlines()
        
        if len(lineList) == 0:
            errMsg = getMsg(lang, "cannot.get.sys.info")
            return (False, cliRet, errMsg)
        
        lastLine = lineList[-1]
        if CLI_RET_END_FLAG in lastLine:
            userName = lastLine.replace(" ", "").replace(CLI_RET_END_FLAG, "")
            return (True, userName, errMsg)
    
    except:
        errMsg = getMsg(lang, "cannot.get.sys.info")
        return (False, cliRet, errMsg)
    
    return (False, cliRet, errMsg)

def getUserPrivilege(cli, lang):
    '''
    @summary: 获取用户权限级别
    @param cli: cli对象
    @param cli: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，返回用户权限级别
            flag为False时，返回cli回显结果
        errMsg: 错误消息
    '''
    cliRet = ""
    errMsg = ""

    try:
        checkRet = getUserName(cli, lang) 
        if checkRet[0] != True:
            return checkRet
        
        userName = checkRet[1]
        cmd = "show user user_name=%s" % userName
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if checkRet[0] != True: 
            return checkRet
        
        cliRet = checkRet[1]
        cliRetLinesList = getHorizontalCliRet(cliRet)
        
        if len(cliRetLinesList) == 0:
            errMsg = getMsg(lang, "cannot.get.user.info")
            return (False, cliRet, errMsg)
        
        for line in cliRetLinesList:
            name = line["Name"]
            if name == userName:
                userPrivilege = line["Level"]
                return (True, userPrivilege, errMsg)

    except:
        errMsg = getMsg(lang, "cannot.get.user.info")
        return (False, cliRet, errMsg)
    
    return (False, cliRet, errMsg)

            
def enterDeveloperMode(cli, lang, debugPasswd=None):
    '''
    @summary: 进入developer模式
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, cliRet, errMsg)
        flag:
            True: 进入developer模式成功
            False: 进入developer模式失败
        cliRet: cli回显
        errMsg: 进入developer模式失败时的错误消息
    '''
    errMsg = ""
    cliRet = ""
    cliRetAll = ""
    cmd = "change user_mode current_mode user_mode=developer"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if not checkRet[0]: 
        return checkRet
    cliRet = checkRet[1]

    # 需要确认直接输入y
    cnt = 1
    while "y/n" in cliRet and cnt <= 3:
        cliRet = cli.execCmd("y")
        cnt += 1
    cliRetAll += cliRet
    
    if isInDeveloperMode(cliRet):
        return (True, cliRetAll, errMsg)
    
    if "password" in cliRet.lower():
        errMsg = getMsg(lang, "cannot.access.developer.mode")
        for i in range(0, 3):
            checkRet = excuteCmdInCliMode(cli, debugPasswd, False, lang)
            cliRet = checkRet[1]
            cliRetAll += cliRet
            if isInDeveloperMode(cliRet):
                return (True, cliRetAll, errMsg)
            if "password is wrong" in cliRet.lower():
                errMsg = getMsg(lang, "developer.password.is.wrong")
            
    return (False, cliRetAll, errMsg)

def developerMode2CliMode(cli):
    '''
    @summary: developer模式下退回到cli模式
    @param cli: cli对象
    '''
    cliRet = cli.execCmd("exit")
    if "y/n" in cliRet:
        cli.execCmd("n")
        
    return None
        
def isInDeveloperMode(cliRet):
    '''
    @summary: 判断当前是否在developer模式下
    @param cliRet: cli回显
    @return: 
        True: 当前在developer模式下
        False: 当前不在developer模式下
    '''
    if Regex.find(DEVELOPER_MODEL_FLAG, cliRet, Regex.IGNORECASE):
        return True
    return False

def checkLineInBlackList(cliLine, lang):
    '''
    @summary: 判断cli语句行是否在黑名单中
    @param cliLine: cli语句行
    @param lang: 语言lang
    @return: 
        True: cli语句行在黑名单中
        False: cli语句行不在黑名单中
    '''
    errMsg = ""
    
    for dictItems in cliResource.BLACKLIST_DICT:
        if dictItems.get("key_word") in cliLine:
            if isChinese(lang):
                errMsg = dictItems.get("msg_zh")
            else:
                errMsg = dictItems.get("msg_en")
            return (True, errMsg)
    return (False, errMsg)

def excuteCmdInCliMode(cli, cmd, isHasLog, lang, endWithSignToDiagnose = None):
    '''
    @summary: 获取cli模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param isHasLog: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @return: (falg, cliRet, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        cliRet: cli回显
        errMsg: 错误消息
    '''
    errMsg = ""
    cliRet = ""
    
    if cli is None:
        errMsg = getMsg(lang, "device.connect.abnormal")
        return (RESULT_NOCHECK, "", errMsg)
    
    try:
        if endWithSignToDiagnose and isHasLog:
            cliRet = cli.execCmd(cmd, endWithSignToDiagnose)
        elif endWithSignToDiagnose and not isHasLog:
            cliRet = cli.execCmdNoLog(cmd, endWithSignToDiagnose)
        elif not endWithSignToDiagnose and isHasLog:
            cliRet = cli.execCmd(cmd)
        else:
            cliRet = cli.execCmdNoLog(cmd)
    except IsmException:
        raise
    
    if len(cliRet) == 0:
        errMsg = getMsg(lang, "cli.result.is.empty")
        return (False, cliRet, errMsg)

    if isSupportFeature(cliRet):
        return (RESULT_NOSUPPORT, cliRet, errMsg)
    
    if not hasCliExecPrivilege(cliRet):
        errMsg = getMsg(lang, "has.not.cli.privilege")
        return (RESULT_NOCHECK, cliRet, errMsg)
    
    if isNoneLicense(cliRet):
        errMsg = getMsg(lang, "has.not.license")
        return (RESULT_NOSUPPORT, cliRet, errMsg)
    
    if queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg)
    
    if isInternalError(cliRet):
        return (True, cliRet, errMsg)

    lineList = cliRet.splitlines()
    for line in lineList:
        checkRet = checkLineInBlackList(line, lang)
        if checkRet[0]:
            errMsg = getMsg(lang, "system.status.abnormal")
            return (False, cliRet, errMsg)  
    
    return (True, cliRet, errMsg)

def excuteCmdInCliMode4Privilege(cli, cmd, isHasLog, lang):
    '''
    @summary: 将没有权限的命令结果"未检查"转换成"不涉及"
    '''
    checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang)
    if isSupportFeature(checkRet[1]):
        return (RESULT_NOSUPPORT, checkRet[1], "")
    if not hasCliExecPrivilege(checkRet[1]):
        errMsg = getMsg(lang, "has.not.cli.privilege")
        return (RESULT_NOSUPPORT, checkRet[1], errMsg)
    return checkRet

def excuteCmdInDeveloperMode(cli, cmd, isHasLog, lang, debugPasswd=None, preExec=True):
    '''
    @summary: 获取developer模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param isHasLog: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        ret: cli回显
        errMsg: 错误消息
    '''
    isOpened = False
    try:
        if preExec:
            #兼容部分命令视图切换，先尝试在普通用户模式下执行，再尝试在研发模式下执行
            checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang)
            if checkRet[0] == True:
                return checkRet
        
        isOpened, checkRet = needOpenDeveloperSwitch(cli, lang)

        if isOpened == True:
            checkRet = openDeveloperSwitch(cli, lang)
            if checkRet[0] != True:
                return checkRet
        
        enterDeveloperCheckRet = enterDeveloperMode(cli, lang, debugPasswd)
        if enterDeveloperCheckRet[0] != True:
            return enterDeveloperCheckRet
    
        checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang)
        cliRet = checkRet[1]
        
        if isInDeveloperMode(cliRet):
            developerMode2CliMode(cli)
        return checkRet
    except IsmException:
        raise
    finally:
        if isOpened == True:
            closeDeveloperSwitch(cli, lang)

def getProductModel(cli, lang):
    '''
    @summary: 获取产品类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，产品类型
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    # 先用新cli接口查询内部华为型号，如果不存在跳过，继续查询老接口
    try :
        pdtModelRet = getInternalPdtModel(cli, lang)
        if pdtModelRet[0] == True:
            return pdtModelRet
    except:
        pass

    cmd = "show system general"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    product_model = ""
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Product Model":
            product_model = fieldValue
            
        if len(product_model) != 0 and product_model != "--":
            return (True, product_model, "")
        
    return (False, cliRet, getMsg(lang, "cannot.get.product.model.info"))


def getInternalPdtModel(cli, lang):
    '''
    @summary: 获取华为产品类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，产品类型
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    pdtModel = ""
    cmd = "show system general|filterColumn exclude columnList=Product\sModel"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True:
        return (False, "", checkRet[2])

    cliRet = checkRet[1]
    cliRetLinesList = getVerticalCliRet(cliRet)

    for retDict in cliRetLinesList:
        if retDict.has_key("Internal Product Model"):
            pdtModel = retDict["Internal Product Model"]
            if len(pdtModel) != 0 and pdtModel != "--":
                return (True, pdtModel, "")

    return (False, "", getMsg(lang, "cannot.get.product.model.info"))

def getStorageType(cli, lang):
    '''
    @summary: 获取存储设备类型
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，存储设备类型
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    getProductModelRet = getProductModel(cli, lang)
    if getProductModelRet[0] != True:
        return getProductModelRet
    
    pdtModel = getProductModelRet[1]
    
    if not STOR_DEV_INFO_DICT.has_key(pdtModel):
        errMsg = getMsg(lang, "cannot.get.storage.type.info")
        return (False, "", errMsg)
    
    storageType = STOR_DEV_INFO_DICT.get(pdtModel)
    return (True, storageType, "")

def getHorizontalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合, 处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            match = Regex.search("^\s*-+(\s+-+)*\s*$", line)
            if match != None:
                headline = match.group(0)
                break
            i += 1
            
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []
        
        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        tuple_idxs = Regex.getStartEndList("\s*-+\s*", headline)
            
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))
        
        requiredLineLen = tuple_idxs[-1][0]
        dictList = []
        for line in field_words:
            if CLI_RET_END_FLAG in line:
                break
            
            if len(line.strip()) == 0:
                continue
            
            if len(line) <= requiredLineLen:
                continue
            
            vals = []
            for item in tuple_idxs:
                vals.append(line[item[0]:item[1]].strip().decode("utf8"))
            dictList.append(dict(zip(keys, vals)))
            
        return dictList
    except:
        return []
    
def getVerticalCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取垂直表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合, 处理不正常时，返回空集合
    '''
    cliRetList = cliRet.encode("utf8").splitlines()
    dictList = []
    lineDict = {}
    
    for line in cliRetList:
        if CLI_RET_END_FLAG in line:
            break
            
        if Regex.find("^-+\r*\n*$", line):
            dictList.append(lineDict.copy())
            lineDict.clear()
            
        fields = line.split(" : ")
        if len(fields) < 2:
            continue
        
        key = fields[0].strip().decode("utf8")
        value = ":".join(fields[1:len(fields)]).strip().decode("utf8")
        
        if lineDict.has_key(key):
            key += "_" + str(str(lineDict.keys()).count(key + "_") + 1)
        lineDict.setdefault(key, value)
    
    if len(lineDict) > 0:
        dictList.append(lineDict.copy())
    
    return dictList

def getHorizontalDebugRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合, 处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines()
        for line in cliRetList:
            match = Regex.search("^\s*-+(\s+-+)*\s*$", line)
            if match != None:
                headline = match.group(0)
                break
            i += 1
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []
        
        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        tuple_idxs = Regex.getStartEndList("\s*-+\s*", headline)
           
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))
        
        dictList = []
        for line in field_words:
            if DIAGNOSE_RET_END_FLAG in line:
                break
            
            if len(line.strip()) == 0:
                continue
            
            vals = line.split()
            dictList.append(dict(zip(keys, vals)))
            
        return dictList
    except:
        return []
    
def getSplitedCliRet(cliRet, splitMark):
    '''
    @summary: 按分割标记从回显中获取仅包含该分割标记对应的的回显
    @param cliRet: cli回显
    @param splitMark: 分割标记
    '''
    cliRetlist = cliRet.split(splitMark)
    if len(cliRetlist) < 2:
        return ""
    splitedCliRet = cliRetlist[1]

    match = Regex.search("^.*:(/>)*\r*\n*$", splitedCliRet, Regex.MULTILINE)
    if match != None:
        splitedCliRet = splitedCliRet[0:match.start()]
    
    if len(splitedCliRet.strip()) == 0:
        return ""
    else:
        return splitedCliRet

def isEngine0(value):
    '''
    @summary: 根据id判断是否为引擎0
    @param value: 回显中的ID
    @return: 
        True: 是引擎0
        False: 不是引擎0
    '''
    if "ENG0" in value or "CTE0" in value:
        return True
    else:
        return False

def isEngineId(value):
    '''
    @summary: 根据id判断是否为引擎ID
    @param value: 回显中的ID
    @return: 
        True: 是引擎ID
        False: 不是引擎ID
    '''
    if "ENG" in value or "CTE" in value:
        return True
    else:
        return False

def isDiskId(value):
    '''
    @summary: 根据id判断是否为硬盘ID
    @param value: 回显中的ID
    @return: 
        True: 是硬盘ID
        False: 不是硬盘ID
    '''
    if "DAE" in value:
        return True
    else:
        return False

def isDataSwitchId(value):
    '''
    @summary: 根据id判断是否为交换机ID
    @param value: 回显中的ID
    @return: 
        True: 是交换机ID
        False: 不是交换机ID
    '''
    if "DSW" in value:
        return True
    else:
        return False

def getEnclosureHeight(cli, enclosureId, lang):
    '''
    @summary: 根据框ID获取框高度(U)
    @param cli: cli对象
    @param enclosureId: 框ID
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，框ID对应的高度(U)
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    
    cmd = "show enclosure enclosure_id=%s|filterColumn include columnList=Height(U)" % enclosureId
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Height(U)":
            enclosureHeight = fieldValue
            return (True, enclosureHeight, "")
        
    errMsg = getMsg(lang, "cannot.get.enclosure.info")
    return (False, cliRet, errMsg)

def getEnclosureType(cli, enclosureId, lang):
    '''
    @summary: 根据框ID获取框类型
    @param cli: cli对象
    @param enclosureId: 框ID
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，框ID对应的类型
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    
    cmd = "show enclosure enclosure_id=%s|filterColumn include columnList=Type" % enclosureId
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Type":
            enclosureType = fieldValue
            return (True, enclosureType, "")
        
    errMsg = getMsg(lang, "cannot.get.enclosure.info")
    return (False, cliRet, errMsg)

def getControllerIdList(cli, lang):
    '''
    @summary: 获取设备所有控制器ID
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，设备所有控制器ID列表
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    cmd = "show controller general|filterColumn include columnList=Controller"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetLinesList = cliRet.splitlines()
    controlList = []
    
    for line in cliRetLinesList:
        fields = line.split(":")
        if len(fields) < 2:
            continue
        
        fieldName = fields[0].strip()
        fieldValue = fields[1].strip()
        
        if fieldName == "Controller":
            ctrlId = fieldValue
            controlList.append(ctrlId)
            
    if len(controlList) == 0:
        errMsg = getMsg(lang, "cannot.get.controller.info")
        return (False, cliRet, errMsg)
    
    return (True, controlList, "")

def getAlarmIdList(ssh, lang):
    alarmIdList = []
    cmd = "show alarm |filterColumn include columnList=Name,ID"
    (ok, cliRet, errMsg) = excuteCmdInCliMode(ssh, cmd, True, lang)
    if not ok:
        return (False, alarmIdList, "")

    cliRet2List = getHorizontalCliRet(cliRet)
    for item in cliRet2List:
        alarmIdList.append(str(item.get("ID")))
    return (True, alarmIdList, cliRet)

def isExistAlarm(ssh, lang, alarmId):
    cmd = "show alarm |filterRow column=ID predict=equal_to value=%s" % alarmId
    (ok, cliRet, errMsg) = excuteCmdInCliMode(ssh, cmd, True, lang)
    if not ok:
        return ("Error", "")
    if queryResultWithNoRecord(cliRet):
        return ("NotExist", cliRet)

    cliRet2List = getHorizontalCliRet(cliRet)
    if len(cliRet2List) == 0:
        return ("Error", cliRet)
    return ("Exist", cliRet)

def getEng0Id(cli, lang):
    '''
    @summary: 获取引擎0ID
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag:
            True: 获取成功
            False: 获取失败
        ret: 
            flag为True时，引擎0ID
            flag为False时，cli回显
        errMsg: 错误消息
    '''
    cmd = "show enclosure"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    cliRetLinesList = getHorizontalCliRet(cliRet)
    
    if len(cliRetLinesList) == 0:
        errMsg = getMsg(lang, "cannot.get.enclosure.info")
        return (False, cliRet, errMsg)
    
    for line in cliRetLinesList:
        enclosureId = line["ID"]
        if isEngine0(enclosureId):
            return (True, enclosureId, "")
    
    return (True, "", "")

def getHighDensityDiskEnclosureIdList(cli, lang):
    '''
    @summary: 获取高密硬盘框ID集合
    @param cli: cli对象
    @param lang: 语言lang
    @return: (falg, ret, errMsg)
        flag: 
            True： 判断时正常
            False： 判断时异常
        ret: 
            flag为True时，高密硬盘框ID集合
            flag为False时，返回cli回显
        errMsg: 错误时的消息
    '''
    cmd = "show enclosure"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    highDensityDiskEnclosureIdList = []
    cliRetLinesList = getHorizontalCliRet(cliRet)
    
    if len(cliRetLinesList) == 0:
        errMsg = getMsg(lang, "cannot.get.enclosure.info")
        return (False, cliRet, errMsg)
    
    for line in cliRetLinesList:
        enclosureType = line["Type"]
        enclosureId = line["ID"]
        if "4U 75" in enclosureType:
            highDensityDiskEnclosureIdList.append(enclosureId)
    
    return (True, highDensityDiskEnclosureIdList, "")



#------------------------------------------------------------------------------
#debug mode factory.           time:2015/8/25 10:26
def isInDebugMode(cliRet):
    """ 
    @summary: check the ":/diagnose>" is in the return strings.
    """
    if Regex.find(DEBUG_MODEL_FLAG, cliRet, Regex.IGNORECASE):
        return True
    return False

def enterDebugMode(cli, lang, debugPasswd=None):
    """
    @summary: first enter the developer mode,then the debug mode.
    """
    enterDeveloperCheckRet = enterDeveloperMode(cli, lang, debugPasswd)
    if not enterDeveloperCheckRet[0]:
        developerMode2CliMode(cli)
        return enterDeveloperCheckRet
    
    cmd = "debug"
    checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
    if checkRet[0] != True: 
        return checkRet
    
    cliRet = checkRet[1]
    if isInDebugMode(cliRet):
        return (True, cliRet, "")
    else:
        errMsg = getMsg(lang, "has.not.cli.privilege") 
        return (RESULT_NOCHECK, cliRet, errMsg)
    
def executeCmdInDebugMode(cli, cmd, isHasLog, lang, debugPasswd=None):
    """@summary: execute cmd in debug mode
    """
    isOpened = False
    try:
        isOpened, checkRet = needOpenDeveloperSwitch(cli, lang)

        if isOpened == True:
            checkRet = openDeveloperSwitch(cli, lang)
            if checkRet[0] != True:
                return checkRet
            
        enterDebugCheckRet = enterDebugMode(cli, lang, debugPasswd)
        if enterDebugCheckRet[0] != True:
            return enterDebugCheckRet
        checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang, [">", "> ", ">\r\n"])
        
        cliRet = checkRet[1]
        if isInDebugMode(cliRet):
            exitDebugModeToCliMode(cli)
        
        return checkRet
    except IsmException:
        raise
    finally:
        if isOpened == True:
            closeDeveloperSwitch(cli, lang)
            
    
def exitDebugModeToCliMode(cli):
    """
    @summary: first exit the debug mode,then exit the developer mode.
    """    
    cliRet = cli.execCmd("exit")
    if "y/n" in cliRet:
        cliRet = cli.execCmd("n")
    if isInDeveloperMode(cliRet):
        developerMode2CliMode(cli)
        

def GetCliRecsInDebugMode(cli, cmdList, logger, lang, debugPasswd=None):
    """
    @summary: Execute the cli command in the debug mode.
    @param cmdList:one or more cmd.
    @return: echo list. 
    """
    cliRecsList = list()
    for cmd in cmdList:
        try:
            logger.logInfo("Start to execute the cli cmd:%s" % cmd)
            checkRet = executeCmdInDebugMode(cli, cmd, False, lang, debugPasswd)
            cmdRecs = checkRet[1]
        except Exception, msg:
            logger.logInfo("The Exception is:%s" % msg)
            cmdRecs = ""
        logger.logInfo("The cliRecs is %s" % unicode(cmdRecs))
        cliRecsList.append(cmdRecs)
    return cliRecsList

def getCtrlNames(cli, logger, lang):
    """
    @summary: 获取系统所有控制器名称
    @param cli:cli连接
    @param logger:日志打印对象
    @param lang:语言
    @return: (flag, ctrlIps)(flag:获取成功True，否则False；ctrlIps:[(ctrlId, ctrlIp)])
    """
    cliRet = ""
    ctrlNames = []
    try:
        cmd = "show controller general"
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if checkRet[0] != True:
            return (False, [])
        
        cliRet = checkRet[1]
        ctrlNameList = getVerticalCliRet(cliRet)
        if len(ctrlNameList) == 0:
            return (False, [])
        
        for line in ctrlNameList:
            ctrlName = line["Controller"]
            ctrlNames.append(ctrlName)
            
        return (True, ctrlNames)
    except Exception, exception:
        logger.logException(exception)
        return (False, [])

def getCtrlIps(cli, logger, lang):
    """
    @summary: 获取控制器的ip（中低端为管理ip，高端为内部ip）
    @param cli:cli连接
    @param logger:日志打印对象
    @param lang:语言
    @return: (flag, ctrlIps)(flag:获取成功True，否则False；ctrlIps:[(ctrlId, ctrlIp)])
    """  
    flag = True
    ctrlIps = []
    checkedCtrls = []
    try:
        cmdIpv4 = "show upgrade package"
        checkIpv4Ret = excuteCmdInCliMode(cli, cmdIpv4, True, lang)
        if checkIpv4Ret[0] != True:
            return (False, [])
        
        ipv4CliRet = checkIpv4Ret[1]
        ipv4cliRetLinesList = getHorizontalCliRet(ipv4CliRet)
        if len(ipv4cliRetLinesList) == 0:
            return (False, [])
        
        ctrlsIpv6Ret = getIpv4andIpv6Map(cli, logger, lang)
        ctrlsIpv6 = ctrlsIpv6Ret[1]
        
        for line in ipv4cliRetLinesList:
            ctrlName = line["Name"]
            ipv4 = line["IP"]
            
            if not Regex.find("^[0-9][A-Z]", ctrlName):
                continue
            
            #如果已经获取了的控制器，无需再获取
            if ctrlName in checkedCtrls:
                continue
            
            checkedCtrls.append(ctrlName)
            ipv6 = ctrlsIpv6.get(ipv4, "")
            ctrlIps.append((ctrlName, ipv4, ipv6)) 
        
        #按照控制器ID（如：0A）从小到大排序
        ctrlIps = sorted(ctrlIps, key=lambda x:x[0])
        return (flag, ctrlIps)
    
    except Exception, exception:
        logger.logException(exception)
        return (False, [])

def getIpv4andIpv6Map(cli, logger, lang):
    """
    @summary: 获取控制器的ipv4与ipv6的映射字典
    @param cli:cli连接
    @param logger:日志打印对象
    @param lang:语言
    @return: (flag, ctrlIps)(flag:获取成功True，否则False；ctrlIps:[(ctrlIpv4, ctrlIpv6)])
    """  
    flag = True
    ctrlsIpv6 = {}
    try:
        cmd = "show port general logic_type=Management_Port"
        checkRet = excuteCmdInCliMode(cli, cmd, True, lang)
        if checkRet[0] != True:
            return (False, {})
        
        cliRet = checkRet[1]
        cliRetLinesList = getHorizontalCliRet(cliRet)
        if len(cliRetLinesList) == 0:
            return (False, {})
        
        for line in cliRetLinesList:
            ipv4 = line["IPv4 Address"]
            ipv6 = line["IPv6 Address"]
            ctrlsIpv6[ipv4] = ipv6
        return (flag, ctrlsIpv6)
    
    except Exception, exception:
        logger.logException(exception)
        return (False, {})    
  
def closeCliConnection(cli):
    """
    @summary: 关闭cli连接
    @param cli:cli连接
    """ 
    try:
        cli.close()
    except:
        return
    return   

#------------------------------------------------------------------------------
#open tlv channel.           time:2016/01/19 14:50
def execCmdInDeveloperModePrompt(cli, cmd, isHasLog, lang, prompt="y"):
    '''
    @summary: 获取developer模式下执行命令后的回显
    @param cli: cli对象
    @param cmd: 待执行命令
    @param isHasLog: 是否需要以有log的方式执行cli命令下发
    @param lang: 语言lang
    @param prompt: 确认命令(y / n)
    @return: (falg, ret, errMsg)
        flag:
            True: 执行命令正常
            False: 执行命令不正常
        ret: cli回显
        errMsg: 错误消息
    '''
    isOpened = False
    try:
        isOpened, checkRet = needOpenDeveloperSwitch(cli, lang)

        if isOpened == True:
            checkRet = openDeveloperSwitch(cli, lang)
            if checkRet[0] != True:
                return checkRet        
        
        cliRetAll = ""
        enterDeveloperCheckRet = enterDeveloperMode(cli, lang)
        if not enterDeveloperCheckRet[0]:
            developerMode2CliMode(cli)
            return enterDeveloperCheckRet
        
        checkRet = excuteCmdInCliMode(cli, cmd, isHasLog, lang)
        cliRet = checkRet[1]
        cliRetAll += cliRet
        
        cnt = 0
        while ("y/n" in cliRet and cnt < 5):
            ret = excuteCmdInCliMode(cli, prompt, isHasLog, lang)
            cliRet = ret[1]
            cliRetAll += cliRet
            cnt += 1
        
        if isInDeveloperMode(cliRet):
            developerMode2CliMode(cli)
            
        return (checkRet[0], cliRetAll, checkRet[2])        
    finally:
        #关闭开关
        if isOpened == True :
            closeDeveloperSwitch(cli, lang)
        
     

def changeTlvChannel(cli, lang, switch):
    '''
    @summary: 打开或关闭tlv端口
    @param cli: cli对象
    @param switch: 开关状态
    @return: (falg, ret)
        flag:
            True:  执行成功
            False: 执行失败
        ret: cli回显
    '''
    
    cliRet = ""
    try:
        
        cmd = "change system external_TLV_channel enabled=%s" % switch
        checkRet = execCmdInDeveloperModePrompt(cli, cmd, True, lang)
        cliRet = checkRet[1]
        if not hasCliExecPrivilege(cliRet):
            return (True, cliRet)
        
        if checkRet[0] != True:
            return (False, cliRet)
        
        if CLI_EXECUTE_CMD_SUCCESS in cliRet:
            return (True, cliRet)
        else:
            return (False, cliRet)
    except:
        return (False, cliRet)
    
def openTlvChannel(cli, lang):
    '''
    @summary: 打开或关闭tlv端口
    @param cli: cli对象
    @return: (falg, ret, errMsg, suggestion)
        flag:
            True:  执行成功
            False: 执行失败
        ret: cli回显
        errMsg: 错误消息
        suggestion: 修复建议
    '''
    errMsg = ""
    suggestion = ""
    flag, cliRet = changeTlvChannel(cli, lang, "yes")
    if flag:
        return (True, cliRet, errMsg, suggestion)
    else:
        if lang == "zh":
            errMsg = u"开启TLV通道失败。"
            suggestion = u"请联系技术支持工程师协助处理。"
        else:
            errMsg = "Change the status of the external TLV channel failed."
            suggestion = "Contact technical support engineers for help."
        return (False, cliRet, errMsg, suggestion)

def closeTlvChannel(cli, lang):
    '''
    @summary: 打开或关闭tlv端口，关闭失败不受影响。
    @param cli: cli对象
    @return: (falg, cliRet)
        flag:
            True:  执行成功
            False: 执行失败
        cliRet: cli回显
    '''
    flag, cliRet = changeTlvChannel(cli, lang, "no")
    return (True, cliRet)

def openDeveloperSwitch(cli, lang):
    '''
            打开切换developer模式开关
    '''
    opencmd = "change user_mode enabled user_mode=developer enabled=yes"
    return excuteCmdInCliMode(cli, opencmd, True, lang)

def needOpenDeveloperSwitch(cli, lang):
    '''
    @summary: #兼容V3R6版本，查看切developer模式开关是否打开
    '''
    developercmd = "show user_mode enabled"
    checkRet = excuteCmdInCliMode(cli, developercmd, True, lang)
    if checkRet[0] != True:
        return (CheckStatus.ERROR, checkRet)
    
    cliRetLinesList = getVerticalCliRet(checkRet[1])
    for line in cliRetLinesList:
        if "Disabled" == line.get("Developer View", ""):
            return (True, checkRet)
        
    return (False, checkRet)
        

def closeDeveloperSwitch(cli, lang):
    '''
    @summary: 关闭developer视图开关
    '''
    try:
        closecmd = "change user_mode enabled user_mode=developer enabled=no"
        excuteCmdInCliMode(cli, closecmd, True, lang)
    except:
        pass
    return


def getHorizontalNostandardCliRet(cliRet):
    '''
    @summary: 按逐行字典的方式获取水平表格形式的cli回显集合, 此方法用来解析CLI回显未对其情况
    @param cliRet: cli回显
    @return: 将表格形式cli回显处理为以表头为key，以项值为键的字典集合, 处理不正常时，返回空集合
    '''
    try:
        headline = ""
        i = 0
        cliRetList = cliRet.encode("utf8").splitlines() 
        for line in cliRetList:
            match = Regex.search("^\s*-+(\s+-+)*\s*$", line)
            if match != None:
                headline = match.group(0)
                break
            i += 1
            
        if headline == "" or i == 0 or i >= len(cliRetList) - 1:
            return []
        
        title = cliRetList[i - 1]
        field_words = cliRetList[(i + 1):]
        tuple_idxs = Regex.getStartEndList("\s*-+\s*", headline)
 
        keys = []
        for item in tuple_idxs:
            key = title[item[0]:item[1]].strip()
            if keys.count(key):
                key += "_" + str(str(keys).count(key + "_") + 1)
            keys.append(key.decode("utf8"))
        
        lenkeys = len(keys)
        dictList = []
        for line in field_words:
            if line.find(":/>") >= 0:
                break
            
            if Regex.find("^-+(\s+-+)*\s*$", line):
                continue
            
            if len(line.strip()) == 0:
                continue
            
            vals = []
            valueList = line.strip().split("  ")
            
            valueSpaceList =[]
            for value in valueList:
                if value != "":
                    vals.append(value.strip().decode("utf8"))
                    valueSpaceList.append(value)
            lenvalue = len(valueSpaceList)
            if lenvalue == lenkeys:
                dictList.append(dict(zip(keys, vals)))
            
        return dictList
    except:
        return []

def isInStorageMode(cliRet):
    '''
    @summary: 判断当前是否在Storage模式下，debug 版本.
    @param cliRet: cli回显
    @return: 
        True: 当前在Storage模式下
        False: 当前不在Storage模式下
    '''
    if Regex.find(STORAGE_MODEL_FLAG, cliRet, Regex.IGNORECASE):
        return True
    return False

def excuteCmdCommon(cli, cmd, lang):
    '''
    @summary: 执行cli命令，只判断是否拿到回显信息
    '''
    cliRet = ""
    errMsg = ""
    
    if cli is None:
        errMsg = getMsg(lang, "device.connect.abnormal")
        return (False, cliRet, errMsg)
    
    try:
        cliRet = cli.execCmd(cmd)
    except:
        errMsg = getMsg(lang, "device.connect.abnormal")
        return (False, cliRet, errMsg)
    
    if len(cliRet) == 0:
        errMsg = getMsg(lang, "cli.result.is.empty")
        return (False, cliRet, errMsg)
    
    return (True, cliRet, errMsg)

def getLunIdListWithCliRet(cli, lang):
    """
    @summary: 获取设备上所有LUN的ID
    """
    lunIdList = []
    
    cmd = "show lun general |filterColumn include columnList=ID"
    flag, cliRet, errMsg = excuteCmdInCliMode(cli, cmd, False, lang)
    if flag != True: 
        return (flag, cliRet, errMsg, lunIdList)
    
    if queryResultWithNoRecord(cliRet):
        return (True, cliRet, errMsg, lunIdList)
        
    lunInfoDictList = getHorizontalNostandardCliRet(cliRet)
    if len(lunInfoDictList) == 0:
        errMsg = getMsg(lang, "query.result.abnormal")
        return (False, cliRet, errMsg, lunIdList)
    
    for lunInfoDict in lunInfoDictList:
        lunIdList.append(lunInfoDict.get("ID"))
    
    return (True, cliRet, errMsg, lunIdList)

def getPresentCtrlNodeIdList(cli, lang):
    '''
    @summary: Get all present controller node ID List, such as, [0,1,2,3], not ['0A', '0B']
    @param cli: CLI connection.
    @param lang: Language.
    @return: (isQrySucc, cliRet, errMsg, nodeIdList) 
    '''
    
    #Implement this method by execute diangose command 'sys showcls'
    isExeSucc, cliRet, _ = executeCmdInDebugMode(cli, 'sys showcls', True, lang)
    if not isExeSucc:
        return False, cliRet, getMsg(lang, "failed.to.get.present.nodeid"), []
    
    presentNodeIdList = []
    for line in cliRet.splitlines():
        fields = line.split()
        if len(fields) < 4 or not fields[0].isdigit():
            continue
        else:
            presentNodeIdList.append(fields[0].strip())
    
    return True, cliRet, '', presentNodeIdList

def getPresentCtrlNodeIdListForSshToRemote(cli, lang, engine):
    """
    @summary: 获取指定引擎下的节点（控制器）ID，然后转换为心跳参数
    """
    isExeSucc, cliRet, _ = executeCmdInDebugMode(cli, DIAGNOSE_SYS_SHOW_CLS_CMD, True, lang)
    if not isExeSucc:
        return False, cliRet, getMsg(lang, "failed.to.get.present.nodeid"), []
    
    presentNodeIdList = []
    for line in cliRet.splitlines():
        fields = line.split()
        if len(fields) < 4 or not fields[0].isdigit():
            continue
        
        enigeId = fields[-1].strip()
        if enigeId == engine:
            nodeId = fields[0].strip()
            nodeId = int(nodeId) % 4
            presentNodeIdList.append(nodeId)
    
    return True, cliRet, '', presentNodeIdList

def isInMinisystemMode(cliRet):
    '''
    @summary: 判断当前是否在minisystem模式下
    '''
    if Regex.find(MINISYSTEM_MODEL_FLAG, cliRet, Regex.IGNORECASE):
        return True
    return False

def excuteCmdInMinisystemModel(cli, cmd, lang):
    """
    @summary: 在minisystem模式下执行命令
    """
    cliRet = cli.execCmd(cmd)
    if "password:" in cliRet:
        return True, cliRet, ''
        
    if isInMinisystemMode(cliRet):
        return (True, cliRet, "")
    
    if isInDebugMode(cliRet):
        cliRet = cli.execCmd("exit")
        if "(y/n)" in cliRet:
            cliRet = cli.execCmd("y")
    
    if isInDeveloperMode(cliRet):
        cliRet = cli.execCmd("minisystem")
        if isInMinisystemMode(cliRet):
            cliRet = cli.execCmd(cmd)
            return (True, cliRet, "")
        
        return (False, cliRet, "")
    
    flag, cliRet, errMsg = enterDeveloperMode(cli, lang)
    if flag != True:
        return flag, cliRet, errMsg
    
    cliRet = cli.execCmd("minisystem")
    if not isInMinisystemMode(cliRet):
        return (False, cliRet, "")
    
    cliRet = cli.execCmd(cmd)
    return (True, cliRet, "")

def sshToRemoteContr(cli, cmd, passWord, lang):
    """
    @summary: 在minisystem模式执行心跳命令到指定的控制器
    """
    allCliRet = ""
    flag, cliRet, errMsg = excuteCmdInMinisystemModel(cli, cmd, lang)
    allCliRet += cliRet
    if flag != True:
        return (flag, cliRet, errMsg)
    
    if '(yes/no)?' in cliRet:#首次心跳问题.
        allCliRet += "\n" +cliRet
        flag, cliRet, errMsg = excuteCmdInCliMode(cli, 'yes', True, lang)
            
    if "password:" in cliRet:
        flag, cliRet, errMsg = excuteCmdInCliMode(cli, passWord, False, lang)
        allCliRet += "\n" + cliRet
        if flag != True:
            return (flag, allCliRet, errMsg)

    if "System Name" not in cliRet:
        errMsg = getMsg(lang, "ssh.remote.failure")
        return (False, allCliRet, errMsg)
    
    return (True, allCliRet, errMsg)

def exitHeartbeatCli(cli, lang):
    '''
    @summary: Exit from minisystem mode of peer controller to local controller CLI mode.
    @param cli: CLI connection
    @return:None
    '''
    try:
        for _ in range(3):
            (_, cliRet, _) = excuteCmdCommon(cli, "exit", lang)
            
            while "(y/n)" in cliRet:
                _, cliRet, _  = excuteCmdCommon(cli, "y", lang)
            
            #Compatible for debug version.
            if isInStorageMode(cliRet):
                (_, cliRet, _) = excuteCmdCommon(cli, "exit", lang)
    except:
        traceback.format_exc(None)
        
def getManagementPortInfo(cli, lang):
    """
    @summary: 获取设备管理端口信息（包含所在的引擎，连接状态，管理IP等）
    """
    managementPortInfo = []
    cmd = "show port general physical_type=ETH logic_type=Management_Port"
    flag, cliRet, errMsg = excuteCmdInCliMode(cli, cmd, True, lang)
    if flag != True: 
        return (flag, cliRet, errMsg, managementPortInfo)
    
    managementPortInfo = getHorizontalCliRet(cliRet)
    if len(managementPortInfo) == 0:
        errMsg = getMsg(lang, "query.result.abnormal")
        return (False, cliRet, errMsg, managementPortInfo)
    
    return (True, cliRet, errMsg, managementPortInfo)

def isSupportFeature(cliRet):
    '''
    @summary: 判断是否支持某特性
    '''
    cliRetList = cliRet.splitlines()
    for line in cliRetList:
        lowerLine = line.lower()
        if "error:" in lowerLine and "not support" in lowerLine:
            return True
        #【注意】升级类工具专用：全是使用admin执行，所以不存在权限不足情况
        if lowerLine.strip() == "^":
            return True

    return False

def isSupportFeatureBySSH(cli, cmd):
    '''
    @summary: 判断是否支持此特性
    '''
    cliRet = cli.execCmd(cmd)
    supportFlag = not isSupportFeature(cliRet)
    return (supportFlag, cliRet)

def getHost(cli, lang):
    cmd = "show host general"
    flag, cliRet, errMsg = excuteCmdInCliMode(cli,cmd,False,lang)
    if flag != True:
        return flag, cliRet, errMsg, []

    hosts = getHorizontalCliRet(cliRet)
    return flag, cliRet, errMsg, hosts

def getHostLun(cli,lang,hostId):
    cmd = "show host lun host_id=%s" %hostId
    flag, cliRet, errMsg = excuteCmdInCliMode(cli,cmd,False,lang)
    if flag != True:
        return flag, cliRet, errMsg, []

    luns = getHorizontalCliRet(cliRet)
    return flag, cliRet, errMsg, luns

def getHostInitiator(cli,lang,hostId):
    cmd = "show initiator host_id=%s" %hostId
    flag, cliRet, errMsg = excuteCmdInCliMode(cli,cmd,False,lang)
    if flag != True:
        return flag, cliRet, errMsg, []

    initiators = getHorizontalCliRet(cliRet)
    return flag, cliRet, errMsg, initiators

def getSystemVersion(cli, lang):
    cmd = "show upgrade package"
    (ok, cliRet, errorMsg) = excuteCmdInCliMode(cli, cmd, True, lang)
    if not ok:
        return (False, "", "")

    hotPatchBeginIndex = cliRet.find("HotPatch Version")
    softwareBeginIndex = cliRet.find("Software Version")
    if softwareBeginIndex == -1 or hotPatchBeginIndex == -1:
        return (False, "", "")
    softwareDict = getHorizontalCliRet(cliRet[softwareBeginIndex:hotPatchBeginIndex])
    softwareVersion = softwareDict[0].get('Current Version')

    hotPatchDict = getHorizontalCliRet(cliRet[hotPatchBeginIndex:])
    validHotpatchVers = list(filter(lambda hdict: hdict.get('Current Version') != '--'
                                                  and hdict.get('Current Version'), hotPatchDict))
    if list(validHotpatchVers):
        hotPatchVersion = validHotpatchVers[0].get('Current Version')
    else:
        hotPatchVersion = ''

    return True, softwareVersion, hotPatchVersion