# -*- coding: UTF-8 -*-
from frame.cli import cliUtil
from frame.common import common
from frame.context import contextUtil
from frame.tlv import tlvUtil

srcVersionCheckList = ["V300R001C00","V300R001C01"]
MemoryRequirements = {
    "Dorado6000 V3": {
        "SAS_V300R001C00": ["512.000GB"],
        "SAS_V300R001C01": ["256.000GB", "512.000GB"],
    },
    "Dorado5000 V3": {
        "NVMe": ["256.000GB"],
        "SAS": ["128.000GB"]
    },
}

def execute(context):
    '''
    @summary: 内存规格检查 
    '''
    lang = contextUtil.getLang(context)
    logger = common.getLogger(contextUtil.getLogger(context), __file__)
    originCliRet = ""
    try:
        ssh = contextUtil.getSSH(context)
        devType = str(contextUtil.getDevType(context)).strip()  # 获取当前设备型号
        currentVersion = common.formatVersion(contextUtil.getCurVersion(context)) #获取当前版本
        logger.logInfo("devType=%s, currentVersion=%s" % (devType, currentVersion))
        # 检查设备版本，只有满足特定的设备型号和目标版本才进行该检查
        if currentVersion not in srcVersionCheckList or devType not in MemoryRequirements.keys():
            logger.logInfo("The device not need check controller memory.")
            return common.getUpgEvaluationRs(cliUtil.RESULT_NOSUPPORT, "", "")

        # 检查
        (ok, checkResults, originCliRet) = ControllerMemory(ssh, logger, lang).check(devType, currentVersion)
        (isPass, errMsg) = _parseCheckResults(checkResults, lang)
        return common.getUpgEvaluationRs(isPass, originCliRet, errMsg)
    except Exception, exception:
        logger.logException(exception)
        return common.getUpgEvaluationRs(cliUtil.RESULT_NOCHECK, originCliRet, common.getMsg(lang, "query.result.abnormal"))

def _parseCheckResults(checkResults, lang):
    nopassControllers = []
    errorControllers = []
    for result in checkResults:
        controller = result.keys()[0]
        itemStateStr = result.get(controller)
        if itemStateStr == "nopass":
            nopassControllers.append(controller)
        elif itemStateStr == "error":
            errorControllers.append(controller)
    if len(errorControllers) > 0:
        return (cliUtil.RESULT_NOCHECK, common.getMsg(lang, "query.result.abnormal"))
    if len(nopassControllers) > 0:
        errMsg = common.getMsg(lang, "software_controllerMemoryConf.nopass", ", ".join(nopassControllers))
        return (False, errMsg)
    return (True, "")

class ControllerMemory():
    def __init__(self, ssh, logger, lang):
        self.ssh = ssh
        self.logger = logger
        self.lang = lang

    def check(self, devType, currentVersion):
        originCliRet = ""
        try:
            # 查询引擎框类型, 只有50000才有NVMe
            (ok, engineType, cliRet) = self._getEngineType(devType)
            originCliRet = originCliRet + cliRet
            if not ok:
                return (False, [{"ALL":"error"}], originCliRet)

            # 检查每个控制器的内存规格
            cmd = "show controller general"
            (ok, cliRet, errorMsg) = cliUtil.excuteCmdInCliMode(self.ssh, cmd, True, self.lang)
            originCliRet = originCliRet + cliRet
            if ok != True:
                self.logger.logInfo("get controller failed.")
                return (False, [{"ALL": "error"}], originCliRet)
            cliRet2List = cliUtil.getVerticalCliRet(cliRet)
            if len(cliRet2List) == 0:
                self.logger.logInfo("controller length is zero.")
                return (False, [{"ALL": "error"}], originCliRet)
            checkResult = []
            for item in cliRet2List:
                # 先获取引擎型号+版本的规格，没有仅根据引擎型号获取规格
                cacheCapacityRequirements = MemoryRequirements.get(devType).get(engineType+"_"+currentVersion)
                if cacheCapacityRequirements == None:
                    cacheCapacityRequirements = MemoryRequirements.get(devType).get(engineType)
                isPass = item.get("Cache Capacity") in cacheCapacityRequirements
                errorKey = "pass" if isPass else "nopass"
                checkResult.append({item.get("Controller") : errorKey})
            return (True, checkResult, originCliRet)
        except Exception, e:
            self.logger.logInfo("controller memory check exception.[%s]" % unicode(e))
            return (False, [{"ALL": "error"}], originCliRet)

    # 查询获取引擎框的类型，要么NVMe，或者SAS
    def _getEngineType(self, devType):
        if devType == "Dorado6000 V3":
            return (True, "SAS", "")
        cmd = "show enclosure"
        (ok, cliRet, errorMsg) = cliUtil.excuteCmdInCliMode(self.ssh, cmd, True, self.lang)
        if ok != True:
            self.logger.logInfo("get enclosure failed.")
            return (False, "", cliRet)

        cliRet2List = cliUtil.getHorizontalCliRet(cliRet)
        engineType = None
        for item in cliRet2List:
            # 找到一个Engine框即可, 除了NVMe，皆为SAS
            if item.get("Logic Type") == "Engine":
                if "NVMe" in item.get("Type"):
                    engineType = "NVMe"
                else:
                    engineType = "SAS"
                break
        if engineType == None:
            self.logger.logInfo("engine length is zero.")
            return (False, "", cliRet)
        else:
            return (True, engineType, cliRet)
