# -*- coding: UTF-8 -*-
import ast
from frame.cli import cliUtil
from frame.common import common
from frame.context import contextUtil
import java.lang.Exception as JException

WRITE_MIRROR_STATUS_UNKNOWN = 'unknown write mirror status'
NO_SNAPSHOT_DEV_LIST = ['2800 V3', '2600 V3 for Video']
OBJ_NOT_EXIT = "object not exsit"
DEDUPCOMPRESS_LUN_LIST = ["V300R003C20", "V300R003C20SPC100", "V300R003C20SPC200", "V300R006C00", "V300R006C00SPC100"]
MAX_MIRROR_CHECK_OBJ_NUM = 1000
FAILEED_TRANSFORM = 'failed transform object to volume'

PROGRESS_NUM_1 = 1
PROGRESS_NUM_5 = 5
PROGRESS_NUM_10 = 10
PROGRESS_NUM_15 = 15
PROGRESS_NUM_20 = 20
PROGRESS_NUM_80 = 80
PROGRESS_NUM_MAX = 100

def execute(context):
    checkResult = LunMirrorCheck(context).run(context)
    return common.getUpgEvaluationRs(checkResult[0], checkResult[1], checkResult[2])

class LunMirrorCheck():
    def __init__(self,context):
        self.LANG = contextUtil.getLang(context)
        self.LOGGER = common.getLogger(contextUtil.getLogger(context), __file__)
        self.context = context
        self.is_big_card = False
        self.VolumeToLunDict = {}
        self.CURRENTPROCESS = 0
        self.CHECK_NODE_NUMBERS = 1
        
    def run(self, context):
        '''
        @summary: 存储设备通过OpenStack创建LUN的过程中如果将LUN的Cache镜像功能关闭，后续业务运行过程中，如果控制器复位，将概率导致数据丢失，业务中断。
                  Check LUN cache status and work mode of every LUN and snapshot.
        '''
        flag = True
        allCliRet = ""
        allErrMsg = ""
    
        try:
            cli = contextUtil.getSSH(context)
            checkRet = cliUtil.getControllerIdList(cli, self.LANG)
            if checkRet[0] != True: 
                self.LOGGER.logError("Query controlled ID list failed, check terminated!")
                return (checkRet[0], checkRet[1], checkRet[2])

            self.refreshProcess(context, PROGRESS_NUM_1)
            
            contrIdList = checkRet[1]
            controllerNum = len(contrIdList)
            #单控不存在lun镜像问题，检查通过
            if controllerNum == 1:
                self.LOGGER.logWarning('Single controller running, check PASSED!')
                return (True, allCliRet, allErrMsg) 
            #>>获取设备版本 
            productVersion = contextUtil.getCurVersion(context)
            
            self.refreshProcess(context, PROGRESS_NUM_5)
            
            self.is_big_card = contextUtil.is_dev_of_12GB_SAS_Share_Expansion(context, productVersion)
            self.LOGGER.logInfo('Device is with:' + (' big card.' if self.is_big_card else ' small card.'))
            
            #>>获取LUN ID List
            flag, cliRet, errMsg, lunIdList = cliUtil.getLunIdListWithCliRet(cli, self.LANG)
            allCliRet = joinLines(allCliRet, cliRet)
            self.refreshProcess(context, PROGRESS_NUM_10)
            if flag != True:
                self.LOGGER.logError('Query LUN ID list failed, check terminated!')
                return (cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg)
            
            lunNum = len(lunIdList)
            if lunNum == 0:
                self.LOGGER.logInfo('No lun configured,check passed!')
                return (True, allCliRet, allErrMsg)
            
            #获取设备型号
            devType = contextUtil.getDevType(context)
            self.LOGGER.logInfo('The device type is:%s.' %str(devType))
            
            #>>获取快照信息
            isLicenseMissing = False
            isQrySucc, cliRet, errMsg, snapshotIdList = self.getSnapshotIdList(cli)
            allCliRet = joinLines(allCliRet, cliRet)
            if isQrySucc != True:
                allErrMsg = joinLines(allErrMsg, errMsg)
                return (cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg)

            # >>获取重删压缩LUN信息
            isQrySucc, cliRet, errMsg, dedupCompressLunIdList = self.getDedupCompressLunIdList(cli)
            allCliRet = joinLines(allCliRet, cliRet)
            if isQrySucc != True:
                allErrMsg = joinLines(allErrMsg, errMsg)
                return (cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg)
            
            self.refreshProcess(context, PROGRESS_NUM_15)

            #>>获取文件系统数量
            flag, cliRet, errMsg, fileSystemNum = self.getFileSystemNum(cli)
            allCliRet = joinLines(allCliRet, cliRet)
            if flag != True:
                self.LOGGER.logWarning('Query file system number not success, the device may not support file system!')
            
            lunAndFileSysTotCnt = len(lunIdList) + fileSystemNum
            self.LOGGER.logInfo('the system product version is %s, the number of lun and filesystem is %s' % (productVersion, str(lunAndFileSysTotCnt)))

            self.CHECK_NODE_NUMBERS = self.getCheckNodeNumbers(productVersion, lunAndFileSysTotCnt, controllerNum)
            
            self.CURRENTPROCESS = PROGRESS_NUM_20
            self.refreshProcess(context, self.CURRENTPROCESS)
            
            #>>product version in (.., V3R3C00) || [v3R6C00SPC100, )
            if productVersion < 'V300R003C00' or productVersion >= 'V300R006C00SPC100':
                if lunAndFileSysTotCnt <= 400:
                    isChkSucc, cliRet, errMsg, notMirrorState, _ ,failedTransformLunOrSnapshotList = self.chk2StatesOfAllObjsOnOneNode(cli, lunIdList, snapshotIdList, dedupCompressLunIdList)
                    allCliRet = joinLines(allCliRet, cliRet)
                    allErrMsg = joinLines(allErrMsg, errMsg)
                    notMirrorLunIdList, notMirrorSnapshotIdList, notMirrorDedupCompressLunList = notMirrorState[0], notMirrorState[1], notMirrorState[2]
                    if not isChkSucc:
                        return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg
                    elif notMirrorLunIdList or notMirrorSnapshotIdList or notMirrorDedupCompressLunList:
                        return False, allCliRet, allErrMsg
                    elif isLicenseMissing:
                        return cliUtil.RESULT_WARNING, allCliRet, allErrMsg
                    elif failedTransformLunOrSnapshotList:
                        return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg
                    else:
                        return True, allCliRet, allErrMsg
                else:#LUN and file system number > 400,check one bye one.
                    chkResult, cliRet, errMsg = self.chkCacheStatusOfAllObjsOnOneNodeOneByOne(cli, lunIdList, snapshotIdList, dedupCompressLunIdList)
                    allCliRet = joinLines(allCliRet, cliRet)
                    allErrMsg = joinLines(allErrMsg, errMsg)
                    
                    if chkResult == True and isLicenseMissing:
                        chkResult = cliUtil.RESULT_WARNING
                        
                    return (chkResult, allCliRet, allErrMsg)
                
            #版本号为：'V300R003C00' <= productVersion < 'V300R006C00SPC100':
            else:
                if lunAndFileSysTotCnt > 400:
                    chkResult, cliRet, errMsg = self.chkCacheStatusOfAllObjsOnOneNodeOneByOne(cli, lunIdList, snapshotIdList, dedupCompressLunIdList)
                    allCliRet = joinLines(allCliRet, cliRet)
                    allErrMsg = joinLines(allErrMsg, errMsg)
                    if True == chkResult:
                        #Remind user to check Event.txt file for further checking. 
                        allErrMsg += common.getMsg(self.LANG, "all.lun.snapshot.write.mirror.plz.chk.further.ref.event.file")
                        return cliUtil.RESULT_WARNING, allCliRet, allErrMsg
                    else:
                        #some LUNs or snapshots not mirror
                        return chkResult, allCliRet, allErrMsg
                else:#LUN and file system number <= 400,check all write mirror and work mode status.
                    chkResult, cliRet, errMsg = self.chk2StatesOfAllObjsOnAllNodes(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, contrIdList)
                    if chkResult == True and isLicenseMissing:
                        chkResult = cliUtil.RESULT_WARNING
                    
                    allCliRet = joinLines(allCliRet, cliRet)
                    allErrMsg = joinLines(allErrMsg, errMsg)
                    return chkResult, allCliRet, allErrMsg
                    
        except (Exception,JException), exception: 
            self.LOGGER.logException(exception)
            return (cliUtil.RESULT_NOCHECK, allCliRet, common.getMsg(self.LANG, "query.result.abnormal"))
        finally:
            self.refreshProcess(context, PROGRESS_NUM_MAX)
            #防止检查时间过长，导致CLI连接不可用，需从新获取一次
            contextUtil.getCLI(context)
    
    def getEngineCtrlNumDict(self, contrIdList):
        """
        @summary: 获取引擎下的控制器数量
        """
        engNumDict = {}
        
        for contrId in contrIdList:
            engineId = contrId[:1]
            engNumDict[engineId] = engNumDict.get(engineId, 0) + 1
        
        return engNumDict

    def getFileSystemNum(self, cli):
        """
        @summary: 获取设备上文件系统的数量
        @param cli: CLI connection.
        @return: (QrySuccess, cliRet, errMsg, fileSystemNumber)
        """
        cmd = "show file_system general |filterColumn include columnList=ID"
        flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, self.LANG)
        if flag != True: 
            self.LOGGER.logInfo("Failed to get information about file system.")
            return (False, cliRet, errMsg, 0)
        
        fileSystemInfoDictList = cliUtil.getHorizontalNostandardCliRet(cliRet)
        fileSystemNum = len(fileSystemInfoDictList)
        if fileSystemNum == 0:
            self.LOGGER.logInfo("The result of parse file system is 0, may be wrong.")
            return (True, cliRet, errMsg, fileSystemNum)
    
        self.LOGGER.logInfo('File system number is:' + unicode(fileSystemNum))           
        return (True, cliRet, errMsg, fileSystemNum)

    def getConnectedEngineId(self, managementPortDictList): 
        """
        @summary: Get the ID of engine that toolkit has connected.
        @param managementPortDictList: Management port information dictionary.
        @return: Number of engine that toolkit has current connected.
        """
        currentContrIp = contextUtil.getCurCtrlIp(self.context)
        self.LOGGER.logInfo("The IP that ToolKit connected is %s." % currentContrIp)
        
        for managementPortDict in managementPortDictList:
            ip = managementPortDict.get("IPv4 Address")
            if ip == currentContrIp:
                managementPortId = managementPortDict.get("ID")
                engineId = managementPortId.split(".")[0][-1:]
                self.LOGGER.logInfo("The engine that ToolKit connected is %s." % engineId)
                return  engineId
    
        return ""


    def getAllEngineReachableIP(self, mgtPortDictList):
        """
        @summary: Get reachable IP address of every engine.
        @param mgtPortDictList: Management port IP address information dictionary.
        @return: {engineId: reachable_IP_address}
        """
        engineIpDict = {}
        for managementPortInfoDict in mgtPortDictList:
            ip = managementPortInfoDict.get("IPv4 Address")
            portRunningStatus = managementPortInfoDict.get("Running Status")
            if portRunningStatus == "Link Up" and ip not in ["", "--"]:
                managementPortId = managementPortInfoDict.get("ID")
                engineId = managementPortId.split(".")[0][-1:]
                if (engineId not in engineIpDict) and ("DSW" not in managementPortId):
                    engineIpDict.setdefault(engineId, ip)
        
        return engineIpDict

    def getSnapshotIdList(self, cli):
        """
        @summary: 获取所有的快照id
        @param cli: CLI connection.
        @return: (isQrySucc, cliRet, errMsg, snapshotIdList)
        """
        snapshotIdList = []
        
        cmd = "show snapshot general |filterColumn include columnList=ID"
        flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, self.LANG)
        if flag == False: 
            self.LOGGER.logInfo("Failed to get information about snapshot.")
            return (False, cliRet, errMsg, snapshotIdList)
        elif flag == cliUtil.RESULT_NOSUPPORT or flag == cliUtil.RESULT_NOCHECK:
            return (True, cliRet, errMsg, snapshotIdList)
        elif cliUtil.queryResultWithNoRecord(cliRet):
            return (True, cliRet, errMsg, [])
          
        snapshotInfoDictList = cliUtil.getHorizontalNostandardCliRet(cliRet)
        if len(snapshotInfoDictList) == 0:
            self.LOGGER.logWarning("Snapshot number is 0, CLI result parsing may have failed.")
            return (False, cliRet, errMsg, snapshotIdList)
        
        for snapshotInfoDict in snapshotInfoDictList:
            snapshotId = snapshotInfoDict.get("ID")
            snapshotIdList.append(snapshotId)
        
        self.LOGGER.logInfo('Snapshot number is:' + unicode(len(snapshotIdList)))   
         
        return (True, cliRet, errMsg, snapshotIdList)

    def getDedupCompressLunIdList(self, cli):
        """
        @summary: 获取所有ROW LUN的id
        @param cli: CLI connection.
        @return: (isQrySucc, cliRet, errMsg, snapshotIdList)
        """
        dedupCompressLunIdList = []

        cmd = "show dedup_compress_lun general|filterColumn include columnList=Relate\sLUN\sID"
        flag, cliRet, errMsg = cliUtil.excuteCmdInDeveloperMode(cli, cmd, True, self.LANG)

        if flag == False:
            self.LOGGER.logInfo("Failed to get information about dedup_compress_lun.")
            return (False, cliRet, errMsg, dedupCompressLunIdList)
        elif flag == cliUtil.RESULT_NOSUPPORT or flag == cliUtil.RESULT_NOCHECK:
            return (True, cliRet, errMsg, dedupCompressLunIdList)
        elif cliUtil.queryResultWithNoRecord(cliRet):
            return (True, cliRet, errMsg, dedupCompressLunIdList)

        dedupCompressLunInfoDictList = cliUtil.getHorizontalNostandardCliRet(cliRet)
        if len(dedupCompressLunInfoDictList) == 0:
            self.LOGGER.logWarning("dedupCompressLun number is 0, CLI result parsing may have failed.")
            return (False, cliRet, errMsg, dedupCompressLunIdList)

        for dedupCompressLunInfoDict in dedupCompressLunInfoDictList:
            dedupCompressLunId = dedupCompressLunInfoDict.get("Relate LUN ID")
            dedupCompressLunIdList.append(dedupCompressLunId)

        self.LOGGER.logInfo('dedupCompressLun number is:' + unicode(len(dedupCompressLunIdList)))
        return (True, cliRet, errMsg, dedupCompressLunIdList)

    def chkSingleObjCacheStatus(self, cli, objId):
        '''
        @summary: 检查单个LUN或者快照的镜像写状态
        @param cli: CLI connection
        @param objId: LUN or snapshot ID
        @return: True --write mirror; False-- Not mirror; 'Unknown'
        '''
        
        cmd = "cache show obj %s state" % objId
        excuFlag, cliRet, errMsg = cliUtil.executeCmdInDebugMode(cli, cmd, True,self.LANG)
        if excuFlag != True: 
            self.LOGGER.logSysAbnormal()
            self.LOGGER.logInfo("Failed to get information of LUN %s cache mirror." % objId)
            return WRITE_MIRROR_STATUS_UNKNOWN, cliRet, errMsg
        
        if "obj not found" in cliRet.lower():
            # 将对象id转换为卷id
            ret = self.transformLunToVolume(cli, objId)
            if ret[0] == False:
                return FAILEED_TRANSFORM, cliRet, ''
            cmd = "cache show obj %s state" % ret[2]
            excuFlag, cliRet, errMsg = cliUtil.executeCmdInDebugMode(cli, cmd, True, self.LANG)
            if excuFlag != True:
                self.LOGGER.logSysAbnormal()
                self.LOGGER.logInfo("Failed to get information of LUN %s cache mirror." % objId)
                return WRITE_MIRROR_STATUS_UNKNOWN, cliRet, errMsg

        if ":" not in cliRet.lower() or "mirror state" not in cliRet.lower():
            return WRITE_MIRROR_STATUS_UNKNOWN, cliRet, ''

        for cliRetLine in cliRet.splitlines():
            fields = cliRetLine.split(":")
            if len(fields) < 2:
                continue
            fieldName = fields[0].strip().lower()
            fieldValue = fields[1].strip().lower()
            if fieldName == "mirror state" and fieldValue == "mirror":
                return True, cliRet, ''
            elif fieldName == "mirror state" and fieldValue != "mirror":
                return False, cliRet, ''
            
        return False, cliRet, ''
    
    def chkCacheStatusOfAllObjsOnOneNodeOneByOne(self, cli, lunIdList, snapshotIdList, dedupCompressLunIdList):
        """
        @summary: check the write mirror status of some LUNs and some snapshots on any controller(not need to heart beat).
        @param cli: CLI connection
        @param lunIdList: LUN ID List
        @param snapshotIdList: Snapshot ID list
        @return: (check result, error message)
        """
        notMirrorLunList = []
        notMirrorSnapshotList = []
        notMirroeDedupCompressLunList = []
        
        unknownLunMirrorStaIdList = []
        unknownSnapshotMirrorStaIdList = []
        unknownDedupCompressLunStaIdList = []
        
        failedTransformLunList = []
        failedTransformSnapshotList = []
        failedTransformDedupCompressLunList = []
        
        notExsitLunList = []
        notExsitSnapshotList = []
        notExsitDedupCompressLunList = []
        
        allCliRet, allErrMsg = '', ''

        checkedObjNum = 0
        
        objTotalNum = len(lunIdList) + len(snapshotIdList) + len(dedupCompressLunIdList)
        self.LOGGER.logInfo("The total number of object is %s." % str(objTotalNum))
        #检查一个对象的进度
        oneObjProcess = (1.0 / objTotalNum) * (PROGRESS_NUM_MAX - self.CURRENTPROCESS)
        self.LOGGER.logInfo("oneObjProcess is %s." % str(oneObjProcess))
        
        for lunId in lunIdList:
            if checkedObjNum > MAX_MIRROR_CHECK_OBJ_NUM:
                break
            wmStatus, cliRet, errMsg = self.chkSingleObjCacheStatus(cli, lunId)
            allCliRet += ('\n' + cliRet)
            if wmStatus == False:
                notMirrorLunList.append(lunId)
                allErrMsg += common.getMsg(self.LANG, "exist.lun.not.mirror", ",".join(notMirrorLunList))
                return False, allCliRet, allErrMsg
            elif wmStatus == WRITE_MIRROR_STATUS_UNKNOWN:
                allErrMsg += ('\n' + errMsg)
                unknownLunMirrorStaIdList.append(lunId)
            elif wmStatus == OBJ_NOT_EXIT:
                notExsitLunList.append(lunId)
            elif wmStatus == FAILEED_TRANSFORM:
                failedTransformLunList.append(lunId)
            checkedObjNum += 1
            
            #刷新检查进度
            if int(self.CURRENTPROCESS + oneObjProcess) - int(self.CURRENTPROCESS) >= 1:
                self.refreshProcess(self.context, self.CURRENTPROCESS + oneObjProcess)
            self.CURRENTPROCESS += oneObjProcess
        
        for snapshotId in snapshotIdList:
            if checkedObjNum > MAX_MIRROR_CHECK_OBJ_NUM:
                break
            wmStatus, cliRet, errMsg = self.chkSingleObjCacheStatus(cli, snapshotId)
            allCliRet += ('\n' + cliRet)
            if wmStatus == False:
                notMirrorSnapshotList.append(snapshotId)
                allErrMsg += common.getMsg(self.LANG, "exist.snapshot.not.mirror", ",".join(notMirrorSnapshotList))
                return False, allCliRet, allErrMsg
            elif wmStatus == WRITE_MIRROR_STATUS_UNKNOWN:
                allErrMsg += ('\n' + errMsg)
                unknownSnapshotMirrorStaIdList.append(snapshotId)
            elif wmStatus == OBJ_NOT_EXIT:
                notExsitSnapshotList.append(snapshotId)
            elif wmStatus == FAILEED_TRANSFORM:
                failedTransformLunList.append(snapshotId)
            checkedObjNum += 1
            
            #刷新检查进度
            if int(self.CURRENTPROCESS + oneObjProcess) - int(self.CURRENTPROCESS) >= 1:
                self.refreshProcess(self.context, self.CURRENTPROCESS + oneObjProcess)
            self.CURRENTPROCESS += oneObjProcess

        for dedupCompressLunId in dedupCompressLunIdList:
            if checkedObjNum > MAX_MIRROR_CHECK_OBJ_NUM:
                break
            wmStatus, cliRet, errMsg = self.chkSingleObjCacheStatus(cli, dedupCompressLunId)
            allCliRet += ('\n' + cliRet)
            if wmStatus == False:
                notMirroeDedupCompressLunList.append(dedupCompressLunId)
                allErrMsg += common.getMsg(self.LANG, "exist.dedupCompressLun.not.mirror", ",".join(notMirroeDedupCompressLunList))
                return False, allCliRet, allErrMsg
            elif wmStatus == WRITE_MIRROR_STATUS_UNKNOWN:
                allErrMsg += ('\n' + errMsg)
                unknownDedupCompressLunStaIdList.append(dedupCompressLunId)
            elif wmStatus == OBJ_NOT_EXIT:
                notExsitDedupCompressLunList.append(dedupCompressLunId)
            elif wmStatus == FAILEED_TRANSFORM:
                failedTransformLunList.append(dedupCompressLunId)
            checkedObjNum += 1
            
            #刷新检查进度
            if int(self.CURRENTPROCESS + oneObjProcess) - int(self.CURRENTPROCESS) >= 1:
                self.refreshProcess(self.context, self.CURRENTPROCESS + oneObjProcess)
            self.CURRENTPROCESS += oneObjProcess

        if notMirrorLunList:
            notMirrorLunList.sort(cmp=None, key=int, reverse=False)
            allErrMsg += common.getMsg(self.LANG, "lun.not.mirror", ",".join(notMirrorLunList))        
        
        if notMirrorSnapshotList:
            notMirrorSnapshotList.sort(cmp=None, key=int, reverse=False)
            allErrMsg += common.getMsg(self.LANG, "snapshot.not.mirror", ",".join(notMirrorSnapshotList))

        if notMirroeDedupCompressLunList:
            notMirroeDedupCompressLunList.sort(cmp=None, key=int, reverse=False)
            allErrMsg += common.getMsg(self.LANG, "dedupCompress.lun.not.mirror", ",".join(notMirroeDedupCompressLunList))

        if unknownLunMirrorStaIdList:
            unknownLunMirrorStaIdList.sort(cmp=None, key=int, reverse=False)
            allErrMsg += common.getMsg(self.LANG, "lun.mirror.status.unknown", ",".join(unknownLunMirrorStaIdList))        
        
        if unknownSnapshotMirrorStaIdList:
            unknownSnapshotMirrorStaIdList.sort(cmp=None, key=int, reverse=False)
            allErrMsg += common.getMsg(self.LANG, "snapshot.mirror.status.unknown", ",".join(unknownSnapshotMirrorStaIdList))

        if unknownDedupCompressLunStaIdList:
            unknownDedupCompressLunStaIdList.sort(cmp=None, key=int, reverse=False)
            allErrMsg += common.getMsg(self.LANG, "dedupCompress.lun.mirror.status.unknown", ",".join(unknownDedupCompressLunStaIdList))

        if failedTransformLunList:
            failedTransformLunList.sort(cmp=None, key=int, reverse=False)
            allErrMsg += common.getMsg(self.LANG, "faied.to.transform.lun.to.volume", ",".join(failedTransformLunList))
        
        if failedTransformSnapshotList:
            failedTransformSnapshotList.sort(cmp=None, key=int, reverse=False)
            allErrMsg += common.getMsg(self.LANG, "faied.to.transform.snapshot.to.volume", ",".join(failedTransformSnapshotList))

        if failedTransformDedupCompressLunList:
            failedTransformDedupCompressLunList.sort(cmp=None, key=int, reverse=False)
            allErrMsg += common.getMsg(self.LANG, "faied.to.transform.dedupCompressLun.to.volume", ",".join(failedTransformDedupCompressLunList))
        
        if notExsitLunList:
            notExsitLunList.sort(cmp=None, key=int, reverse=False)
            allErrMsg += common.getMsg(self.LANG, "cant.find.lun.mirrorinfo", ",".join(notExsitLunList))
        
        if notExsitSnapshotList:
            notExsitSnapshotList.sort(cmp=None, key=int, reverse=False)
            allErrMsg += common.getMsg(self.LANG, "cant.find.snapshot.mirrorinfo", ",".join(notExsitSnapshotList))

        if notExsitDedupCompressLunList:
            notExsitDedupCompressLunList.sort(cmp=None, key=int, reverse=False)
            allErrMsg += common.getMsg(self.LANG, "cant.find.dedupCompressLun.mirrorinfo", ",".join(notExsitDedupCompressLunList))

        # 已经检查超过1000个对象时
        if checkedObjNum > MAX_MIRROR_CHECK_OBJ_NUM:
            if notMirrorLunList or notMirrorSnapshotList or notMirroeDedupCompressLunList:
                allNotMirrorIds = common.getMsg(self.LANG, "lun.id.list",
                                                ",".join(notMirrorLunList)) if notMirrorLunList else ""
                allNotMirrorIds += common.getMsg(self.LANG, "snapshot.id.list",
                                                 ",".join(notMirrorSnapshotList)) if notMirrorSnapshotList else ""
                allNotMirrorIds += common.getMsg(self.LANG, "decompresssublun.id.list", ",".join(
                    notMirroeDedupCompressLunList)) if notMirroeDedupCompressLunList else ""
                allErrMsg = common.getMsg(self.LANG, "check.mirror.largeMaxCheckNum.notpass", allNotMirrorIds)
                return False, allCliRet, allErrMsg
            else:
                allErrMsg = common.getMsg(self.LANG, "check.mirror.largeMaxCheckNum.warning")
                return cliUtil.RESULT_WARNING, allCliRet, allErrMsg

        # 总数未超过1000个对象时
        if notMirrorLunList or notMirrorSnapshotList or notMirroeDedupCompressLunList:
            return False, allCliRet, allErrMsg
        elif unknownLunMirrorStaIdList or unknownSnapshotMirrorStaIdList or unknownDedupCompressLunStaIdList or failedTransformLunList or failedTransformSnapshotList or failedTransformDedupCompressLunList or notExsitLunList or notExsitSnapshotList or notExsitDedupCompressLunList:
            return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg
        else:
            return True, allCliRet, ''

    def chk2StatesOfAllObjsOnOneNode(self, cli, lunIdList, snapshotIdList, dedupCompressLunIdList, chkWorkMode=False):
        """
        @summary: 在CLI连接所在节点上检查每个lLUN和快照的Cache写镜像状态是否开启（当LUN与文件系统个数之和小于300）以及工作模式。
        @param cli: CLI connection(SSH connection)
        @param lunIdList: LUN ID List
        @param snapshotIdList: Snapshot ID List
        @return: (isQrySucc, cliRet, errMsg, 
                    (notMirrorLunIdList, notMirrorSnapshotIdList),
                    (singleWorkmodeLunIdList, singlgeWorkmodeSnapshotIdList))
        """
        
        cmd = "cache show obj v switch all"
        flag, cliRet, errMsg = cliUtil.executeCmdInDebugMode(cli, cmd, True, self.LANG)
        if flag != True: 
            self.LOGGER.logSysAbnormal()
            return (False, cliRet, errMsg, (None, None, None), (None, None, None), None)
        
        lunWriteMirrorDictList = cliUtil.getHorizontalCliRet(cliRet)
        if len(lunWriteMirrorDictList) == 0:
            errMsg = common.getMsg(self.LANG, "query.result.abnormal")
            self.LOGGER.logError("Failed to get information of LN cache mirror status.")
            return (False, cliRet, errMsg, (None, None, None), (None, None, None), None)

        #未开启镜像功能的LUN
        notMirrorLunList = []
        #未开启镜像功能的快照
        notMirrorSnapshotList = []
        # 未开启镜像功能的开启过重删压缩开关的LUN
        notMirrorDedupCompressLunList = []
        
        singleCtrlModeOfLunList = []
        singleCtrlModeOfSnapshotList = []
        singleCtrlModeOfDedupCompressLunList = []
        
        #转换LUN或者快照失败的卷ID
        failedTransformLunOrSnapshotList = []
        
        for lunWirteMirrorDict in lunWriteMirrorDictList:
            #判断是否为Lun对应的volumeId，大于65535为元数据卷，非Lun镜像的卷
            volumeId = lunWirteMirrorDict.get("volumeid")
            if not volumeId:
                volumeId = lunWirteMirrorDict.get("volumeId")
            try:
                if volumeId == None or ast.literal_eval(volumeId) > ast.literal_eval('0xffff'):
                    continue
            except:
                self.LOGGER.logError("get except!")
            
            lunWirteMirrorState = lunWirteMirrorDict.get("writeMirror")
            workMode = lunWirteMirrorDict.get("workmode")
            if not lunWirteMirrorState:
                lunWirteMirrorState = lunWirteMirrorDict.get("wmi")
            
            if not workMode:
                workMode = lunWirteMirrorDict.get("wmo")
                
            if lunWirteMirrorState != "1":
                #将卷ID转换为LUN ID/快照ID
                ret = self.transformVolumeToLunOrSnapshot(cli, volumeId)
                if ret[0] == False:
                    failedTransformLunOrSnapshotList.append(volumeId)
                    cliRet = joinLines(cliRet, ret[1])
                    continue
                lunOrSnapshot = ret[2]
                try:
                    #diagnose下获取到的LUN ID是16进制，需要转换成用户更易理解的10进制
                    lunOrSnapshot = str(int(ast.literal_eval(lunOrSnapshot)))
                except:
                    self.LOGGER.logInfo("Failed to transform volumeId")
                    
                if lunOrSnapshot in lunIdList:
                    notMirrorLunList.append(lunOrSnapshot)
                    
                if lunOrSnapshot in snapshotIdList:
                    notMirrorSnapshotList.append(lunOrSnapshot)

                if lunOrSnapshot in dedupCompressLunIdList:
                    notMirrorDedupCompressLunList.append(lunOrSnapshot)
                    
            if workMode != "1":
                #将卷ID转换为LUN ID/快照ID
                ret = self.transformVolumeToLunOrSnapshot(cli, volumeId)
                if ret[0] == False:
                    failedTransformLunOrSnapshotList.append(volumeId)
                    cliRet = joinLines(cliRet, ret[1])
                    continue
                lunOrSnapshot = ret[2]
                try:
                    #diagnose下获取到的LUN ID是16进制，需要转换成用户更易理解的10进制
                    lunOrSnapshot = str(int(ast.literal_eval(lunOrSnapshot)))
                except Exception, exception:
                    self.LOGGER.logInfo("Transform volume ID(LUN ID/Snapshot ID) exception from hex to decimal base.")
                    self.LOGGER.logException(exception)
                    
                if lunOrSnapshot in lunIdList:
                    singleCtrlModeOfLunList.append(lunOrSnapshot)
                    
                if lunOrSnapshot in snapshotIdList:
                    singleCtrlModeOfSnapshotList.append(lunOrSnapshot)

                if lunOrSnapshot in dedupCompressLunIdList:
                    singleCtrlModeOfDedupCompressLunList.append(lunOrSnapshot)
        
        if len(notMirrorLunList) != 0:
            notMirrorLunList.sort(cmp=None, key=int, reverse=False)
            errMsg += common.getMsg(self.LANG, "lun.not.mirror", ",".join(notMirrorLunList))
    
        if len(notMirrorSnapshotList) != 0:
            notMirrorSnapshotList.sort(cmp=None, key=int, reverse=False)
            errMsg += common.getMsg(self.LANG, "snapshot.not.mirror", ",".join(notMirrorSnapshotList))

        if len(notMirrorDedupCompressLunList) != 0:
            notMirrorDedupCompressLunList.sort(cmp=None, key=int, reverse=False)
            errMsg += common.getMsg(self.LANG, "dedupCompress.lun.not.mirror", ",".join(notMirrorDedupCompressLunList))
        
        if chkWorkMode and not self.is_big_card:  
            if singleCtrlModeOfLunList:
                singleCtrlModeOfLunList.sort(cmp=None, key=int, reverse=False)
                errMsg += common.getMsg(self.LANG, "lun.work.in.single.ctrl.mode", ",".join(singleCtrlModeOfLunList))
        
            if len(singleCtrlModeOfSnapshotList) != 0:
                singleCtrlModeOfSnapshotList.sort(cmp=None, key=int, reverse=False)
                errMsg += common.getMsg(self.LANG, "snapshot.work.in.single.ctrl.mode", ",".join(singleCtrlModeOfSnapshotList))

            if len(singleCtrlModeOfDedupCompressLunList) != 0:
                singleCtrlModeOfDedupCompressLunList.sort(cmp=None, key=int, reverse=False)
                errMsg += common.getMsg(self.LANG, "dedupCompress.work.in.single.ctrl.mode",
                                        ",".join(singleCtrlModeOfDedupCompressLunList))

        if len(failedTransformLunOrSnapshotList) != 0:
            errMsg += common.getMsg(self.LANG, "faied.to.transform.volume.to.LunOrSnapshot", ",".join(failedTransformLunOrSnapshotList))
            
        self.CURRENTPROCESS += (PROGRESS_NUM_80 / self.CHECK_NODE_NUMBERS)
        self.refreshProcess(self.context, self.CURRENTPROCESS)
        
        return (True, cliRet, errMsg, (notMirrorLunList, notMirrorSnapshotList, notMirrorDedupCompressLunList),
                (singleCtrlModeOfLunList, singleCtrlModeOfSnapshotList, singleCtrlModeOfDedupCompressLunList), failedTransformLunOrSnapshotList)
    
    def chk2StatesOfAllObjsOnAllNodes(self, cli, lunIdList, snapshotIdList, dedupCompressLunIdList, ctrlIdList):
        '''
        @summary: check every LUN and snapshot write mirror and work mode status on every controller.
        @param cli: CLI connection
        @param lunIdList: LUN ID List
        @param snapshot: Snapshot ID list
        @return: (checkResult, cliRet, errMsg)
        '''   
        allCliRet, errMsg = '', ''
        
        engNumDict = self.getEngineCtrlNumDict(ctrlIdList)
        engineNum = len(engNumDict)
        
        if engineNum == 1:
            return self.chk2StatesOfAllObjsOnAllNodesInEngine(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, ctrlIdList, '0')
        else:
            qryRet, cliRet, errMsg, mgtPortInfo = cliUtil.getManagementPortInfo(cli, self.LANG)
            if True != qryRet:
                self.LOGGER.logError('Query management port information failed.')
                return cliUtil.RESULT_NOCHECK, cliRet, errMsg 
            else:
                self.LOGGER.logInfo('Query management port information success:' + unicode(mgtPortInfo))
                
            engReachableIPDict = self.getAllEngineReachableIP(mgtPortInfo)
            self.LOGGER.logInfo('All engine reachable IP:' + unicode(engReachableIPDict))
    
            allCliRet = ""
            
            curConnectedEngId = self.getConnectedEngineId(mgtPortInfo)
            curConnectedEngNodeNum = engNumDict.get(curConnectedEngId, 0)
            self.LOGGER.logInfo("The engine of ToolKit connecting is %(engId)s, and its controller number is %(engNodeNum)s."
                           % {'engId':unicode(curConnectedEngId), 'engNodeNum':unicode(curConnectedEngNodeNum)})
            
            #先在工具连接的当前引擎内检查
            if curConnectedEngId and curConnectedEngNodeNum:
                #控制器数量大于2时心跳到每个控制器检查
                checkResult, cliRet, errMsg = self.chk2StatesOfAllObjsOnAllNodesInEngine(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, ctrlIdList, curConnectedEngId)
                allCliRet = joinLines(allCliRet, cliRet)
                if checkResult != True:
                    self.LOGGER.logError('Current connected engine check not PASS, check finished.')
                    return checkResult, allCliRet, errMsg
                else:
                    self.LOGGER.logError('Current connected engine check PASS, check other engine further.')
                    
            for engId in engReachableIPDict:
                if engId == curConnectedEngId:
                    continue
                
                engMgtReachableIP = engReachableIPDict.get(engId)
                self.LOGGER.logInfo("Begin connect engine %s, and its management IP is %s." % (engId, engMgtReachableIP))
                cilConnection = contextUtil.createCliConnection(self.context, engMgtReachableIP)
                if not cilConnection:
                    self.LOGGER.logInfo("Fail to connect engine %s, and its management IP is %s." % (engId, engMgtReachableIP))
                    errMsg = common.getMsg(self.LANG, "dev.connEngine.failure", (engId, engId))
                    return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
                
                checkResult, cliRet, errMsg = self.chk2StatesOfAllObjsOnAllNodesInEngine(cilConnection, lunIdList, snapshotIdList, dedupCompressLunIdList, ctrlIdList, engId)
                allCliRet = joinLines(allCliRet, cliRet)
                
                #关闭当前引擎的连接（不是工具与阵列的原始CLI连接）
                contextUtil.destroyCliConn(cilConnection)
                
                if checkResult != True:
                    self.LOGGER.logError('This engine(Engine number:' + unicode(engId) + '), check not PASS, check finished.')
                    return (checkResult, allCliRet, errMsg)
            
            self.LOGGER.logInfo('All connected engine has been checked.')
            
            #已经检查过的引擎   
            chkedEngIdList = engReachableIPDict.keys()
            #集群中的所有引擎
            allEngIdList = engNumDict.keys()
            
            notConnectedEngIdList = [engId for engId in allEngIdList if engId not in chkedEngIdList]
            if notConnectedEngIdList:
                notConnectedEngIdList.sort()
                self.LOGGER.logError("There has not connected engine that can not be checked: " + ','.join(notConnectedEngIdList))
                errMsg += common.getMsg(self.LANG, "dev.connEngine.failure", (','.join(notConnectedEngIdList), ','.join(notConnectedEngIdList)))
                return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
               
            return (True, allCliRet, "")    
    
    def chk2StatesOfAllObjsOnCtrlPair(self, cli, lunIdList, snapshotIdList, dedupCompressLunIdList, isPeerPresent=True):
        '''
        @summary: Check write mirror status and work mode on 2-controllers pair in one engine.
        @param cli:CLI connection
        @param lunIdList: LUN ID List.
        @param snapshotIdList: Snapshot ID List.
        @return: (checkResult, cliRet, errMsg)
        '''
        #先在工具当前连接的控制器上检查
        allCliRet = ''
        isChkSucc, cliRet, errMsg, notMirrorState, singleWorkMode, failedTransformLunOrSnapshotList = self.chk2StatesOfAllObjsOnOneNode(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, True)
        notMirrorLunIdList, notMirrorSnapshotIdList, notMirrorDedupCompressLunList = notMirrorState[0], notMirrorState[1], notMirrorState[2]
        singleWorkModeLunList, singleWorkModeSnapshotList, singleCtrlModeOfDedupCompressLunList = singleWorkMode[0], singleWorkMode[1], singleWorkMode[2]
        allCliRet = joinLines(allCliRet, cliRet)
        if not isChkSucc:
            return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
        elif notMirrorLunIdList or notMirrorSnapshotIdList or notMirrorDedupCompressLunList or singleWorkModeLunList or singleWorkModeSnapshotList or singleCtrlModeOfDedupCompressLunList:
            return False, allCliRet, errMsg
        elif failedTransformLunOrSnapshotList:
            return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
        else:
            self.LOGGER.logInfo('All LUNs or snapshots cache write mirror on this controller are enabled and work on double'
            ' controller mode, check other controller now...')
        
        if not isPeerPresent:
            self.LOGGER.logInfo('Peer controller not present, check finished.') 
            return True, allCliRet, errMsg

        priKey = contextUtil.getDevObj(self.context).getPriKey()
        if priKey:
            # 不支持PublicKey鉴权方式进行心跳控制器
            errMsg = common.getMsg(self.LANG, "no.support.publickey.forensics")
            return (cliUtil.RESULT_NOCHECK, "", errMsg)

        #心跳到对端控制器检查
        cmd = "sshtoremote"
        passWord = contextUtil.getDevLoginUserPwd(self.context)
        flag, cliRet, errMsg = cliUtil.sshToRemoteContr(cli, cmd, passWord, self.LANG)
        
        allCliRet = joinLines(allCliRet, cliRet)
        if flag != True:
            self.LOGGER.logError('SSH to remote controller failed.')
            return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
        try:
            isChkSucc, cliRet, errMsg, notMirrorState, singleWorkMode, failedTransformLunOrSnapshotList = self.chk2StatesOfAllObjsOnOneNode(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, True)
            notMirrorLunIdList, notMirrorSnapshotIdList, notMirrorDedupCompressLunList = notMirrorState[0], notMirrorState[1], notMirrorState[2]
            singleWorkModeLunList, singleWorkModeSnapshotList, singleCtrlModeOfDedupCompressLunList = singleWorkMode[0], singleWorkMode[1], singleWorkMode[2]
            allCliRet = joinLines(allCliRet, cliRet)
            if not isChkSucc:
                return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
            elif notMirrorLunIdList or notMirrorSnapshotIdList or notMirrorDedupCompressLunList or singleWorkModeLunList or singleWorkModeSnapshotList or singleCtrlModeOfDedupCompressLunList:
                return False, allCliRet, errMsg
            elif failedTransformLunOrSnapshotList:
                return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
            else:
                self.LOGGER.logInfo('All LUNs or snapshots cache write mirror on this controller are enabled and work on double'
                ' controller mode, check other controller now...')
                return True, allCliRet, errMsg
        except Exception, e:
            self.LOGGER.logError('Check LUNs and snapshots cache write mirror state and work mode exception:' + unicode(e))
            self.LOGGER.logException(e)
            return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
        finally:
            cliUtil.exitHeartbeatCli(cli, self.LANG)
        
    
    def chk2StatesOfAllObjsOnAllNodesInEngine(self, cli, lunIdList, snapshotIdList, dedupCompressLunIdList, ctrlIdList, engId):
        '''
        @summary: Check write mirror status and work mode on every nodes in one engine by sshtoremoteExt nodeId.
        @param cli: CLI connection.
        @param lunIdList:LUN ID list.
        @param snapshotIdList: Snapshot ID List.
        @param ctrlNodeList: Controller ID List.
        @param engId: Engine number to be checked such as engine 0, engine 1 and so on.
        @return: (checkResult, cliRet, errMsg)
        '''
        engNumDict = self.getEngineCtrlNumDict(ctrlIdList)
        engNodeNum = engNumDict.get(engId)
        self.LOGGER.logInfo('Engine number map is:' + unicode(engNumDict))
        self.LOGGER.logInfo('This engine node number is:' + unicode(engNodeNum))
        
        if engNodeNum == 1:
            return self.chk2StatesOfAllObjsOnCtrlPair(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, isPeerPresent=False)
        elif engNodeNum == 2:
            return self.chk2StatesOfAllObjsOnCtrlPair(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, isPeerPresent=True)
        else:
            pass
        
        #Get controller IDs that are present.
        isQrySucc, cliRet, errMsg, presentNodeIdList = cliUtil.getPresentCtrlNodeIdListForSshToRemote(cli, self.LANG, engId)
        if not isQrySucc:
            self.LOGGER.logError('Query controller present node id list failed.')
            return cliUtil.RESULT_NOCHECK, cliRet, errMsg
        else:
            self.LOGGER.logInfo('Query controller node id list success:' + unicode(presentNodeIdList))
              
        allCliRet, errMsg = '', ''
        passWord = contextUtil.getDevLoginUserPwd(self.context)

        allSingleCtrlModeLunIdList, allSingleSnapshotIdList, allSingleDedupCompressLunIdList = [], [], []

        priKey = contextUtil.getDevObj(self.context).getPriKey()
        if priKey:
            # 不支持PublicKey鉴权方式进行心跳控制器
            errMsg = common.getMsg(self.LANG, "no.support.publickey.forensics")
            return (cliUtil.RESULT_NOCHECK, "", errMsg)

        for nodeId in presentNodeIdList:
            cmd = "sshtoremoteExt %s" %str(nodeId)
            flag, cliRet, errMsg = cliUtil.sshToRemoteContr(cli, cmd, passWord, self.LANG)
            allCliRet = joinLines(allCliRet, cliRet)
            if flag != True:
                self.LOGGER.logError('sshtoremoteExt to remote node(' + unicode(nodeId) + ') failed.')
                errMsg = common.getMsg(self.LANG, "heart.beat.to.node.failed", unicode(nodeId))
                return (cliUtil.RESULT_NOCHECK, allCliRet, errMsg)
            else:
                self.LOGGER.logInfo('sshtoremoteExt to remote node(' + unicode(nodeId) + ') success.')

            isChkSucc, cliRet, errMsg, notMirrorState, singleWorkMode, failedTransformLunOrSnapshotList = self.chk2StatesOfAllObjsOnOneNode(cli, lunIdList, snapshotIdList, dedupCompressLunIdList, True)
            notMirrorLunIdList, notMirrorSnapshotIdList, notMirrorDedupCompressLunList = notMirrorState[0], notMirrorState[1], notMirrorState[2]
            singleWorkModeLunList, singleWorkModeSnapshotList, singleCtrlModeOfDedupCompressLunList = singleWorkMode[0], singleWorkMode[1], singleWorkMode[2]
            allSingleCtrlModeLunIdList.extend(singleWorkMode[0])
            allSingleSnapshotIdList.extend(singleWorkMode[1])
            allSingleDedupCompressLunIdList.extend(singleWorkMode[2])

            cliUtil.exitHeartbeatCli(cli, self.LANG)
            
            allCliRet = joinLines(allCliRet, cliRet)
            if not isChkSucc:
                self.LOGGER.logError('All LUNs or snapshots cache write mirror and work mode are not checked successfully on'
                ' node(node ID:' + unicode(nodeId) + '), return NO_CHECK now!')
                return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
            elif notMirrorLunIdList or notMirrorSnapshotIdList or notMirrorDedupCompressLunList:
                self.LOGGER.logError('Some LUNs or snapshots cache write mirror are not enabled on'
                ' node(node ID:' + unicode(nodeId) + '), return False now!')
                return False, allCliRet, errMsg
            elif (singleWorkModeLunList or singleWorkModeSnapshotList or singleCtrlModeOfDedupCompressLunList) and (not self.is_big_card):
                self.LOGGER.logError('Some LUNs or snapshots work in single controller mode on'
                ' node(node ID:' + unicode(nodeId) + '), and this is small card device, return False now!')
                return False, allCliRet, errMsg
            elif failedTransformLunOrSnapshotList:
                return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
            else:
                self.LOGGER.logInfo('All LUNs or snapshots cache write mirror are enabled'
                ' on current node(node ID:' + unicode(nodeId) + ').')
        
        #End loop of node ID list (sshtoremoteExt nodeID).
        notPasswdLunIdList, notPasswdSnapshotIdList, notPasswdDedupCompressLunIdList = [], [], []
        if self.is_big_card:
            notPasswdLunIdList = filter(lambda lunId: allSingleCtrlModeLunIdList.count(lunId) > 2, allSingleCtrlModeLunIdList)
            notPasswdSnapshotIdList = filter(lambda snapshotId: allSingleSnapshotIdList.count(snapshotId) > 2, allSingleSnapshotIdList)
            notPasswdDedupCompressLunIdList = filter(lambda dedupCompressLunId: allSingleDedupCompressLunIdList.count(dedupCompressLunId) > 2, allSingleDedupCompressLunIdList)

            notPasswdLunIdList = list(set(notPasswdLunIdList))
            notPasswdLunIdList.sort(cmp=None, key=int, reverse=False)
            
            notPasswdSnapshotIdList = list(set(notPasswdSnapshotIdList))
            notPasswdSnapshotIdList.sort(cmp=None, key=int, reverse=False)
            
            notPasswdDedupCompressLunIdList = list(set(notPasswdDedupCompressLunIdList))
            notPasswdDedupCompressLunIdList.sort(cmp=None, key=int, reverse=False)

            if notPasswdLunIdList:
                notPasswdLunIdList.sort(cmp=None, key=int, reverse=False)
                errMsg += common.getMsg(self.LANG, "lun.work.in.single.ctrl.mode", ",".join(notPasswdLunIdList))
        
            if notPasswdSnapshotIdList:
                notPasswdSnapshotIdList.sort(cmp=None, key=int, reverse=False)
                errMsg += common.getMsg(self.LANG, "snapshot.work.in.single.ctrl.mode", ",".join(notPasswdSnapshotIdList))

            if notPasswdDedupCompressLunIdList:
                notPasswdDedupCompressLunIdList.sort(cmp=None, key=int, reverse=False)
                errMsg += common.getMsg(self.LANG, "dedupCompress.work.in.single.ctrl.mode", ",".join(notPasswdDedupCompressLunIdList))
       
        if notPasswdLunIdList or notPasswdSnapshotIdList or notPasswdDedupCompressLunIdList:
            return False, allCliRet, errMsg
        else:    
            return True, allCliRet, errMsg
        
    def transformVolumeToLunOrSnapshot(self, cli, volumeId):
        """
        @summary: 将卷ID转换为LUN ID/快照ID。当需要使用debug下查询到的卷ID查找cli下命令使用的LUN ID对象时，
                                         需要使用命令 volume showctrl [volumeID]通过卷ID查找LUN ID，再进行后续操作。
        """
        lunId = self.VolumeToLunDict.get(volumeId, None)
        if lunId is not None:
            self.LOGGER.logInfo("The lunId is %s of volumeId(%s)." %(lunId, volumeId))
            return (True, "", lunId)
        
        cmd = "volume showctrl %s" % volumeId
        flag, cliRet, _ = cliUtil.executeCmdInDebugMode(cli, cmd, True, self.LANG)
        if flag != True: 
            self.LOGGER.logSysAbnormal()
            return (False, cliRet, "")
    
        for line in cliRet.splitlines():
            if ":" not in line:
                continue
            key = line.split(":")[0]
            if key.strip().lower() == "lun id":
                lunId = line.split(":")[1].strip()
                self.VolumeToLunDict[volumeId] = lunId
                return (True, cliRet, lunId)
            
        return (False, cliRet, "")
            
    def transformLunToVolume(self, cli, lunId):
        """
        @summary: 将LUN ID转换为卷ID。当需要使用cli下命令查询到的LUN ID或快照ID查找debug下的卷对象时，
                                        需要使用命令devlun show [LUN ID]通过LUN ID查找卷ID，再进行后续操作。
        """
        cmd = "devlun showctrl %s" % lunId
        flag, cliRet, _ = cliUtil.executeCmdInDebugMode(cli, cmd, True, self.LANG)
        if flag != True: 
            self.LOGGER.logSysAbnormal()
            return (False, cliRet, "")
    
        for line in cliRet.splitlines():
            if ":" not in line:
                continue
            fields = line.split(":")
            if len(fields) <= 1:
                continue
            key = fields[0]
            if key.strip().lower() == "data volume id":
                volumeId = fields[1].strip()
                return (True, cliRet, volumeId)
            
        return (False, cliRet, "")
    
    def transformSnapshotToVolume(self, cli, snapshotId):
        """
        @summary: 将快照ID转换为卷ID。转换方式同LUN
        """
        return self.transformLunToVolume(cli, snapshotId)

    def transformDedupCompressLunToVolume(self, cli, dedupCompressLun):
        """
        @summary: 将重删压缩LUN ID转换为卷ID。转换方式同LUN
        """
        return self.transformLunToVolume(cli, dedupCompressLun)
    
    def refreshProcess(self, context, percentNumber):
        """
        @summary: 设置巡检当前进度
        """
        dev = context.get("dev")
        observer = context.get("progressObserver")
        try:
            if observer != None:
                observer.updateItemProgress(dev, int(percentNumber), "cli_software_lunMirror")
        except (JException, Exception), exption:
            self.LOGGER.logInfo("refresh progress of the lun mirror check item exception:%s" %unicode(exption))
    
    def getCheckNodeNumbers(self, productVersion, checkObjNum, contrNum):
        """
        @summary: 获取需要检查的节点数量
        """
        if productVersion < 'V300R003C00' or productVersion >= 'V300R006C00SPC100':
            return 1
        if ('V300R003C00' <= productVersion < 'V300R006C00SPC100') and (checkObjNum > 400):
            return 1
        if ('V300R003C00' <= productVersion < 'V300R006C00SPC100') and (checkObjNum <= 400):
            return contrNum
        
        return 1
    
def joinLines(oldLines, newLines):
    if oldLines == None or oldLines == "" :
        return newLines
    return '\n'.join([oldLines, newLines]) if newLines else oldLines