# -*- coding: UTF-8 -*-
import traceback
from frame.cli import cliUtil
from frame.common import common
from frame.context import contextUtil
import java.lang.Exception as JException

ITEM = "cli_software_lunNoMirror"

def execute(context):
    checkResult = LunMirrorCheck(context).run(context)
    common.setProgress(context, common.PROGRESS_NUM_MAX, ITEM)
    return common.getUpgEvaluationRs(checkResult[0], checkResult[1], checkResult[2])


class LunMirrorCheck():
    def __init__(self, context):
        self.LANG = contextUtil.getLang(context)
        self.LOGGER = common.getLogger(contextUtil.getLogger(context), __file__)
        self.context = context
        self.is_big_card = False
        self.VolumeToLunDict = {}
        self.CURRENTPROCESS = 0
        self.CHECK_NODE_NUMBERS = 1

    def run(self, context):
        """
        @summary: 存储设备通过OpenStack创建LUN的过程中如果将LUN的Cache镜像功能关闭，后续业务运行过程中，如果控制器复位，将概率导致数据丢失，业务中断。
                  Check LUN cache status and work mode of every LUN and snapshot.
        """
        allCliRet = ""
        allErrMsg = ""

        try:
            cli = contextUtil.getSSH(context)
            checkRet = cliUtil.getControllerIdList(cli, self.LANG)
            if checkRet[0] != True:
                self.LOGGER.logError("Query controlled ID list failed, check terminated!")
                return checkRet[0], checkRet[1], checkRet[2]

            common.setProgress(context, common.PROGRESS_NUM_1, ITEM)

            contrIdList = checkRet[1]
            controllerNum = len(contrIdList)
            # 单控不存在lun镜像问题，检查通过
            if controllerNum == 1:
                self.LOGGER.logWarning('Single controller running, check PASSED!')
                return True, allCliRet, allErrMsg

            isQrySucc, lvlOrRet, errMsg = cliUtil.getUserPrivilege(cli, self.LANG)
            if not isQrySucc:
                allCliRet = joinLines(allCliRet, lvlOrRet)
                return cliUtil.RESULT_NOCHECK, allCliRet, errMsg

            if lvlOrRet.lower() != 'super_admin':
                errMsg = common.getMsg(self.LANG, "user.authority.failure")
                return cliUtil.RESULT_NOCHECK, allCliRet, errMsg
            common.setProgress(context, common.PROGRESS_NUM_10, ITEM)
            # >>获取LUN ID List
            flag, cliRet, errMsg, lunIdList = cliUtil.getLunIdListWithCliRet(cli, self.LANG)
            allCliRet = joinLines(allCliRet, cliRet)
            if flag != True:
                self.LOGGER.logError('Query LUN ID list failed, check terminated!')
                return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg
            common.setProgress(context, common.PROGRESS_NUM_15, ITEM)
            lunNum = len(lunIdList)
            if lunNum == 0:
                self.LOGGER.logInfo('No lun configured,check passed!')
                return True, allCliRet, allErrMsg

            # >>获取快照信息
            isQrySucc, cliRet, errMsg, snapshotIdList = self.getSnapshotIdList(cli)
            allCliRet = joinLines(allCliRet, cliRet)
            if not isQrySucc:
                allErrMsg = joinLines(allErrMsg, errMsg)
                return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg
            common.setProgress(context, common.PROGRESS_NUM_20, ITEM)
            # 获取设备型号
            devType = contextUtil.getDevType(context)
            self.LOGGER.logInfo('The device type is:%s.' % str(devType))
            if '18' in devType:
                cliConToCtrl, cliRet, errMsg = self.create18000CtrlCliConnection(cli)
                allCliRet = joinLines(allCliRet, cliRet)
                if not cliConToCtrl:
                    allErrMsg = joinLines(allErrMsg,
                                          common.getMsg(self.LANG, "build.ssh.connection.to.controller.failed"))
                    return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg
                else:
                    chkRet = self.chkAllNotMirrorLuns(cliConToCtrl, allCliRet, allErrMsg, lunIdList, snapshotIdList)
                    contextUtil.closeCliConnection(cliConToCtrl)
                    return chkRet

            return self.chkAllNotMirrorLuns(cli, allCliRet, allErrMsg, lunIdList, snapshotIdList)

        except (Exception, JException), exception:
            self.LOGGER.logException(exception)
            return cliUtil.RESULT_NOCHECK, allCliRet, common.getMsg(self.LANG, "query.result.abnormal")
        finally:
            contextUtil.getCLI(context)

    def chkAllNotMirrorLuns(self, cliConToCtrl, allCliRet, allErrMsg, lunIdList, snapshotIdList):
        # >>检查登录控制器不镜像的LUN和快照
        isQrySucc, qryRet, errMsg, notMirrorObjIdList = self.checkLoginCtrlNotMirrorObj(cliConToCtrl)
        allCliRet = joinLines(allCliRet, qryRet)
        if not isQrySucc:
            allErrMsg = joinLines(allErrMsg, errMsg)
        common.setProgress(self.context, common.PROGRESS_NUM_40, ITEM)
        # >>查询其他正常状态控制器node ID
        isQrySucc, qryRet, errMsg, otherNidList, nodeNum = self.getOtherNormalNidList(cliConToCtrl)
        allCliRet = joinLines(allCliRet, qryRet)
        if not isQrySucc:
            allErrMsg = joinLines(allErrMsg, errMsg)
            notMirrorErrMsg = self.getErrMsg(notMirrorObjIdList, lunIdList, snapshotIdList, [])
            if notMirrorErrMsg:
                allErrMsg = joinLines(allErrMsg, notMirrorErrMsg)
                return False, allCliRet, allErrMsg
            else:
                return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg
        common.setProgress(self.context, common.PROGRESS_NUM_60, ITEM)
        # >>检查归属其他控制器的单控工作模式的LUN
        isExtCmd = nodeNum > 2
        priKey = contextUtil.getDevObj(self.context).getPriKey()
        if priKey:
            # 不支持PublicKey鉴权方式进行心跳控制器
            errMsg = common.getMsg(self.LANG, "no.support.publickey.forensics")
            return (cliUtil.RESULT_NOCHECK, "", errMsg)
        singleWorkModeObjIdList, heartbeatFailNids, checkFailNidList, chkRet = self.checkOtherNid(cliConToCtrl,
                                                                                                  otherNidList,
                                                                                                  isExtCmd)
        allCliRet = joinLines(allCliRet, chkRet)
        notMirrorErrMsg = self.getErrMsg(notMirrorObjIdList, lunIdList, snapshotIdList, singleWorkModeObjIdList)
        if heartbeatFailNids:
            allErrMsg = joinLines(allErrMsg,
                                  common.getMsg(self.LANG, "heart.beat.to.node.failed", ",".join(heartbeatFailNids)))
        common.setProgress(self.context, common.PROGRESS_NUM_80, ITEM)
        if checkFailNidList:
            allErrMsg = joinLines(allErrMsg, common.getMsg(self.LANG, "heart.beat.to.node.to.check.failed",
                                                           ",".join(checkFailNidList)))
        if notMirrorErrMsg:
            allErrMsg = joinLines(allErrMsg, notMirrorErrMsg)
            return False, allCliRet, allErrMsg

        if heartbeatFailNids or checkFailNidList:
            return cliUtil.RESULT_NOCHECK, allCliRet, allErrMsg

        return True, allCliRet, ''

    def getEngineCtrlNumDict(self, contrIdList):
        """
        @summary: 获取引擎下的控制器数量
        """
        engNumDict = {}

        for contrId in contrIdList:
            engineId = contrId[:1]
            engNumDict[engineId] = engNumDict.get(engineId, 0) + 1

        return engNumDict

    def getConnectedEngineId(self, managementPortDictList):
        """
        @summary: Get the ID of engine that toolkit has connected.
        @param managementPortDictList: Management port information dictionary.
        @return: Number of engine that toolkit has current connected.
        """
        currentContrIp = contextUtil.getCurCtrlIp(self.context)
        self.LOGGER.logInfo("The IP that ToolKit connected is %s." % currentContrIp)

        for managementPortDict in managementPortDictList:
            ip = managementPortDict.get("IPv4 Address")
            if ip == currentContrIp:
                managementPortId = managementPortDict.get("ID")
                engineId = managementPortId.split(".")[0][-1:]
                self.LOGGER.logInfo("The engine that ToolKit connected is %s." % engineId)
                return engineId

        return ""

    def getAllEngineReachableIP(self, mgtPortDictList):
        """
        @summary: Get reachable IP address of every engine.
        @param mgtPortDictList: Management port IP address information dictionary.
        @return: {engineId: reachable_IP_address}
        """
        engineIpDict = {}
        for managementPortInfoDict in mgtPortDictList:
            ip = managementPortInfoDict.get("IPv4 Address")
            portRunningStatus = managementPortInfoDict.get("Running Status")
            if portRunningStatus == "Link Up" and ip not in ["", "--"]:
                managementPortId = managementPortInfoDict.get("ID")
                engineId = managementPortId.split(".")[0][-1:]
                if (engineId not in engineIpDict) and ("DSW" not in managementPortId):
                    engineIpDict.setdefault(engineId, ip)

        return engineIpDict

    def getSnapshotIdList(self, cli):
        """
        @summary: 获取所有的快照id
        @param cli: CLI connection.
        @return: (isQrySucc, cliRet, errMsg, snapshotIdList)
        """
        snapshotIdList = []

        cmd = "show snapshot general |filterColumn include columnList=ID"
        flag, cliRet, errMsg = cliUtil.excuteCmdInCliMode(cli, cmd, True, self.LANG)
        if flag == False:
            self.LOGGER.logInfo("Failed to get information about snapshot.")
            return False, cliRet, errMsg, snapshotIdList
        elif flag == cliUtil.RESULT_NOSUPPORT or flag == cliUtil.RESULT_NOCHECK:
            return True, cliRet, errMsg, snapshotIdList
        elif cliUtil.queryResultWithNoRecord(cliRet):
            return True, cliRet, errMsg, []

        snapshotInfoDictList = cliUtil.getHorizontalNostandardCliRet(cliRet)
        if len(snapshotInfoDictList) == 0:
            self.LOGGER.logWarning("Snapshot number is 0, CLI result parsing may have failed.")
            return False, cliRet, errMsg, snapshotIdList

        for snapshotInfoDict in snapshotInfoDictList:
            snapshotId = snapshotInfoDict.get("ID")
            snapshotIdList.append(snapshotId)

        self.LOGGER.logInfo('Snapshot number is:' + unicode(len(snapshotIdList)))

        return True, cliRet, errMsg, snapshotIdList

    # noinspection PyBroadException,PyBroadException
    def checkLoginCtrlNotMirrorObj(self, cli):
        """

        :param cli:
        :return:isQrySucc, qryRet, errMsg, notMirrorObjIdList
        """
        cmd = "cache show obj v switch nomirrorlun"
        flag, cliRet, errMsg = cliUtil.executeCmdInDebugMode(cli, cmd, True, self.LANG)
        if flag != True:
            self.LOGGER.logSysAbnormal()
            return False, cliRet, errMsg, []

        notMirrorObjIdList = []
        for line in cliRet.splitlines():
            fields = line.split()
            if fields[0].strip().isdigit():
                objId = fields[0].strip()
                notMirrorObjIdList.append(objId)

        return True, cliRet, '', notMirrorObjIdList

    def getOtherNormalNidList(self, cli):
        cmd = "sys showcls"
        flag, cliRet, errMsg = cliUtil.executeCmdInDebugMode(cli, cmd, True, self.LANG)
        if flag != True:
            self.LOGGER.logSysAbnormal()
            return False, cliRet, errMsg, [], 0

        otherNidList = []
        localNid = None
        nodeNum = 0
        for line in cliRet.splitlines():
            if 'local node id' in line:
                localNid = line.split(':')[1].strip()
                continue
            if 'node cfg' in line:
                nodeNum = int(line.split(':')[1].strip())
                continue

            fields = line.split()
            if fields[0].isdigit() and len(fields) >= 3 and fields[2] == 'normal' and localNid != fields[0]:
                otherNidList.append(fields[0])
                continue

        return True, cliRet, '', otherNidList, nodeNum

    def checkOtherNid(self, cli, otherNidList, isExtCmd=True):
        heartbeatFailNids = []
        cliRetList = []
        checkFailNidList = []
        passWord = contextUtil.getDevLoginUserPwd(self.context)
        notMirrorObjIdList = []
        for nid in otherNidList:
            if isExtCmd:
                sshCmd = 'sshtoremoteExt %(nodeId)s' % {'nodeId': nid}
            else:
                sshCmd = 'sshtoremote'

            heartbeatSucc, hearbeatRet, heartbeatErrMsg = cliUtil.sshToRemoteContr(cli, sshCmd, passWord, self.LANG)
            cliRetList.append(hearbeatRet)
            if heartbeatSucc != True:
                self.LOGGER.logError('heartbeat to node: %(nodeId)s failed.' % {'nodeId': nid})
                heartbeatFailNids.append(nid)
                continue

            try:
                dbgCmd = 'cache show obj v switch singlectrl'
                dbgCmdSucc, dbgRet, dbgErrMsg = cliUtil.executeCmdInDebugMode(cli, dbgCmd, True, self.LANG)
                cliRetList.append(dbgRet)
                if dbgCmdSucc != True:
                    checkFailNidList.append(nid)

                for line in dbgRet.splitlines():
                    fields = line.split()
                    if fields[0].isdigit():
                        notMirrorObjIdList.append(fields[0])
            except:
                self.LOGGER.logError(traceback.format_exc())
            finally:
                cliUtil.exitHeartbeatCli(cli, self.LANG)

        return notMirrorObjIdList, heartbeatFailNids, checkFailNidList, '\n'.join(cliRetList)

    def create18000CtrlCliConnection(self, cli):
        """

        :param cli:
        :return:
        """
        isQrySucc, ctrlIpTupList = cliUtil.getCtrlIps(cli, self.LOGGER, self.LANG)
        if not isQrySucc:
            self.LOGGER.logError('Query management port information failed.')
            return None, '', ''
        else:
            self.LOGGER.logInfo('Query management port information success:' + unicode(ctrlIpTupList))

        for _, innerIpV4, _ in ctrlIpTupList:
            self.LOGGER.logInfo("Begin connect inner IP %s." % innerIpV4)
            cliConnection = contextUtil.createCliConnection(self.context, innerIpV4)
            if cliConnection:
                self.LOGGER.logInfo("Build ssh connection success to inner IP %s." % innerIpV4)
                return cliConnection, '', ''

        return None, '', ''

    def sortByInt(self, sortList):
        """
        @summary: 对元素类型为int的列表排序
        """
        try:
            sortList.sort(cmp=None, key=int, reverse=False)
        except:
            self.LOGGER.logError("list sort by int failed, maybe the list item is not int type.")
            self.LOGGER.logError(str(traceback.format_exc()))

    def getErrMsg(self, notMirrorObjIdList, lunIdList, snapshotIdList, allSingleWorkingModeIdList):
        notMirrorLunIdList = []
        notMirrorSnapshotIdList = []
        for objId in notMirrorObjIdList:
            if objId in lunIdList:
                notMirrorLunIdList.append(objId)
            if objId in snapshotIdList:
                notMirrorSnapshotIdList.append(objId)

        for objId in allSingleWorkingModeIdList:
            if objId in lunIdList:
                notMirrorLunIdList.append(objId)
            if objId in snapshotIdList:
                notMirrorSnapshotIdList.append(objId)

        if not notMirrorLunIdList and not notMirrorSnapshotIdList:
            return ''

        errMsgList = []
        if notMirrorLunIdList:
            self.sortByInt(notMirrorLunIdList)
            lunErrMsg = common.getMsg(self.LANG, "not.mirror.lun.list", ",".join(notMirrorLunIdList))
            errMsgList.append(lunErrMsg)

        if notMirrorSnapshotIdList:
            self.sortByInt(notMirrorSnapshotIdList)
            snapshotErrMsg = common.getMsg(self.LANG, "not.mirror.snapshot.list", ",".join(notMirrorSnapshotIdList))
            errMsgList.append(snapshotErrMsg)

        return '\n'.join(errMsgList)


def joinLines(oldLines, newLines):
    if not oldLines:
        return newLines
    return '\n'.join([oldLines, newLines]) if newLines else oldLines
