# -*- coding: UTF-8 -*-

import re
from cmdRetManager import getCliRet
from constant import DeviceType

def switchDeviceType(deviceType):
    '''
    @summary: 将设备类型转换成统一类型（S2600T、S3900、S6900）
    @param deviceType: the type of device
    @return: DeviceType
    '''
    #查询设备类型
    if deviceType in ["S2200T", "S2200T-S", "S2600T", "S2900"]:
        return DeviceType.S2900
    elif deviceType in ["S3900-M200", "S3900-M300", "S3900", "S5500T"]:
        return DeviceType.S3900
    elif deviceType in ["S5600T", "S5900-M100", "S5900-M200", "S5800T", "S6800T", "S6900-M100", "S8000-I", "S6900"]:
        return DeviceType.S6900
    else:
        return DeviceType.UNKNOWN
    
def getCurRealDeviceType(dataDict):
    '''
    @summary: 获取当前真实设备类型
    @param dataDict: the context object provided by tool framework
    @return: deviceType
    '''
    #查询设备类型
    deviceType = ""
    sysInfo = getCliRet(dataDict, "showsys")
    lineList = sysInfo.splitlines()
    if re.search("System Information", sysInfo, re.IGNORECASE):
        for line in lineList:
            if line.find("Device Type") != -1:
                field = line.split()
                if len(field) == 5:
                    deviceType = field[-1]
                break
    return (sysInfo, deviceType)
    
    
def getCurDeviceType(dataDict):
    '''
    @summary: 获取当前等价设备类型（S2600T、S3900、S6900）
    @param dataDict: the context object provided by tool framework
    @return: DeviceType
    '''
    sysInfo, DeviceType = getCurRealDeviceType(dataDict)
    #转换成统一的设备型号
    DeviceType = switchDeviceType(DeviceType)
    return (sysInfo, DeviceType)




def switchValidVersion(version):
    '''
    @summary: 将版本号转化成有效的版本号（17位：V100R002C00SPC001 或者11位：V100R005C02）
    @param version: the version of device
    @return: validVersion
    '''
    validVersion = ""
    
    startIndex = version.index("V100")
    if re.search("SPC", version, re.IGNORECASE):
        endIndex = startIndex + 17  #格式：V100R002C00SPC001
    else:
        endIndex = startIndex + 11  #格式：V100R005C02
    
    #提取有效的版本号信息
    validVersion = version[startIndex: endIndex]
    
    return validVersion

def switchHotPatchVersion(version):
    '''
    @summary: 将热补丁版本号转化成有效的版本号（V100R002C00SPC011SPHb04转变成V100R002C00SPHb04）
    @param version: the hot patch version of device
    @return: version
    '''
    startIndex = version.index("V100")
    #提取有效的版本号信息
    version = version[startIndex:]

    #老版本热补丁版本不规范：V100R002C00SPC011SPHb04
    if "SPC" in version:
        startIndex = version.index("SPC")
        endIndex = startIndex + 6
        delStr = version[startIndex: endIndex]
        version = version.replace(delStr, "")
    return version


def getCurSystemVersion(dataDict):
    '''
    @summary: 获取当前系统SPC版本号
    @param dataDict: the context object provided by tool framework
    @return: flag, sysSpcVersion, sysVerInfo
    '''
    #查询系统版本号
    sysSpcVersion = ""
    sysVerInfo = getCliRet(dataDict, "showupgradepkginfo -t 1")
    flag = False
    lines = sysVerInfo.splitlines()
    # 命令执行结果小于7行，说明执行该命令后并没有输出有效信息
    if len(lines) < 7:
        return (flag, "", sysVerInfo)

    # 从第7行开始，取出有效信息
    field = lines[6].split()
    sysSpcVersion = switchValidVersion(field[1])
    flag = True
    return (flag, sysSpcVersion, sysVerInfo)


def getCurHotPatchVersion(dataDict):
    '''
    @summary: 获取当前系统热补丁SPH版本号
    @param dataDict: the context object provided by tool framework
    @return: flag, curHotPatchVer, hotPatchVerInfo
    '''
    #查询当前设备已安装热补丁
    curHotPatchVer = ""
    hotPatchVerInfo = getCliRet(dataDict, "showupgradepkginfo -t 3")
    #不支持热补丁，检查通过
    if re.search("not exist.|Error : Invalid argument", hotPatchVerInfo, re.IGNORECASE):
        return (True, '', hotPatchVerInfo)

    #没有安装任何热补丁，返回空
    if re.search("command operates successfully", hotPatchVerInfo, re.IGNORECASE):
        return (True, '', hotPatchVerInfo)

    #导入了热补丁包但是没有激活
    if not re.search("Running", hotPatchVerInfo, re.IGNORECASE):
        return (True, '', hotPatchVerInfo)
    
    lineList = hotPatchVerInfo.splitlines()
    #信息少于7行，查询信息失败
    if len(lineList) < 7:
        return (False, '', hotPatchVerInfo)
    
    #正常情况下获取Running的热补丁版本
    for line in lineList:
        if re.search("A|B", line) and re.search("Running", line, re.IGNORECASE):
            field = line.split()
            if len(field) >= 2:
                #兼容不规范的热补丁版本V100R002C00SPC011SPHb04
                curHotPatchVer = switchHotPatchVersion(field[1])
                break
        else:
            continue 

    #检查结果返回
    return (True, curHotPatchVer, hotPatchVerInfo)

def isAfterSecure(devVersion):
    '''
    @summary: 是否安全红线后设备
    @param devVersion: the version of device
    @return: flag
    '''
    R2C00_SECURE_VERSION = 'V100R002C00SPC010'
    R2C01_SECURE_VERSION = 'V100R002C01SPC007'
    isSecure = False
    devVersion = switchValidVersion(devVersion)
    if bool(re.search('V100R001', devVersion, re.IGNORECASE)):
        isSecure = False
    elif bool(re.search('V100R002C00', devVersion, re.IGNORECASE)):
        if devVersion < R2C00_SECURE_VERSION:
            isSecure = False
        else:
            isSecure = True
    elif bool(re.search('V100R002C01', devVersion, re.IGNORECASE)):
        if devVersion < R2C01_SECURE_VERSION:
            isSecure = False
        else:
            isSecure = True
    elif bool(re.search('V100R005', devVersion, re.IGNORECASE)):
        isSecure = True
    else:
        isSecure = False
    return isSecure

def isDoubleCtrlsModel(dataDict):
    '''
    @summary: 判断系统为单控还是双控
    @param dataDict: the context object provided by tool framework
    @return: flag
    '''
    systemInfo = getCliRet(dataDict, "showsys")
    if "Single Controller" in systemInfo:
        return False
    else:
        return True
