# -*- coding: UTF-8 -*-

import os
import re
from common.sysInfoManager import getCurDeviceType
from common.constant import CheckedResult, DeviceType
from common.cmdRetManager import getCliRet, checkCliInfoValid
from common.contextUtil import getLang, getLogger, getCurSysVer, getRet4Datacollect, getMainOrSlaveAndCtrlIdMap


def getPangeaMmlFileName(deCompressDestDir, fileName):
    '''
    @summary: 通过控制器路径名称，获取一键收集日志中的指定pangea_mmlinfo文件路径
    @param deCompressDestDir: 已解压的日志目录
    @param fileName: IP开头的文件夹名称
    @return: pangea_mmlinfo文件的绝对路径
    '''
    #文件夹名称匹配
    mmlFilePath = os.sep.join([deCompressDestDir, fileName, 'msg_other', 'msg_other', 'Other', 'pangea_mmlinfo'])
    
    #返回信息
    return mmlFilePath


# **************************************************************************** #
# 函数名称: checkFaultByParsePangeaMml
# 功能说明: 解析pangea的mml文件，查看纽扣电池告警
# 输入参数: deviceType, deCompressDestDir, fileName, context
# 输出参数: flag
# **************************************************************************** # 
def checkFaultByParsePangeaMml(deviceType, deCompressDestDir, fileName, context):
    
    flag = CheckedResult.PASS
    errMsg = ""
    checkStr = ""
    lang = getLang(context)
    log = getLogger(context)
    
    #获取控制器ID信息
    ctrlId = ''
    if -1 != fileName.find('_MAIN'):#主控
        ctrlId = getMainOrSlaveAndCtrlIdMap(context).get('Primary')
    else:
        ctrlId = getMainOrSlaveAndCtrlIdMap(context).get('Secondary')

   #通过控制器Ip查找mml文件位置
    mmlFilePath = getPangeaMmlFileName(deCompressDestDir, fileName)

    log.info("Getting mml file path of controller( ID: "  + ctrlId + ") is: "+ mmlFilePath)
    
    #判断文件是否存在
    if not os.path.exists(mmlFilePath):
        flag = CheckedResult.WARN
        if lang == "zh":
            errMsg = u"控制器%s：无法找到文件pangea_mmlinfo，可能原因是日志收集失败或日志解压失败。" % ctrlId
        else:
            errMsg = "Controller %s: Failed to find the pangea_mmlinfo file. The possible cause is log collection or decompression failure." % ctrlId
        return (flag, checkStr, errMsg)

    #解析mml信息文件
    file = open(mmlFilePath)
    mmlInfo = file.read()
    lineList = mmlInfo.splitlines()
    file.close()

    #信息有效性判断
    if not re.search("singlealarmget 2|board voltage alarm", mmlInfo, re.IGNORECASE):
        flag = CheckedResult.WARN
        if lang == "zh":
            errMsg = u"控制器" + ctrlId + u"：获取纽扣告警信息失败。"
        else:
            errMsg = "Controller " + ctrlId + ": Getting button cell alarm information failed."
        return (flag, checkStr, errMsg)
    
    #轮询查看每行的信息
    checkIndex = 0
    checkStr = "\n[Controller " + ctrlId + "]: pangea_mmlinfo file content:\n"
    for line in lineList:
        
        #文件有效内容记录
        if checkIndex != 0:
            checkStr += line + "\n"
        
        if re.search("singlealarmget 2", line, re.IGNORECASE): 
            log.info("Single alarm get line:" + line)
            #将字段返回
            checkStr += line + "\n"
            
            #S2900查询第7位
            if deviceType == DeviceType.S2900:
                checkIndex = 6
            #S3900查询第15位
            elif deviceType == DeviceType.S3900:
                checkIndex = 15
            #S6900查询第8位
            elif deviceType == DeviceType.S6900:
                checkIndex = 7
            #未知型号，不存在此场景
            else:
                pass
        elif re.search("board voltage alarm", line, re.IGNORECASE):
            if checkIndex != 0:
                log.info("board voltage alarm line:" + line)
                
                temp = line.split(":")[-1].strip()
                checkAlarm = temp.split()[checkIndex]
                
                #纽扣电池位如果不为0，说明发送了故障告警
                if checkAlarm == "0":
                    return (flag, checkStr, errMsg)
                elif checkAlarm == "1":
                    flag = CheckedResult.NOTPASS
                    if lang == "zh":
                        errMsg = u"控制器" + ctrlId + u"：纽扣电池发生故障，检查不通过。"
                    else:
                        errMsg = "Controller " + ctrlId + ": The button cell is faulty, check not pass."
                    return (flag, checkStr, errMsg)
                break

    #返回错误信息
    flag = CheckedResult.WARN
    if lang == "zh":
        errMsg = u"控制器" + ctrlId + u"：获取纽扣告警信息失败。"
    else:
        errMsg = "Controller " + ctrlId + ": Getting button cell alarm information failed."
    return (flag, checkStr, errMsg)


def execute(dataDict):
    '''
    @summary: the entrance of main method, this check item is used to check button cell status
    @param dataDict: the dictionary of data which provided by tool framework
    @return: (pass status, CLI information, error message) as (boolean, string, string)
    '''
    
    log = getLogger(dataDict)
    lang = getLang(dataDict)

    flag = CheckedResult.PASS
    cliRet = ''
    errMsg = ''
    
    #步骤1：获取设备类型
    iRet = getCurDeviceType(dataDict)
    cliRet = iRet[0]
    deviceType = iRet[1]
    if deviceType == DeviceType.UNKNOWN:
        flag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"获取设备类型失败。"
        else:
            errMsg = "Getting device type failed."
        return (flag, cliRet, errMsg)        
    
    deviceCurVer = getCurSysVer(dataDict)   #获取设备当前版本号
    deviceVerInfo = getCliRet(dataDict, "showupgradepkginfo -t 1")
    cliRet += '\n' + deviceVerInfo
    if 'V100' not in deviceVerInfo:
        flag = CheckedResult.NOTPASS
        if lang == "zh":
            errMsg = u"获取设备系统版本失败。"
        else:
            errMsg = "Getting device system version failed."
        return (flag, cliRet, errMsg)
    log.info('[TV100 check_ButtonCellStat] Device current version is ' + deviceCurVer)
    
    if deviceCurVer >= 'V100R005C02SPC300':
        return (flag, cliRet, errMsg)
    
    #获取一键收集日志解压文件夹
    flag4Datacollect, dataCollectRet, errMsg4Datacollect, deCompressDestDir, isDoubleCtrlLostOne = getRet4Datacollect(dataDict)
    cliRet += "\n" + dataCollectRet
    #解压路径为空或者不存在说明一定收集不成功(包括解压失败)；双控只收集到一个控制器的日志说明部分收集成功
    if not (deCompressDestDir and os.path.exists(deCompressDestDir)) or isDoubleCtrlLostOne:
        return (CheckedResult.NOCHECK, cliRet, errMsg4Datacollect)

    #步骤2：查看所有控制器的mml文件，确认是否存在纽扣电池故障
    #文件夹名称匹配
    for fileName in os.listdir(deCompressDestDir):
        if not fileName.startswith("DebugLog"):
            #解析pangea的mml文件，查看纽扣电池告警
            iRet = checkFaultByParsePangeaMml(deviceType, deCompressDestDir, fileName, dataDict)
            #判断回文信息
            checkFlag = iRet[0]
            cliRet += "\n" + iRet[1]
            if checkFlag != CheckedResult.PASS:
                flag = checkFlag
                errMsg += "\n" + iRet[2]
    
    #检查结果返回
    return (flag, cliRet, errMsg.strip()) 
